﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_MemoryHeapCommon.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h>
#include <nn/sf/sf_ExpHeapAllocator.h>
#include <nn/sf/sf_ISharedObject.h>
#include <nn/grc/grc_ServiceName.h>
#include <nn/grc/grc_Result.private.h>
#include <nn/grc/sf/grc_IUserManager.sfdl.h>

namespace nn { namespace gamecast {

namespace {

// アロケータのサイズ
const size_t AllocatorSize = 16 * 1024;

// HIPC プロキシ用の拡張ヒープを使用したアロケータ
struct CreateUserManagerByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<AllocatorSize, CreateUserManagerByHipcTag> ProxyAllocator;

// ProxyAllocator を静的コンストラクタで初期化するためのヘルパー
class ProxyAllocatorInitializer
{
public:
    ProxyAllocatorInitializer() NN_NOEXCEPT
    {
        ProxyAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }
} g_ProxyAllocatorInitializer;

nn::sf::SharedPointer<nn::grc::sf::IUserManager> CreateUserManagerByHipc() NN_NOEXCEPT
{
    nn::sf::SharedPointer<nn::grc::sf::IUserManager> pAccessor;
    auto result = nn::sf::CreateHipcProxyByName<nn::grc::sf::IUserManager, ProxyAllocator::Policy>(
        &pAccessor,
        nn::grc::UserPortName);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    NN_SDK_ASSERT(pAccessor);
    return pAccessor;
}

nn::sf::SharedPointer<nn::grc::sf::IUserManager> pUserManager = nullptr;

}

::nn::Result Initialize(void* workBuffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(workBuffer);
    NN_SDK_REQUIRES_GREATER(size, 0u);

    pUserManager = CreateUserManagerByHipc();

    ::nn::os::TransferMemoryType transferMemory;

    NN_RESULT_THROW_UNLESS(::nn::os::CreateTransferMemory(
        &transferMemory,
        workBuffer,
        size,
        ::nn::os::MemoryPermission_None).IsSuccess(),
        ResultGamecastCreateTransferMemoryFailed());

    auto transferMemoryHandle = ::nn::os::DetachTransferMemory(&transferMemory);

    NN_RESULT_DO(pUserManager->Initialize(::nn::sf::NativeHandle(transferMemoryHandle, true), size));

    NN_RESULT_SUCCESS;
}

::nn::Result Finalize() NN_NOEXCEPT
{
    NN_RESULT_DO(pUserManager->Finalize());
    pUserManager = nullptr;

    NN_RESULT_SUCCESS;
}

}} // namespace nn::gamecast
