﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_GameMovieTrimming.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

#include <nn/am/am_Shim.h>
#include <nn/am/am_Result.h>
#include <nn/grc/grc_Result.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/os/os_Thread.h>
#include <cstring>

namespace nn { namespace capsrv {

bool ReserveResourceForMovieOperation() NN_NOEXCEPT
{
    bool ret;
    NN_RESULT_ABORTING_BLOCK
    {
        NN_RESULT_TRY(am::GetLibraryAppletSelfAccessor()->ReserveResourceForMovieOperation())
            NN_RESULT_CATCH(am::ResultResourceForMovieOperationInUse)
            {
                ret = false;
                NN_RESULT_SUCCESS;
            }
        NN_RESULT_END_TRY
        ret = true;
        NN_RESULT_SUCCESS;
    };
    return ret;
}

void UnreserveResourceForMovieOperation() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(am::GetLibraryAppletSelfAccessor()->UnreserveResourceForMovieOperation());
}

Result TrimGameMovie(AlbumFileId* pOut, const TrimGameMovieParameters& p, void* workMemory, size_t workMemorySize) NN_NOEXCEPT
{
retry:
    os::TransferMemory tm{workMemory, workMemorySize, os::MemoryPermission_None};
    sf::SharedPointer<grcsrv::IGameMovieTrimmer> trimmer;
    NN_RESULT_TRY(am::GetLibraryAppletSelfAccessor()->CreateGameMovieTrimmer(&trimmer, sf::NativeHandle{tm.Detach(), true}, workMemorySize))
        NN_RESULT_CATCH(grc::ResultGrcBusy)
        {
            os::SleepThread(TimeSpan::FromMilliSeconds(100));
            goto retry;
        }
    NN_RESULT_END_TRY
    if (p.thumbnailImageRgba)
    {
        NN_RESULT_DO(trimmer->SetThumbnailRgba(sf::InBuffer(static_cast<const char*>(p.thumbnailImageRgba), p.thumbnailImageRgbaSize), 1280, 720));
    }
    sf::NativeHandle eventHandle;
    NN_RESULT_DO(trimmer->GetNotTrimmingEvent(&eventHandle));
    {
        grcsrv::GameMovieId id = {};
        std::memcpy(&id, &p.sourceId, sizeof(p.sourceId));
        NN_RESULT_DO(trimmer->BeginTrim(id, p.beginIndex, p.endIndex));
    }
    os::SystemEvent e;
    e.AttachReadableHandle(eventHandle.GetOsHandle(), false, os::EventClearMode_ManualClear);
    e.Wait();
    {
        grcsrv::GameMovieId id;
        NN_RESULT_DO(trimmer->EndTrim(&id));
        std::memcpy(pOut, &id, sizeof(*pOut));
    }
    NN_RESULT_SUCCESS;
}

}}
