﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/capsrv/capsrv_AlbumAccess.h>
#include <nn/capsrv/capsrv_AlbumAccessForDevMenuCommand.h>
#include <nn/capsrv/capsrv_AlbumTesting.h>

#include <nn/time.h>
#include <nn/sf/sf_ShimLibraryUtility.h>
#include <nn/capsrv/sf/capsrv_Services.sfdl.h>

#include <nn/capsrv/capsrv_Result.h>
#include <nn/capsrv/capsrv_EditedScreenShotInfo.h>
#include <nn/capsrv/capsrv_AlbumFileSizeLimit.h>

#include "capsrv_AlbumAccessServiceHolder.h"
#include "capsrv_ScreenShotServiceHolder.h"
#include "capsrv_Macro.h"
#include "capsrv_LibraryState.h"

#if defined(NN_CAPSRV_USE_DIRECT_FUNCTION_CALL)
#include "capsrv_InitializeForLibrary.h"
#endif

namespace nn{ namespace capsrv{

    nn::Result InitializeAlbumAccess() NN_NOEXCEPT
    {
        NN_RESULT_DO(g_AlbumAccessServiceHolder.Initialize(nullptr));
        // NOTE:
        //   ScreenShotService の権限を持っているのは photoViewer だけ。
        //   それ以外のアプレットで失敗するのは想定内なので無視する。
        g_ScreenShotServiceHolder.Initialize(nullptr);
        NN_RESULT_SUCCESS;
    }

    void FinalizeAlbumAccess() NN_NOEXCEPT
    {
        // NOTE:
        //   Holder の Finalize で ServiceObject 用のアロケータが破棄されるので
        //   先にすべてのオブジェクトを始末しておく。
        if(g_AlbumAccessServiceHolder.GetInitializeCount() == 1)
        {
            g_LibraryState.SetMovieReadStreamSession(nullptr, LibraryState::ServiceLevel_AlbumAccess);
        }

        g_AlbumAccessServiceHolder.Finalize();

        if(g_ScreenShotServiceHolder.IsInitialized())
        {
            g_ScreenShotServiceHolder.Finalize();
        }
    }

    int GetAlbumFileCountLimit(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        switch(storage)
        {
        case AlbumStorage_Nand:
            {
                return AlbumFileCountLimit_Nand;
            }
        case AlbumStorage_Sd:
            {
                return AlbumFileCountLimit_Sd;
            }
        default: NN_UNEXPECTED_DEFAULT;
        };
    }

    nn::Result GetAlbumFileCount(
        int* pOutFileCount,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileCount);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        *pOutFileCount = 0;

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t count = 0;
        NN_RESULT_DO(pService->GetAlbumFileCountEx0(&count, storage, contentsMask));

        *pOutFileCount = static_cast<int>(count);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumFileCount(
        int* pOutFileCount,
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot)
                    .Set(AlbumFileContents_Movie);
        return GetAlbumFileCount(pOutFileCount, storage, contentsMask);
    }

    nn::Result GetAlbumFileList(
        int* pOutFileCount,
        AlbumEntry* pAlbumEntryBuffer,
        int albumEntryBufferLength,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileCount);
        NN_SDK_REQUIRES_NOT_NULL(pAlbumEntryBuffer);
        NN_SDK_REQUIRES_ALIGNED(pAlbumEntryBuffer, NN_ALIGNOF(AlbumEntry));
        NN_SDK_REQUIRES_GREATER(albumEntryBufferLength, 0);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        bool isSuccess = false;
        *pOutFileCount = 0;
        NN_UTIL_SCOPE_EXIT{
            if(!isSuccess)
            {
                std::memset(pAlbumEntryBuffer, 0, sizeof(AlbumEntry) * albumEntryBufferLength);
            }
        };

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t count = 0;
        NN_RESULT_DO(pService->GetAlbumFileListEx0(&count, nn::sf::OutArray<AlbumEntry>(pAlbumEntryBuffer, albumEntryBufferLength), storage, contentsMask));

        *pOutFileCount = static_cast<int>(count);
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumFileList(
        int* pOutFileCount,
        AlbumEntry* pAlbumEntryBuffer,
        int albumEntryBufferLength,
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot)
                    .Set(AlbumFileContents_Movie);
        return GetAlbumFileList(pOutFileCount, pAlbumEntryBuffer, albumEntryBufferLength, storage, contentsMask);
    }

    nn::Result LoadAlbumScreenShotFile(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pFileId);

        NN_CAPSRV_PROCESS_START();
        *pOutSize = 0;
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(pBuffer, 0, bufferSize));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t fileSize = 0;
        NN_RESULT_DO(pService->LoadAlbumFile(&fileSize, nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize), *pFileId));

        NN_CAPSRV_PROCESS_SUCCESS();
        *pOutSize = static_cast<size_t>(fileSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result LoadAlbumFileThumbnail(
        size_t* pOutSize,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);
        NN_SDK_REQUIRES_NOT_NULL(pBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pFileId);

        *pOutSize = 0;
        std::memset(pBuffer, 0, bufferSize);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t size;
        NN_RESULT_DO(pService->LoadAlbumFileThumbnail(&size, nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize), *pFileId));

        *pOutSize = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    namespace {
        nn::Result LoadAlbumScreenShotImageImpl(
            int* pOutWidth,
            int* pOutHeight,
            ScreenShotAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId* pFileId,
            const ScreenShotDecodeOption& option,
            void* pWorkBuffer,
            size_t workBufferSize
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pBuffer);
            NN_SDK_REQUIRES_NOT_NULL(pFileId);
            NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);

            NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

            auto cleanupOutput = [&]() {
                if(pOutWidth)
                {
                    *pOutWidth = 0;
                }
                if(pOutHeight)
                {
                    *pOutHeight = 0;
                }
                if(pOutAttribute)
                {
                    pOutAttribute->SetDefault();
                }
                if(pOutAppletData)
                {
                    *pOutAppletData = {};
                }
            };

            NN_CAPSRV_PROCESS_START();

            cleanupOutput();
            NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

            NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

            nn::capsrv::sf::LoadAlbumScreenShotImageOutputEx1 output = {};
            NN_RESULT_DO(pService->LoadAlbumScreenShotImageEx1(
                &output,
                nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
                *pFileId,
                option,
                nn::sf::OutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize)
            ));

            ScreenShotAttribute attribute = {};
            attribute.SetDefault();
            reinterpret_cast<detail::ScreenShotAttributeEx0&>(attribute) = output.attribute;

            NN_CAPSRV_PROCESS_SUCCESS();
            if(pOutWidth)
            {
                *pOutWidth = static_cast<int>(output.width);
            }
            if(pOutHeight)
            {
                *pOutHeight = static_cast<int>(output.height);
            }
            if(pOutAttribute)
            {
                *pOutAttribute = attribute;
            }
            if(pOutAppletData)
            {
                *pOutAppletData = output.appletData;
            }
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result LoadAlbumScreenShotImage(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId,
        const ScreenShotDecodeOption& option,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT
    {
        return LoadAlbumScreenShotImageImpl(pOutWidth, pOutHeight, pOutAttribute, pOutAppletData, pBuffer, bufferSize, pFileId, option, pWorkBuffer, workBufferSize);
    }

    namespace {
        nn::Result LoadAlbumScreenShotThumbnailImageImpl(
            int* pOutWidth,
            int* pOutHeight,
            ScreenShotAttribute* pOutAttribute,
            AppletData* pOutAppletData,
            void* pBuffer,
            size_t bufferSize,
            const AlbumFileId* pFileId,
            const ScreenShotDecodeOption& option,
            void* pWorkBuffer,
            size_t workBufferSize
            ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_NULL(pBuffer);
            NN_SDK_REQUIRES_NOT_NULL(pFileId);
            NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);

            NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

            auto cleanupOutput = [&]() {
                if(pOutWidth)
                {
                    *pOutWidth = 0;
                }
                if(pOutHeight)
                {
                    *pOutHeight = 0;
                }
                if(pOutAttribute)
                {
                    pOutAttribute->SetDefault();
                }
                if(pOutAppletData)
                {
                    *pOutAppletData = {};
                }
            };

            NN_CAPSRV_PROCESS_START();

            cleanupOutput();
            NN_CAPSRV_PROCESS_ROLLBACK(cleanupOutput());

            NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

            nn::capsrv::sf::LoadAlbumScreenShotImageOutputEx1 output = {};
            NN_RESULT_DO(pService->LoadAlbumScreenShotThumbnailImageEx1(
                &output,
                nn::sf::OutBuffer(reinterpret_cast<char*>(pBuffer), bufferSize),
                *pFileId,
                option,
                nn::sf::OutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize)
            ));

            ScreenShotAttribute attribute = {};
            attribute.SetDefault();
            reinterpret_cast<detail::ScreenShotAttributeEx0&>(attribute) = output.attribute;

            NN_CAPSRV_PROCESS_SUCCESS();
            if(pOutWidth)
            {
                *pOutWidth = static_cast<int>(output.width);
            }
            if(pOutHeight)
            {
                *pOutHeight = static_cast<int>(output.height);
            }
            if(pOutAttribute)
            {
                *pOutAttribute = attribute;
            }
            if(pOutAppletData)
            {
                *pOutAppletData = output.appletData;
            }
            NN_RESULT_SUCCESS;
        }
    }

    nn::Result LoadAlbumScreenShotThumbnailImage(
        int* pOutWidth,
        int* pOutHeight,
        ScreenShotAttribute* pOutAttribute,
        AppletData* pOutAppletData,
        void* pBuffer,
        size_t bufferSize,
        const AlbumFileId* pFileId,
        const ScreenShotDecodeOption& option,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT
    {
        return LoadAlbumScreenShotThumbnailImageImpl(pOutWidth, pOutHeight, pOutAttribute, pOutAppletData, pBuffer, bufferSize, pFileId, option, pWorkBuffer, workBufferSize);
    }

    nn::Result DeleteAlbumFile(
        const AlbumFileId* pFileId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pFileId);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        NN_RESULT_DO(pService->DeleteAlbumFile(*pFileId));
        NN_RESULT_SUCCESS;
    }

    nn::Result StorageCopyAlbumFile(
        const AlbumFileId* pSrcFileId,
        AlbumStorageType destStorage
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pSrcFileId);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(destStorage);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        NN_RESULT_DO(pService->StorageCopyAlbumFile(*pSrcFileId, destStorage));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetRequiredStorageFreeSpaceSizeToCopyAll(
        size_t* pOutValue,
        AlbumStorageType dstStorage,
        AlbumStorageType srcStorage
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(dstStorage);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(srcStorage);

        *pOutValue = 0;

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t size = 0;
        NN_RESULT_DO(pService->GetRequiredStorageSpaceSizeToCopyAll(&size, dstStorage, srcStorage));

        *pOutValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    bool IsAlbumMounted(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        bool isMounted = false;
        NN_ABORT_UNLESS_RESULT_SUCCESS(
            pService->IsAlbumMounted(&isMounted, storage)
        );
        return isMounted;
    }

    nn::Result GetAlbumMountResult(
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        NN_RESULT_DO(pService->GetAlbumMountResult(storage));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumUsage(
        AlbumUsage* pOutValue,
        AlbumStorageType storage,
        AlbumFileContentsFlag contentsMask
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);
        NN_STATIC_ASSERT(sizeof(AlbumUsage) == sizeof(detail::AlbumUsage16));

        NN_CAPSRV_PROCESS_START();
        std::memset(pOutValue, 0, sizeof(AlbumUsage));
        NN_CAPSRV_PROCESS_ROLLBACK(std::memset(pOutValue, 0, sizeof(AlbumUsage)));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        NN_RESULT_DO(pService->GetAlbumUsage16(pOutValue, storage, contentsMask));

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumUsage(
        AlbumUsage* pOutValue,
        AlbumStorageType storage
        ) NN_NOEXCEPT
    {
        AlbumFileContentsFlag contentsMask = {};
        contentsMask.Set(AlbumFileContents_ScreenShot);
        return GetAlbumUsage(pOutValue, storage, contentsMask);
    }

    size_t GetAlbumFileSizeLimit(AlbumFileContentsType contents) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_CONTENTS(contents);
        auto sizeLimit = GetAlbumFileSizeLimitImpl(contents);
        NN_ABORT_UNLESS(sizeLimit > 0);
        return sizeLimit;
    }

    nn::Result GetAlbumFileSize(size_t* pOutValue, const AlbumFileId* pFileId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_SDK_REQUIRES_NOT_NULL(pFileId);

        *pOutValue = 0;

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        uint64_t size = 0;
        NN_RESULT_DO(pService->GetAlbumFileSize(&size, *pFileId));

        *pOutValue = static_cast<size_t>(size);
        NN_RESULT_SUCCESS;
    }

    // -> ScreenShotApi
    //nn::Result SaveEditedScreenShot(
    //    AlbumEntry* pOutEntry,
    //    const void* imageData,
    //    size_t imageDataSize,
    //    int width,
    //    int height,
    //    const void* thumbData,
    //    size_t thumbDataSize,
    //    int thumbWidth,
    //    int thumbHeight,
    //    const AlbumFileId* pOriginalFileId
    //    ) NN_NOEXCEPT;

    nn::Result GetLastThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT
    {
        return GetLastOverlayScreenShotThumbnail(pOutFileId, pOutDataSize, pOutBuffer, bufferSize);
    }

    nn::Result GetLastOverlayScreenShotThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileId);
        NN_SDK_REQUIRES_NOT_NULL(pOutDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);

        bool isSuccess = false;
        std::memset(pOutFileId, 0, sizeof(AlbumFileId));
        *pOutDataSize = 0;
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                std::memset(pOutBuffer, 0, bufferSize);
            }
        };

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        AlbumFileId fileId = {};
        uint64_t size = 0;
        NN_RESULT_DO(pService->GetLastOverlayScreenShotThumbnail(&fileId, &size, nn::sf::OutBuffer(reinterpret_cast<char*>(pOutBuffer), bufferSize)));

        *pOutFileId = fileId;
        *pOutDataSize = static_cast<size_t>(size);
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetLastOverlayMovieThumbnail(
        AlbumFileId* pOutFileId,
        size_t* pOutDataSize,
        void* pOutBuffer,
        size_t bufferSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutFileId);
        NN_SDK_REQUIRES_NOT_NULL(pOutDataSize);
        NN_SDK_REQUIRES_NOT_NULL(pOutBuffer);

        bool isSuccess = false;
        std::memset(pOutFileId, 0, sizeof(AlbumFileId));
        *pOutDataSize = 0;
        NN_UTIL_SCOPE_EXIT {
            if(!isSuccess)
            {
                std::memset(pOutBuffer, 0, bufferSize);
            }
        };

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        AlbumFileId fileId = {};
        uint64_t size = 0;
        NN_RESULT_DO(pService->GetLastOverlayMovieThumbnail(&fileId, &size, nn::sf::OutBuffer(reinterpret_cast<char*>(pOutBuffer), bufferSize)));

        *pOutFileId = fileId;
        *pOutDataSize = static_cast<size_t>(size);
        isSuccess = true;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAutoSavingStorage(
        AlbumStorageType* pOutValue
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        *pOutValue = 0;

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        AlbumStorageType storage = 0;
        NN_RESULT_DO(pService->GetAutoSavingStorage(&storage));

        *pOutValue = storage;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumEntryFromApplicationAlbumEntry(
        AlbumEntry* pOutValue,
        const ApplicationAlbumEntry& entry,
        nn::ncm::ApplicationId applicationId
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);

        std::memset(pOutValue, 0, sizeof(AlbumEntry));

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        AlbumEntry dstEntry = {};
        NN_RESULT_DO(pService->GetAlbumEntryFromApplicationAlbumEntry(&dstEntry, entry, applicationId));

        *pOutValue = dstEntry;
        NN_RESULT_SUCCESS;
    }


    //-----------------------------
    // セッション構築
    //-----------------------------
    namespace {

        // MovieReadStreamSession を取得する。
        // セッションがまだ作られていない場合、 AlbumAccessService から作成する
        // （AlbumControl から MovieReadStream を使用する場合、先に明示的にセッションを作っておく必要がある）
        nn::Result EnsureMovieReadStreamSessionConnected(LibraryState::MovieReadStreamServicePointerType* pOutValue) NN_NOEXCEPT
        {
            auto pSession = g_LibraryState.GetMovieReadStreamSession();
            if(pSession)
            {
                *pOutValue = pSession;
                NN_RESULT_SUCCESS;
            }

            NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

            AlbumAccessServiceHolder::SessionPointerType pAccessSession;
            NN_RESULT_DO(pService->OpenAccessorSession(&pAccessSession, LibraryState::GetMyAruid()));
            g_AlbumAccessServiceHolder.StoreSession(pAccessSession);
            g_LibraryState.SetMovieReadStreamSession(pAccessSession, LibraryState::ServiceLevel_AlbumAccess);
            *pOutValue = pAccessSession;
            NN_RESULT_SUCCESS;
        }

    }


    //-----------------------------
    // 動画 API
    //-----------------------------

    nn::Result OpenAlbumMovieReadStream(AlbumMovieReadStreamHandle* pOutHandle, const AlbumFileId& fileId) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        detail::AlbumMovieReadStreamHandleType handleValue = {};
        NN_RESULT_DO(pSession->OpenAlbumMovieReadStream(&handleValue, fileId));

        *pOutHandle = AlbumMovieReadStreamHandle(handleValue);
        NN_RESULT_SUCCESS;
    }

    nn::Result CloseAlbumMovieReadStreamImpl(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        NN_RESULT_DO(pSession->CloseAlbumMovieReadStream(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    void CloseAlbumMovieReadStream(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(CloseAlbumMovieReadStreamImpl(handle));
    }

    nn::Result GetAlbumMovieReadStreamDataSize(int64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        int64_t size = 0;
        NN_RESULT_DO(pSession->GetAlbumMovieReadStreamMovieDataSize(&size, handle.GetInnerValue()));

        *pOutSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadDataFromAlbumMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 0u);
        NN_SDK_REQUIRES(size == 0 || buffer != nullptr);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        int64_t readSize = 0;
        NN_RESULT_DO(pSession->ReadMovieDataFromAlbumMovieReadStream(&readSize, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size), handle.GetInnerValue(), offset));

        *pOutReadSize = static_cast<size_t>(readSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumMovieReadStreamImageDataSize(int64_t* pOutSize, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutSize);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        int64_t size = 0;
        NN_RESULT_DO(pSession->GetAlbumMovieReadStreamImageDataSize(&size, handle.GetInnerValue()));

        *pOutSize = size;
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadImageDataFromAlbumMovieReadStream(size_t* pOutReadSize, void* buffer, size_t size, AlbumMovieReadStreamHandle handle, int64_t offset) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
        NN_SDK_REQUIRES_GREATER_EQUAL(size, 0u);
        NN_SDK_REQUIRES(size == 0 || buffer != nullptr);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        int64_t readSize = 0;
        NN_RESULT_DO(pSession->ReadImageDataFromAlbumMovieReadStream(&readSize, nn::sf::OutBuffer(reinterpret_cast<char*>(buffer), size), handle.GetInnerValue(), offset));

        *pOutReadSize = static_cast<size_t>(readSize);
        NN_RESULT_SUCCESS;
    }

    nn::Result ReadAttributeFromAlbumMovieReadStream(ScreenShotAttribute* pOutAttribute, AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutAttribute);

        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        detail::ScreenShotAttributeEx0 attr = {};
        NN_RESULT_DO(pSession->ReadFileAttributeFromAlbumMovieReadStream(&attr, handle.GetInnerValue()));

        ScreenShotAttribute outAttr = {};
        outAttr.SetDefault();
        static_cast<detail::ScreenShotAttributeEx0&>(outAttr) = attr;

        *pOutAttribute = outAttr;
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumMovieReadStreamBrokenReason(AlbumMovieReadStreamHandle handle) NN_NOEXCEPT
    {
        LibraryState::MovieReadStreamServicePointerType pSession;
        NN_RESULT_DO(EnsureMovieReadStreamSessionConnected(&pSession));

        NN_RESULT_DO(pSession->GetAlbumMovieReadStreamBrokenReason(handle.GetInnerValue()));
        NN_RESULT_SUCCESS;
    }

    //-----------------------------
    // テスト用 API
    //-----------------------------

    nn::Result RefreshAlbumCache(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        NN_RESULT_DO(pService->RefreshAlbumCache(storage));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumCache(AlbumCacheData* pOutValue, AlbumStorageType storage, AlbumFileContentsType contents) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);
        NN_CAPSRV_REQUIRES_VALID_CONTENTS(contents);
        std::memset(pOutValue, 0, sizeof(AlbumCacheData));
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        AlbumCacheData data = {};
        nn::Result result = pService->GetAlbumCacheEx(&data, storage, contents);
        *pOutValue = data;
        return result;
    }

    nn::Result ForceAlbumUnmounted(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        NN_RESULT_DO(pService->ForceAlbumUnmounted(storage));
        NN_RESULT_SUCCESS;
    }

    nn::Result ResetAlbumMountStatus(AlbumStorageType storage) NN_NOEXCEPT
    {
        NN_CAPSRV_REQUIRES_VALID_STORAGE(storage);
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        NN_RESULT_DO(pService->ResetAlbumMountStatus(storage));
        NN_RESULT_SUCCESS;
    }

    nn::Result GetAlbumEntryFromApplicationAlbumEntry(AlbumEntry* pOutValue, const nn::capsrv::ApplicationAlbumEntry& srcEntry) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutValue);
        std::memset(pOutValue, 0, sizeof(AlbumEntry));
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        AlbumEntry entry = {};
        NN_RESULT_DO(pService->GetAlbumEntryFromApplicationAlbumEntryAruid(
            &entry,
            srcEntry,
            LibraryState::GetMyAruid()
        ));
        *pOutValue = entry;
        NN_RESULT_SUCCESS;
    }

    nn::Result SetInternalConversionEnabled(bool value) NN_NOEXCEPT
    {
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        NN_RESULT_DO(pService->SetInternalErrorConversionEnabled(value));
        NN_RESULT_SUCCESS;
    }

    nn::Result OpenAlbumAccessorSession() NN_NOEXCEPT
    {
        AlbumAccessServiceHolder::SessionPointerType pSession;

        pSession = g_AlbumAccessServiceHolder.GetSession();
        NN_RESULT_THROW_UNLESS(!pSession, ResultAlbumAlreadyOpened());

        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);
        NN_RESULT_DO(pService->OpenAccessorSession(&pSession, LibraryState::GetMyAruid()));
        g_AlbumAccessServiceHolder.StoreSession(pSession);
        NN_RESULT_SUCCESS;
    }

    void CloseAlbumAccessorSession() NN_NOEXCEPT
    {
        g_AlbumAccessServiceHolder.ReleaseSession();
    }

    //----------------------------------------
    // デバッグ機能（DevMenuCommandSystem 用）
    //----------------------------------------

    nn::Result LoadMakerNoteInfoForDebug(
        uint64_t* pOutFileSize,
        void* pOutInfoBuffer,
        size_t infoBufferSize,
        const AlbumFileId& fileId,
        void* pWorkBuffer,
        size_t workBufferSize
        ) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pOutInfoBuffer);
        NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);
        NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, AlbumFileSizeLimit_ScreenShot);

        NN_UTIL_SCOPE_EXIT{ std::memset(pWorkBuffer, 0, workBufferSize); };

        NN_CAPSRV_PROCESS_START();
        NN_CAPSRV_GET_SERVICE_POINTER(pService, g_AlbumAccessServiceHolder);

        auto result = pService->LoadMakerNoteInfoForDebug(
            pOutFileSize,
            nn::sf::OutBuffer(reinterpret_cast<char*>(pOutInfoBuffer), infoBufferSize),
            fileId,
            nn::sf::OutBuffer(reinterpret_cast<char*>(pWorkBuffer), workBufferSize)
        );
        NN_RESULT_TRY(result)
            NN_RESULT_CATCH(capsrv::ResultNotSupported)
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        NN_RESULT_END_TRY;

        NN_CAPSRV_PROCESS_SUCCESS();
        NN_RESULT_SUCCESS;
    }

}}
