﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring>
#include <algorithm>

#include <nn/album/album_AlbumFileAccess.private.h>
#include <nn/album/album_Result.h>

#include <nn/nn_Common.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/account/account_Types.h>
#include <nn/capsrv/capsrv_AlbumAccessForApplication.h>
#include <nn/capsrv/capsrv_AlbumFileContents.h>
#include <nn/capsrv/capsrv_Result.h>

#include "album_Log.h"
#include "album_LibraryState.h"
#include "album_ConvertCapsrvResult.h"

namespace nn { namespace album {

//-----------------------------------------------------------------------------
// Scoop 静止画ファイル用
//-----------------------------------------------------------------------------

// アルバムの静止画ファイルリストの取得
Result GetAlbumScreenshotFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, NN_ALIGNOF(AlbumFileEntry));
    NN_SDK_REQUIRES_GREATER(bufferLength, 0);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    auto result = capsrv::GetAlbumContentsFileListForApplication(pOutCount, reinterpret_cast<capsrv::ApplicationAlbumFileEntry*>(buffer), bufferLength, capsrv::AlbumFileContents_ScreenShot, capsrv::AlbumFileDateTimeMin, capsrv::AlbumFileDateTimeMax);
    if (!result.IsSuccess())
    {
        *pOutCount = 0;
    }
    NN_RESULT_SUCCESS;
}

// アルバムの静止画ファイルリストの取得（ユーザ識別子フィルタ付き）
Result GetAlbumScreenshotFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength, const account::Uid& uid) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, NN_ALIGNOF(AlbumFileEntry));
    NN_SDK_REQUIRES_GREATER(bufferLength, 0);
    NN_SDK_REQUIRES(uid != account::InvalidUid);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    auto result = capsrv::GetAlbumContentsFileListForApplicationEx1(pOutCount, reinterpret_cast<capsrv::ApplicationAlbumFileEntry*>(buffer), bufferLength, capsrv::AlbumFileContents_ScreenShot, uid, capsrv::AlbumFileDateTimeMin, capsrv::AlbumFileDateTimeMax);
    if (!result.IsSuccess())
    {
        *pOutCount = 0;
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// Scoop 動画ファイル用
//-----------------------------------------------------------------------------

// アルバムの動画ファイルリストの取得
Result GetAlbumMovieFileList(int* pOutCount, AlbumFileEntry buffer[], int bufferLength) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_ALIGNED(buffer, NN_ALIGNOF(AlbumFileEntry));
    NN_SDK_REQUIRES_GREATER(bufferLength, 0);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    auto result = capsrv::GetAlbumContentsFileListForApplication(pOutCount, reinterpret_cast<capsrv::ApplicationAlbumFileEntry*>(buffer), bufferLength, capsrv::AlbumFileContents_Movie, capsrv::AlbumFileDateTimeMin, capsrv::AlbumFileDateTimeMax);
    if (!result.IsSuccess())
    {
        *pOutCount = 0;
    }
    NN_RESULT_SUCCESS;
}

//-----------------------------------------------------------------------------
// メディアファイル共通
//-----------------------------------------------------------------------------

// メディアファイルの静止画の読み込み
Result LoadAlbumImage(
    int* pOutWidth,
    int* pOutHeight,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    return LoadAlbumImage(pOutWidth, pOutHeight, nullptr, nullptr, 0, pBuffer, bufferSize, pWorkBuffer, workBufferSize, entry);
}

Result LoadAlbumImage(
    int* pOutWidth,
    int* pOutHeight,
    size_t* pOutUserDataSize,
    void* pOutUserDataBuffer,
    size_t userDataBufferSize,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutWidth);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeight);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, AlbumScreenShotImageDataSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, RequiredWorkMemorySizeToLoadImage);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::ApplicationAlbumFileEntry dstEntry;
    std::memcpy(&dstEntry, &entry, sizeof(dstEntry));

    capsrv::ApplicationData applicationData;
    auto result = capsrv::LoadAlbumContentsFileScreenShotImageForApplication(pOutWidth, pOutHeight, &applicationData, pBuffer, bufferSize, pWorkBuffer, workBufferSize, dstEntry);
    if (result.IsSuccess())
    {
        if (pOutUserDataBuffer)
        {
            auto copySize = std::min(userDataBufferSize, static_cast<size_t>(applicationData.size));
            std::memcpy(pOutUserDataBuffer, applicationData.value, copySize);
        }
        if (pOutUserDataSize)
        {
            *pOutUserDataSize = applicationData.size;
        }
    }
    return ConvertCapsrvResult(result);
}

// メディアファイルのサムネイル画像の読み込み
Result LoadAlbumThumbnailImage(
    int* pOutWidth,
    int* pOutHeight,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    return LoadAlbumThumbnailImage(pOutWidth, pOutHeight, nullptr, nullptr, 0, pBuffer, bufferSize, pWorkBuffer, workBufferSize, entry);
}

Result LoadAlbumThumbnailImage(
    int* pOutWidth,
    int* pOutHeight,
    size_t* pOutUserDataSize,
    void* pOutUserDataBuffer,
    size_t userDataBufferSize,
    void* pBuffer,
    size_t bufferSize,
    void* pWorkBuffer,
    size_t workBufferSize,
    const AlbumFileEntry& entry
    ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutWidth);
    NN_SDK_REQUIRES_NOT_NULL(pOutHeight);
    NN_SDK_REQUIRES_NOT_NULL(pBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pWorkBuffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, AlbumThumbnailImageDataSize);
    NN_SDK_REQUIRES_GREATER_EQUAL(workBufferSize, RequiredWorkMemorySizeToLoadThumbnailImage);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::ApplicationAlbumFileEntry dstEntry;
    std::memcpy(&dstEntry, &entry, sizeof(dstEntry));

    capsrv::ApplicationData applicationData;
    auto result = capsrv::LoadAlbumContentsFileThumbnailImageForApplication(pOutWidth, pOutHeight, &applicationData, pBuffer, bufferSize, pWorkBuffer, workBufferSize, dstEntry);
    if (result.IsSuccess())
    {
        if (pOutUserDataBuffer)
        {
            auto copySize = std::min(userDataBufferSize, static_cast<size_t>(applicationData.size));
            std::memcpy(pOutUserDataBuffer, applicationData.value, copySize);
        }
        if (pOutUserDataSize)
        {
            *pOutUserDataSize = applicationData.size;
        }
    }
    return ConvertCapsrvResult(result);
}


//-----------------------------------------------------------------------------
// 動画ファイルの読込みストリーム操作
//-----------------------------------------------------------------------------

NN_STATIC_ASSERT(MovieFileDataUnitSize == capsrv::AlbumMovieDataUnitSize);

// 動画ファイルの読込みストリームのオープン
Result OpenAlbumMovieStream(MovieStreamHandle* pOutHandle, const AlbumFileEntry& srcEntry) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutHandle);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::ApplicationAlbumFileEntry entry;
    std::memcpy(&entry, &srcEntry, sizeof(entry));

    capsrv::AlbumMovieReadStreamHandle streamHandle;
    auto result = capsrv::OpenAlbumMovieReadStreamForApplication(&streamHandle, entry);
    pOutHandle->value = streamHandle.GetInnerValue();
    return ConvertCapsrvResult(result);
}

// 動画ファイルの読込みストリームのクローズ
void CloseAlbumMovieStream(MovieStreamHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(handle.value, 0);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::AlbumMovieReadStreamHandle streamHandle(handle.value);
    capsrv::CloseAlbumMovieReadStreamForApplication(streamHandle);
}

// 動画ファイルの動画データ部分のサイズ取得
Result GetAlbumMovieStreamSize(uint64_t* pOutSize, MovieStreamHandle handle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(handle.value, 0);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::AlbumMovieReadStreamHandle streamHandle(handle.value);
    auto result = capsrv::GetAlbumMovieReadStreamDataSizeForApplication(pOutSize, streamHandle);
    return ConvertCapsrvResult(result);
}

// 動画ファイルの動画データ部分の読込み
Result ReadAlbumMovieStream(size_t* pOutReadSize, void* buffer, size_t size, MovieStreamHandle handle, int64_t offset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(handle.value, 0);
    NN_SDK_REQUIRES_NOT_NULL(pOutReadSize);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER_EQUAL(offset, 0);
    NN_SDK_REQUIRES_ALIGNED(size, MovieFileDataUnitSize);
    NN_SDK_REQUIRES_ALIGNED(offset, MovieFileDataUnitSize);

    NN_SDK_REQUIRES(g_LibraryState.IsInitialized());
    g_LibraryState.EnsureAlbumAvailable();

    capsrv::AlbumMovieReadStreamHandle streamHandle(handle.value);
    auto result = capsrv::ReadDataFromAlbumMovieReadStreamForApplication(pOutReadSize, buffer, size, streamHandle, offset);
    return ConvertCapsrvResult(result);
}

}}  // namespace nn::album
