﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <sstream>
#include "systemi.h"
#include "exceptioni.h"

#if defined(NN_NINTENDO_SDK)
#include <nn/os.h>
#endif

#include <winext/cafe/mem.h>
#include <windows.h>

const TCHAR szAppName[] = TEXT("CAFE");

int                PCSDKMain(int argc, char** argv);

namespace {

// グローバル変数:
HINSTANCE           shInst;     // 現在のインターフェイス

}   // namespace {



namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
  Name:         GetWin32ExceptionMessage

  Description:  Win32Exception例外のメッセージを取得します。

  Arguments:    ex: Win32Exception例外への参照。

  Returns:      例外メッセージ文字列を返します。
 *---------------------------------------------------------------------------*/
std::basic_string<TCHAR>
GetWin32ExceptionMessage(const Win32Exception& ex)
{
    std::basic_stringstream<TCHAR> st;
    st << TEXT("Win32 API error occurs. - ");

    LPTSTR lpMsgBuf = 0;
    FormatMessage(
              FORMAT_MESSAGE_ALLOCATE_BUFFER
            | FORMAT_MESSAGE_FROM_SYSTEM
            | FORMAT_MESSAGE_IGNORE_INSERTS,
            NULL,
            ex.GetErrorCode(),
            MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
            reinterpret_cast<LPTSTR>(&lpMsgBuf),
            0,
            NULL);
    st << lpMsgBuf;
    LocalFree(lpMsgBuf);

    st << std::endl;
    return st.str();
}

/*---------------------------------------------------------------------------*
  Name:         ReportWin32Exception

  Description:  Win32Exception例外のメッセージを出力します。

  Arguments:    ex: Win32Exception例外への参照。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ReportWin32Exception(const Win32Exception& ex)
{
    OutputDebugString(GetWin32ExceptionMessage(ex).c_str());
}

namespace {

/*---------------------------------------------------------------------------*
  Name:         CommonMain

  Description:  GUIアプリケーションの共通のエントリポイント

  Arguments:    hInstance: インスタンスのハンドル
                nCmdShow:  表示状態

  Returns:      アプリケーションの終了コード。
 *---------------------------------------------------------------------------*/
int
CommonMain(
    HINSTANCE   hInstance,
    int         nCmdShow
)
{
    shInst = hInstance;     // グローバル変数にインスタンス処理を格納します。

    int exitCode = 0;
    try
    {
        nw::internal::winext::Init_OS_System();
        nw::internal::winext::InitCommandWindow();
        nw::internal::winext::CreateMainWindow();

        try
        {
            static const int ARENA_SIZE = 256 * 1024 * 1024;
            void* arena = std::malloc(ARENA_SIZE);
            MEMInitDefaultHeap(arena,ARENA_SIZE);
            exitCode = PCSDKMain(__argc, __argv);
            std::free(arena);
        }
        catch (ApplicationExitException& ex)
        {
            exitCode = ex.GetExitCode();
        }

        if (winext::IsCreateMainWindow())
        {
            DestroyMainWindw();
        }

        nw::internal::winext::Finalize_OS_System();
    }
    catch (Win32Exception& ex)
    {
        std::basic_string<TCHAR> msg = GetWin32ExceptionMessage(ex);
        MessageBox(NULL, msg.c_str(), szAppName, MB_ICONERROR);
        OutputDebugString(msg.c_str());
        return 1;
    }
    catch (MultimediaApiException& ex)
    {
        std::basic_stringstream<TCHAR> st;
        st << TEXT("Windows Multimedia API error occurs. - " << ex.GetResultCode()) << std::endl;
        OutputDebugString(st.str().c_str());

        return 1;
    }
    catch (MultimediaTimerSetEventException&)
    {
        OutputDebugString(TEXT("Windows Multimedia API timeSetEvent error occurs.\n"));
        return 1;
    }

    return exitCode;
}

}   // namespace {
}   // namespace winext {
}   // namespace internal {
}   // namespace nw


#if defined(NN_NINTENDO_SDK)
/*---------------------------------------------------------------------------*
  Name:         nnMain

  Description:  コンソールアプリケーションのエントリポイント(NintendoSDK 用)

  Arguments:    なし（必要なら nn::os::GetHostArgc(), nn::os::GetHostArgv() で取得可能）
  Returns:      なし
 *---------------------------------------------------------------------------*/
extern "C" void nnMain()
{
    nw::internal::winext::CommonMain(GetModuleHandle(NULL), SW_SHOWDEFAULT);
}

extern "C" void nninitStartup()
{
    // CommonMain() にて ARENA_SIZE を確保可能にするために、デフォルトの初期化処理を上書きします。
    // 特別な初期化はなにも行いません。
}

#else // defined(NN_NINTENDO_SDK)
/*---------------------------------------------------------------------------*
  Name:         main

  Description:  コンソールアプリケーションのエントリポイント(ANSI用)

  Arguments:    argc: コマンドライン引数の数
                argv: コマンドライン引数

  Returns:      アプリケーションの終了コード。
 *---------------------------------------------------------------------------*/
int
main(
    int     /* argc  */,
    char*   /* argv[] */
)
{
    return nw::internal::winext::CommonMain(GetModuleHandle(NULL), SW_SHOWDEFAULT);
}

/*---------------------------------------------------------------------------*
  Name:         wmain

  Description:  コンソールアプリケーションのエントリポイント(UNICODE用)

  Arguments:    argc: コマンドライン引数の数
                argv: コマンドライン引数

  Returns:      アプリケーションの終了コード。
 *---------------------------------------------------------------------------*/
int
wmain(
    int         /* argc */,
    wchar_t*    /* argv[] */
)
{
    return nw::internal::winext::CommonMain(GetModuleHandle(NULL), SW_SHOWDEFAULT);
}

/*---------------------------------------------------------------------------*
  Name:         WinMain

  Description:  GUIアプリケーションのエントリポイント(ANSI用)

  Arguments:    hInstance:     現在のインスタンスのハンドル
                hPrevInstance: 以前のインスタンスのハンドル
                lpCmdLine:     コマンドライン
                nCmdShow:      表示状態

  Returns:      アプリケーションの終了コード。
 *---------------------------------------------------------------------------*/
int APIENTRY
WinMain(
    HINSTANCE   hInstance,
    HINSTANCE   hPrevInstance,
    LPSTR       lpCmdLine,
    int         nCmdShow
)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    return nw::internal::winext::CommonMain(hInstance, nCmdShow);
}

/*---------------------------------------------------------------------------*
  Name:         wWinMain

  Description:  GUIアプリケーションのエントリポイント(UNICODE用)

  Arguments:    hInstance:     現在のインスタンスのハンドル
                hPrevInstance: 以前のインスタンスのハンドル
                lpCmdLine:     コマンドライン
                nCmdShow:      表示状態

  Returns:      アプリケーションの終了コード。
 *---------------------------------------------------------------------------*/
int APIENTRY
wWinMain(
    HINSTANCE   hInstance,
    HINSTANCE   hPrevInstance,
    LPWSTR      lpCmdLine,
    int         nCmdShow
)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);

    return nw::internal::winext::CommonMain(hInstance, nCmdShow);
}
#endif // defined(NN_NINTENDO_SDK)

