﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/os.h>
#include <algorithm>
#include "systemi.h"
#include "exceptioni.h"

#include <winext/cafe/pad.h>
#include <winext/cafe/os/win32/os_WindowMessage.h>

#include <Windows.h>

const TCHAR szWindowClass[] = TEXT("CTR");  // メイン ウィンドウ クラス名
const TCHAR GlWindowClass[] = TEXT("GlWindow"); // OpenGLウィンドウクラス名
const TCHAR SubWindowClass[] = TEXT("SubWindow"); // 2画面用ウィンドウクラス名

// LCDタイプの拡張ウィンドウメモリのオフセットとサイズ
const int           LcdTypeOffset   = 0;
const int           LcdTypeSize     = 4;

namespace nw {
namespace internal {
namespace winext {

    HWND                ghCommandWnd;  // コマンドウィンドウのハンドル
    HWND                ghFrameWnd;    // フレームウィンドウのハンドル

// パッドの押下状態
USHORT              gPadStatus;

//
WindowMessageCallback   shWindowMessageCallback = NULL;

namespace {

/*---------------------------------------------------------------------------*
  Name:         FindCommandWindow

  Description:  コマンドウィンドウを検索します。
 *---------------------------------------------------------------------------*/
BOOL
CALLBACK FindCommandWindow(HWND hWnd, LPARAM lParam)
{
    DWORD processId = 0;
    ::GetWindowThreadProcessId(hWnd, &processId);

    if(lParam == processId && IsWindowVisible(hWnd))
    {
        ghCommandWnd = hWnd;
        return FALSE;
    }

    return TRUE;
}

/*---------------------------------------------------------------------------*
  Name:         ThRegisterClass

  Description:  RegisterClassEx() を呼び出します。
                関数が失敗した場合は、nn::os::detail::Win32Exception 例外を
                送出します。

  Arguments:    lpwcx: クラスの情報を含むWNDCLASSEX構造体へのポインタ

  Returns:      クラスを一意的に識別するATOMを返します。
 *---------------------------------------------------------------------------*/
ATOM
ThRegisterClass(const WNDCLASSEX* lpwcx)
{
    if (const ATOM atom = RegisterClassEx(lpwcx))
    {
        return atom;
    }

    throw internal::winext::Win32Exception(GetLastError());
}


/*---------------------------------------------------------------------------*
  Name:         ThCreateWindow

  Description:  CreateWindowEx() を呼び出します。
                関数が失敗した場合は、nn::os::detail::Win32Exception 例外を
                送出します。

  Arguments:    dwExStyle:    拡張ウィンドウスタイル
                lpClassName:  登録されているクラス名
                lpWindowName: ウィンドウ名
                dwStyle:      ウィンドウスタイル
                x:            ウィンドウの横方向の位置
                y:            ウィンドウの縦方向の位置
                nWidth:       ウィンドウの幅
                nHeight:      ウィンドウの高さ
                hWndParent:   親ウィンドウまたはオーナーウィンドウのハンドル
                hInstance:    アプリケーションインスタンスのハンドル
                lpParam:      ウィンドウ作成用パラメータ

  Returns:      ウィンドウハンドルを返します。
 *---------------------------------------------------------------------------*/
HWND
ThCreateWindow(
    DWORD       dwExStyle,
    LPCTSTR     lpClassName,
    LPCTSTR     lpWindowName,
    DWORD       dwStyle,
    int         x,
    int         y,
    int         nWidth,
    int         nHeight,
    HWND        hWndParent,
    HINSTANCE   hInstance,
    LPVOID      lpParam         = 0
)
{
    if (const HWND hWnd = CreateWindowEx(
        dwExStyle,
        lpClassName,
        lpWindowName,
        dwStyle,
        x,
        y,
        nWidth,
        nHeight,
        hWndParent,
        NULL,               //   HMENU hMenu,
        hInstance,
        lpParam)
    )
    {
        return hWnd;
    }

    throw internal::winext::Win32Exception(GetLastError());
}

/*---------------------------------------------------------------------------*
  Name:         SetLcdType

  Description:  LCDタイプをウィンドウに関連付けます。

  Arguments:    hWnd:    ウィンドウハンドル
                lcdType: LCDタイプ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
SetLcdType(
    HWND    hWnd,
    int     lcdType
)
{
    SetLastError(0);
    const LONG result = SetWindowLong(hWnd, LcdTypeOffset, lcdType + 1);
    DWORD errorCode = 0;
    if (result == 0 && 0 != (errorCode = GetLastError()))
    {
        throw internal::winext::Win32Exception(errorCode);
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetLcdType

  Description:  ウィンドウに関連付けられているLCDタイプを取得します。

  Arguments:    hWnd: ウィンドウハンドル

  Returns:      LCDタイプを返します。
 *---------------------------------------------------------------------------*/
int
GetLcdType(HWND hWnd)
{
    SetLastError(0);
    LONG result = GetWindowLong(hWnd, LcdTypeOffset);
    DWORD errorCode = 0;
    if (result == 0 && 0 != (errorCode = GetLastError()))
    {
        throw internal::winext::Win32Exception(errorCode);
    }
    return result - 1;
}

/*---------------------------------------------------------------------------*
  Name:         GlWndProc

  Description:  ウィンドウのメッセージを処理します。

  Arguments:    hWnd:    ウィンドウハンドル
                message: メッセージ
                wParam:  メッセージパラメータ
                lParam:  メッセージパラメータ

  Returns:      メッセージの処理結果に応じた値。
 *---------------------------------------------------------------------------*/
LRESULT CALLBACK
GlWndProc(
    HWND    hWnd,
    UINT    message,
    WPARAM  wParam,
    LPARAM  lParam
)
{
    //コールバックが仕掛けられていたら、それを呼び出す
    if (shWindowMessageCallback != NULL)
    {
        BOOL handled = FALSE;
        LRESULT result = shWindowMessageCallback(hWnd, message, wParam, lParam, &handled);
        //処理された場合には戻り値0で戻る
        if (handled)
        {
            return result;
        }
    }

    switch(message)
    {
    case WM_CREATE:
        try
        {
            const int lcdType = reinterpret_cast<LONG>(reinterpret_cast<const CREATESTRUCT*>(lParam)->lpCreateParams);
            SetLcdType(hWnd, lcdType);
        }
        catch (internal::winext::Win32Exception& ex)
        {
            internal::winext::ReportWin32Exception(ex);
            return -1;  // -1 を返すとCreateWindow()がNULLを返す。
        }
        return 0;   // メッセージを処理した。

    case WM_PAINT:
        {
            PAINTSTRUCT ps;
            if (const HDC hdc = BeginPaint(hWnd, &ps))
            {
                // 現状では、ただ無効化領域をクリアする目的のためだけにこの
                // WM_PAINTハンドラがあります。
                EndPaint(hWnd, &ps);
            }
        }
        return 0;   // メッセージを処理した。

    case WM_ERASEBKGND:
        {
            // OpenGLで背景の消去を行うのでGDIでは何もしないように、
            // このメッセージを処理しておきます。
            //
            // 戻り値に関しては、BeginPaintで取得するPAINTSTRUCT構造体のfEraseメンバの
            // 値に影響しますが、どの道このウィンドウプロシージャでは
            // WM_PAINTにおいて何もしないため、
            // TRUE でも FALSE でも結果は変わらないと思われます。

            BOOL bBackgroundErased = FALSE; // 背景を消去していない。
            return bBackgroundErased;
        }
    }

    return DefWindowProc(hWnd,message,wParam,lParam);
}

/*---------------------------------------------------------------------------*
  Name:         RegisterGlWindowClass

  Description:  ウィンドウクラスを登録します。

  Arguments:    hInstance: インスタンスのハンドル

  Returns:      ウィンドウクラスを一意に識別するATOMが返ります。
 *---------------------------------------------------------------------------*/
ATOM
RegisterGlWindowClass(HINSTANCE hInstance)
{
    WNDCLASSEX wcex = { sizeof(WNDCLASSEX) };

    wcex.style			=     0
                        //  | CS_HREDRAW    // サイズ変更により全体を書き直す
                        //  | CS_VREDRAW    //   OpenGLにより常に描画するので指定しないでおきます。
                            | CS_OWNDC;     // OpenGLで描画する場合は、DCを占有するので
                                            // Windows毎に割り当てられるDC(プライベートDC)に
                                            // しておきます。
    wcex.lpfnWndProc	= GlWndProc;
    wcex.hInstance		= hInstance;
    wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground  = NULL;             // OpenGLで背景の消去を行うので、ここにはブラシを
                                            // 登録しないでおきます。
    wcex.lpszClassName	= GlWindowClass;

    wcex.cbWndExtra     = LcdTypeSize;      // 拡張ウィンドウメモリのサイズ

    return ThRegisterClass(&wcex);
}

/*---------------------------------------------------------------------------*
  Name:         CreateGlWindow

  Description:  ウィンドウを作成します。

  Arguments:    x:          ウィンドウの横方向の初期位置
                y:          ウィンドウの縦方向の初期位置
                nWidth:     ウィンドウの幅
                nHeight:    ウィンドウの高さ
                hWndParent: 親ウィンドウ
                lcdType:    LCDタイプ

  Returns:      ウィンドウハンドルを返します。
 *---------------------------------------------------------------------------*/
HWND
CreateGlWindow(
    int         x,
    int         y,
    int         nWidth,
    int         nHeight,
    HWND        hWndParent,
    HINSTANCE   hInstance,
    int         lcdType
)
{
    const int dwStyle = WS_CHILD
                    |   WS_VISIBLE
                    |   WS_CLIPCHILDREN
                    |   WS_CLIPSIBLINGS
                ;

    return ThCreateWindow(
        0,                      // 拡張ウィンドウスタイル
        GlWindowClass,          // window class
        TEXT(""),               // window title
        dwStyle,                // ウィンドウスタイル
        x,
        y,
        nWidth,
        nHeight,
        hWndParent,
        hInstance,
        reinterpret_cast<LPVOID>(lcdType)); // creation parameter
}


/*---------------------------------------------------------------------------*
  Name:         ReportEglException

  Description:  EglException例外のメッセージを出力します。

  Arguments:    ex: EglException例外への参照。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
ReportEglException(const EglException& ex)
{
    OSReport("egl error. - [%d]\n", ex.GetErrorCode());
}

/*---------------------------------------------------------------------------*
  Name:         UpdateKeyStatus

  Description:  パッドの扱いとするキーの入力状態を更新します。

  Arguments:    keyCode: 仮想キーコード。
                bDown:   キーが押されている場合は真、離された場合は偽。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
UpdateKeyStatus(
    DWORD   keyCode,
    bool    bDown
)
{
    u16 padType = 0;
    switch (keyCode)
    {
    case 'A':       padType = NW_WINEXT_PAD_BUTTON_A;        break;
    case 'B':       padType = NW_WINEXT_PAD_BUTTON_B;        break;
    case 'E':       padType = NW_WINEXT_PAD_BUTTON_MENU;     break;
    case 'T':       padType = NW_WINEXT_PAD_BUTTON_START;    break;
    case VK_RIGHT:  padType = NW_WINEXT_PAD_BUTTON_RIGHT;    break;
    case VK_LEFT:   padType = NW_WINEXT_PAD_BUTTON_LEFT;     break;
    case VK_UP:     padType = NW_WINEXT_PAD_BUTTON_UP;       break;
    case VK_DOWN:   padType = NW_WINEXT_PAD_BUTTON_DOWN;     break;
    case 'Z':       padType = NW_WINEXT_PAD_TRIGGER_Z;       break;
    case 'R':       padType = NW_WINEXT_PAD_TRIGGER_R;       break;
    case 'L':       padType = NW_WINEXT_PAD_TRIGGER_L;       break;
    case 'X':       padType = NW_WINEXT_PAD_BUTTON_X;        break;
    case 'Y':       padType = NW_WINEXT_PAD_BUTTON_Y;        break;
//    case 'D':       padType = PAD_BUTTON_DEBUG;    break;
    }

    if (padType == 0)   // チェック対象のキーが押されていない
    {
        return ;
    }

    if (bDown)
    {
        gPadStatus |= padType;
    }
    else
    {
        gPadStatus &= ~padType;
    }
}

/*---------------------------------------------------------------------------*
  Name:         FrameWndProc

  Description:  フレームウィンドウのメッセージを処理します。

  Arguments:    hWnd:    ウィンドウハンドル
                message: メッセージ
                wParam:  メッセージパラメータ
                lParam:  メッセージパラメータ

  Returns:      メッセージの処理結果に応じた値。
 *---------------------------------------------------------------------------*/
LRESULT CALLBACK
FrameWndProc(
    HWND    hWnd,
    UINT    message,
    WPARAM  wParam,
    LPARAM  lParam
)
{
    //コールバックが仕掛けられていたら、それを呼び出す
    if (shWindowMessageCallback != NULL)
    {
        BOOL handled = FALSE;
        LRESULT result = shWindowMessageCallback(hWnd, message, wParam, lParam, &handled);
        //処理された場合には戻り値0で戻る
        if (handled)
        {
            return result;
        }
    }

    switch(message)
    {
    case WM_CREATE:
        try
        {
            const HINSTANCE hInstance = reinterpret_cast<HINSTANCE>(GetWindowLong(hWnd, GWL_HINSTANCE));
//            InitGl(hInstance, hWnd);
        }
        catch (EglException& ex)
        {
            ReportEglException(ex);
            return -1;  // -1 を返すとCreateWindow()がNULLを返す。
        }
        catch (internal::winext::Win32Exception& ex)
        {
            ReportWin32Exception(ex);
            return -1;  // -1 を返すとCreateWindow()がNULLを返す。
        }
        return 0;   // メッセージを処理した。

    case WM_PAINT:
        {
            PAINTSTRUCT ps;
            if (const HDC hdc = BeginPaint(hWnd, &ps))
            {
                // TODO: 描画コードをここに追加してください...
                EndPaint(hWnd, &ps);
            }
        }
        return 0;   // メッセージを処理した。

    case WM_CLOSE:
        // ウィンドウを破棄してしまうと、以後の GL API 呼び出しが失敗するので
        // ウィンドウを破棄する前に WM_CLOSE でメインループを抜けさせます。
        PostQuitMessage(0);
        return 0;   // メッセージを処理した。

    case WM_KEYDOWN:
        UpdateKeyStatus(wParam, true);
        return 0;

    case WM_KEYUP:
        UpdateKeyStatus(wParam, false);
        return 0;
    }

    return DefWindowProc(hWnd, message, wParam, lParam);
}

/*---------------------------------------------------------------------------*
  Name:         RegisterFrameWindowClass

  Description:  ウィンドウクラスを登録します。

  Arguments:    hInstance: インスタンスのハンドル

  Returns:      ウィンドウクラスを一意に識別するATOMが返ります。
 *---------------------------------------------------------------------------*/
ATOM
RegisterFrameWindowClass(HINSTANCE hInstance)
{
    WNDCLASSEX wcex = { sizeof(WNDCLASSEX) };

    wcex.style			=     CS_DBLCLKS
                         /* | CS_HREDRAW
                            | CS_VREDRAW */
                            ;
    wcex.lpfnWndProc    = FrameWndProc;
    wcex.hInstance      = hInstance;
    wcex.hIcon          = LoadIcon(NULL, IDI_APPLICATION);
    wcex.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground  = reinterpret_cast<HBRUSH>(COLOR_BTNFACE + 1);
    wcex.lpszMenuName   = NULL; // メニューなし。
    wcex.lpszClassName  = szWindowClass;
    wcex.hIconSm        = NULL; // hIcon より適切に判断してもらう。

    return ThRegisterClass(&wcex);
}

/*---------------------------------------------------------------------------*
  Name:         CreateFrameWindow

  Description:  フレームウィンドウを作成します。

  Arguments:    hInstance: インスタンスのハンドル
                nCmdShow:  表示状態

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CreateFrameWindow(
    HINSTANCE   hInstance,
    int         nCmdShow
)
{
    const int dwStyle =
                        WS_OVERLAPPED
                    |   WS_CAPTION
                    |   WS_SYSMENU
                //  |   WS_THICKFRAME
                    |   WS_MINIMIZEBOX
                //  |   WS_MAXIMIZEBOX
                    |   WS_CLIPCHILDREN // OpenGLウィンドウに描画されるとちらつくのでクリップします。
                ;

    const DWORD dwExStyle = 0;

    RECT windowRect = { 0, 0, 0 + 1280, 0 + 720 };
    BOOL bSuccess = AdjustWindowRectEx(
        &windowRect,            // クライアント領域の座標が入った構造体へのポインタ
        dwStyle,                // ウィンドウスタイル
        false,                  // メニューを持つかどうかの指定
        dwExStyle);             // 拡張ウィンドウスタイル
    if (! bSuccess)
    {
        throw new internal::winext::Win32Exception(GetLastError());
    }

    RECT workAreaRect;
    SystemParametersInfo( SPI_GETWORKAREA, 0, &workAreaRect, 0 );

    HWND hWnd = ThCreateWindow(
        dwExStyle,
        szWindowClass,          // window class
        TEXT("Cafe-EMU"),       // window title
        dwStyle,                // window style
        workAreaRect.left,      // x
        workAreaRect.top,       // y
        windowRect.right - windowRect.left,
        windowRect.bottom - windowRect.top,
        NULL,                   // parent window
        hInstance);             // program instance

    ShowWindow(hWnd, nCmdShow);
    bSuccess = UpdateWindow(hWnd);
    if (! bSuccess)
    {
        throw new internal::winext::Win32Exception(GetLastError());
    }

    ghFrameWnd = hWnd;
}

}   // namespace

/*---------------------------------------------------------------------------*
  Name:         InitCommandWindow

  Description:  コマンドウィンドウを初期化します。
 *---------------------------------------------------------------------------*/
void
InitCommandWindow()
{
    ::EnumWindows(FindCommandWindow, ::GetCurrentProcessId());

    RECT workAreaRect;
    SystemParametersInfo( SPI_GETWORKAREA, 0, &workAreaRect, 0 );

    RECT commandWindowRect;
    GetWindowRect( ghCommandWnd, &commandWindowRect );

    // コマンドウィンドウ位置を調整
    if(ghCommandWnd != NULL)
    {
        SetWindowPos(
            ghCommandWnd,
            NULL,
            workAreaRect.right - ( commandWindowRect.right - commandWindowRect.left ),
            workAreaRect.bottom - ( commandWindowRect.bottom - commandWindowRect.top ),
            0,
            0,
            SWP_NOSIZE);
    }
}

/*---------------------------------------------------------------------------*
  Name:         CreateMainWindow

  Description:  ウィンドウを作成します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
CreateMainWindow()
{
    const HINSTANCE hInstance = GetModuleHandle(NULL);

    RegisterFrameWindowClass(hInstance);

    // フレームメインウィンドウを作成します。
    CreateFrameWindow(hInstance, SW_SHOWDEFAULT);
}

/*---------------------------------------------------------------------------*
  Name:         DestroyMainWindw

  Description:  ウィンドウを破棄します。

  Arguments:    なし。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
DestroyMainWindw()
{
    if (ghFrameWnd != NULL)
    {
        try
        {
//            FinalizeGl();
        }
        catch (EglException& ex)
        {
            ReportEglException(ex);
        }
        DestroyWindow(ghFrameWnd);    // 子ウィンドウもまとめて削除される。

        ghFrameWnd = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         IsCreateMainWindow

  Description:  メインウィンドウが作成されているかどうかを判別する値を取得します。

  Arguments:    なし。

  Returns:      メインウィンドウが作成されていたら真を返します。
 *---------------------------------------------------------------------------*/
bool
IsCreateMainWindow()
{
    return ghFrameWnd != NULL;
}

/*---------------------------------------------------------------------------*
  Name:         CreateSubWindow

  Description:  2 画面用のウィンドウを作成します。

  x:            ウィンドウの横方向の位置
  y:            ウィンドウの縦方向の位置
  nWidth:       ウィンドウの幅
  nHeight:      ウィンドウの高さ
  hWndParent:   親ウィンドウまたはオーナーウィンドウのハンドル
  hInstance:    アプリケーションインスタンスのハンドル

  Returns:      ウィンドウハンドルを返します。
 *---------------------------------------------------------------------------*/
HWND
CreateSubWindow(
    int         x,
    int         y,
    int         nWidth,
    int         nHeight,
    HWND        hWndParent,
    HINSTANCE   hInstance
)
{
    const int dwStyle =
        WS_OVERLAPPED
        |   WS_CAPTION
        |   WS_SYSMENU
        |   WS_MINIMIZEBOX
        |   WS_CLIPCHILDREN
        ;

    const DWORD dwExStyle = 0;

    return nw::internal::winext::ThCreateWindow(
        dwExStyle,              // 拡張ウィンドウスタイル
        SubWindowClass,         // window class
        TEXT(""),               // window title
        dwStyle,                // ウィンドウスタイル
        x,
        y,
        nWidth,
        nHeight,
        hWndParent,
        hInstance
    );
}

/*---------------------------------------------------------------------------*
  Name:         RegisterSubWindowClass

  Description:  2 画面用のウィンドウクラスを登録します。

  Arguments:    hInstance: インスタンスのハンドル

  Returns:      ウィンドウクラスを一意に識別するATOMが返ります。
 *---------------------------------------------------------------------------*/
ATOM
RegisterSubWindowClass(HINSTANCE hInstance)
{
    WNDCLASSEX wcex = { sizeof(WNDCLASSEX) };

    wcex.style          = 0;
    wcex.lpfnWndProc    = nw::internal::winext::FrameWndProc;
    wcex.hInstance      = hInstance;
    wcex.hIcon          = LoadIcon(NULL, IDI_APPLICATION);
    wcex.hCursor        = LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground  = reinterpret_cast<HBRUSH>(COLOR_BTNFACE + 1);
    wcex.lpszMenuName   = NULL; // メニューなし。
    wcex.lpszClassName  = SubWindowClass;
    wcex.hIconSm        = NULL; // hIcon より適切に判断してもらう。

    return nw::internal::winext::ThRegisterClass(&wcex);
}

}   // namespace winext
}   // namespace internal
}   // namespace nw
