﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/mem/unitHeap.h>
#include "heapCommoni.h"

namespace nw {
namespace internal {
namespace winext {

/* ========================================================================
    マクロ定数
   ======================================================================== */

// アライメントの最小値
#define MIN_ALIGNMENT           4


/* ========================================================================
    static関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    メモリブロックリスト操作
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         PopMBlock_

  Description:  リストの先頭からメモリブロックを取得します。

  Arguments:    link:   リストへのポインタ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static MEMiUntHeapMBlockHead*
PopMBlock_( MEMiUntMBlockList* list )
{
    MEMiUntHeapMBlockHead*  block = list->head;
    if ( block )
    {
        list->head = block->pMBlkHdNext;
    }

    return block;
}

/*---------------------------------------------------------------------------*
  Name:         PushMBlock_

  Description:  メモリブロックをリストの先頭に追加します。

  Arguments:    link:   追加するリスト
                block:  追加するメモリブロック

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
PushMBlock_(
    MEMiUntMBlockList*       list,
    MEMiUntHeapMBlockHead*   block
)
{
    block->pMBlkHdNext = list->head;
    list->head = block;
}


/*---------------------------------------------------------------------------*
  Name:         GetUnitHeapHeadPtrFromHeapHead_

  Description:  ヒープヘッダへのポインタから、ユニットヒープヘッダへのポインタを取得します。

  Arguments:    pHeapHd:  ヒープヘッダへのポインタ。

  Returns:      ユニットヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiUntHeapHead*
GetUnitHeapHeadPtrFromHeapHead_( MEMiHeapHead* pHeapHd )
{
    return (MEMiUntHeapHead*)AddU32ToPtr( pHeapHd, sizeof(MEMiHeapHead) );
}


/*---------------------------------------------------------------------------*
  Name:         IsValidUnitHeapHandle_

  Description:  有効なヒープハンドルかどうかシグニチャをチェックします

  Arguments:    handle:  ヒープハンドル

  Returns:      TRUE    有効なヒープハンドルである場合
                FALSE   有効なヒープハンドルではない場合
 *---------------------------------------------------------------------------*/
static inline BOOL
IsValidUnitHeapHandle_( MEMHeapHandle handle )
{
    if ( handle == MEM_HEAP_INVALID_HANDLE )
    {
        return FALSE;
    }

    {
        MEMiHeapHead* pHeapHd = handle;
        return  pHeapHd->signature == MEMi_UNTHEAP_SIGNATURE;
    }
}


/* ========================================================================
    外部関数(非公開)
   ======================================================================== */


/*---------------------------------------------------------------------------*
  Name:         MEMiDumpUnitHeap

  Description:  ユニットヒープ内部の情報を表示します。
                これはデバッグ用の関数です。

  Arguments:    heap:    ユニットヒープのハンドル。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
#if defined(_DEBUG)

void
MEMiDumpUnitHeap( MEMHeapHandle heap )
{
    //ASSERT( IsValidUnitHeapHandle_( heap ) );

    {
        MEMiHeapHead    *const pHeapHd     = heap;
        MEMiUntHeapHead *const pUnitHeapHd = GetUnitHeapHeadPtrFromHeapHead_( pHeapHd );
        const u32 heapSize = GetOffsetFromPtr( pHeapHd->heapStart, pHeapHd->heapEnd );
        const u32 freeSize = MEMCountFreeBlockForUnitHeap( heap ) * pUnitHeapHd->mBlkSize;
        const u32 usedSize = heapSize - freeSize;

        MEMiDumpHeapHead(pHeapHd);

        OSReport("    %d / %d bytes (%6.2f%%) used\n",
                                                usedSize, heapSize, 100.0f * usedSize / heapSize);
    }
}

// #if defined(_DEBUG)
#endif


/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMCreateUnitHeapEx

  Description:  ユニットヒープを作成します。

  Arguments:    startAddress:  ヒープ領域の先頭アドレス。
                heapSize:      ヒープ領域のサイズ。
                memBlockSize:  メモリブロックのサイズ。
                alignment:     メモリブロックのアライメント。
                               4,8,16,32のいずれかの値が指定できます。
                optFlag:       オプションフラグ。

  Returns:      関数が成功した場合、作成されたユニットヒープのハンドルが返ります。
                関数が失敗すると、MEM_INVALID_HEAP_HANDLE が返ります。
 *---------------------------------------------------------------------------*/
MEMHeapHandle
MEMCreateUnitHeapEx(
    void*   startAddress,
    u32     heapSize,
    u32     memBlockSize,
    int     alignment,
    u16     optFlag
)
{
    MEMiHeapHead* pHeapHd;
    void* heapEnd;

    //ASSERT(startAddress != NULL);

    // alignment のチェック
    //ASSERT( alignment % MIN_ALIGNMENT == 0 );
    //ASSERT( MIN_ALIGNMENT <= alignment && alignment <= 32 );

    pHeapHd = (MEMiHeapHead*)RoundUpPtr( startAddress, MIN_ALIGNMENT );
    heapEnd = RoundDownPtr( AddU32ToPtr( startAddress, heapSize ), MIN_ALIGNMENT );

    if ( ComparePtr( pHeapHd, heapEnd ) > 0 )
    {
        return MEM_HEAP_INVALID_HANDLE;
    }

    memBlockSize = RoundUp( memBlockSize, alignment );    // 実質のブロックサイズ

    {
        MEMiUntHeapHead* pUntHeapHd = GetUnitHeapHeadPtrFromHeapHead_( pHeapHd );
        void* heapStart = RoundUpPtr( AddU32ToPtr( pUntHeapHd, sizeof(MEMiUntHeapHead) ), alignment );
        u32 elementNum;

        if ( ComparePtr( heapStart, heapEnd ) > 0 )
        {
            return MEM_HEAP_INVALID_HANDLE;
        }

        elementNum = GetOffsetFromPtr( heapStart, heapEnd ) / memBlockSize;
        if ( elementNum == 0 )
        {
            return MEM_HEAP_INVALID_HANDLE;
        }

        heapEnd = AddU32ToPtr( heapStart, elementNum * memBlockSize );

        MEMiInitHeapHead(           // ヒープ共通初期化
                pHeapHd,
                MEMi_UNTHEAP_SIGNATURE,
                heapStart,
                heapEnd,
                optFlag );

        pUntHeapHd->mbFreeList.head = (MEMiUntHeapMBlockHead*)heapStart;
        pUntHeapHd->mBlkSize = memBlockSize;

        {
            MEMiUntHeapMBlockHead* pMBlkHd = pUntHeapHd->mbFreeList.head;
            int i;

            for ( i = 0; i < (int)elementNum - 1; ++i, pMBlkHd = pMBlkHd->pMBlkHdNext )
            {
                pMBlkHd->pMBlkHdNext = (MEMiUntHeapMBlockHead*)AddU32ToPtr( pMBlkHd, memBlockSize );
            }

            pMBlkHd->pMBlkHdNext = NULL;
        }

        return pHeapHd;
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMDestroyUnitHeap

  Description:  ユニットヒープを破棄します。

  Arguments:    heap: ユニットヒープのハンドル。

  Returns:      破棄したヒープが占めていた領域へのポインタを返します。
 *---------------------------------------------------------------------------*/
void*
MEMDestroyUnitHeap( MEMHeapHandle heap )
{
    //ASSERT( IsValidUnitHeapHandle_(heap) );

    MEMiFinalizeHeap(heap);
    return (void*)heap;
}


/*---------------------------------------------------------------------------*
  Name:         MEMAllocFromUnitHeap

  Description:  ユニットヒープからメモリブロックを確保します。

  Arguments:    heap:   ユニットヒープのハンドル。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
void*
MEMAllocFromUnitHeap( MEMHeapHandle heap )
{
    MEMiUntHeapMBlockHead* pMBlkHd;

    //ASSERT( IsValidUnitHeapHandle_( heap ) );

    {

        MEMiUntHeapHead* pUntHeapHd = GetUnitHeapHeadPtrFromHeapHead_( heap );

        LockHeap( heap );
        pMBlkHd                     = PopMBlock_( &pUntHeapHd->mbFreeList );
        UnlockHeap( heap );

        if ( pMBlkHd )
        {
            FillAllocMemory( heap, pMBlkHd, pUntHeapHd->mBlkSize );
        }
    }
    return pMBlkHd;
}

/*---------------------------------------------------------------------------*
  Name:         MEMFreeToUnitHeap

  Description:  ユニットヒープへメモリブロックを返却します。

  Arguments:    heap:     ユニットヒープのハンドル。
                memBlock: 返却するメモリブロックへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMFreeToUnitHeap(
    MEMHeapHandle   heap,
    void*           memBlock
)
{
    //ASSERT( IsValidUnitHeapHandle_( heap ) );
    if ( memBlock == NULL )
    {
        return;
    }

    {
        MEMiUntHeapHead* pUntHeapHd = GetUnitHeapHeadPtrFromHeapHead_( heap );

        FillFreeMemory( heap, memBlock, pUntHeapHd->mBlkSize );

        LockHeap( heap );
        PushMBlock_( &pUntHeapHd->mbFreeList, (MEMiUntHeapMBlockHead*)memBlock );
        UnlockHeap( heap );
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMCountFreeBlockForUnitHeap

  Description:  ユニットヒープの空きメモリブロック数を取得します。

  Arguments:    heap:     ユニットヒープのハンドル。

  Returns:      ユニットヒープの空きメモリブロック数を返します。
 *---------------------------------------------------------------------------*/
u32
MEMCountFreeBlockForUnitHeap( MEMHeapHandle heap )
{
    u32 cnt = 0;

    //ASSERT(IsValidUnitHeapHandle_(heap));

    LockHeap( heap );
    {
        MEMiUntHeapHead*       pUntHeapHd = GetUnitHeapHeadPtrFromHeapHead_( heap );
        MEMiUntHeapMBlockHead* pMBlkHd    = pUntHeapHd->mbFreeList.head;

        for ( ; pMBlkHd; pMBlkHd = pMBlkHd->pMBlkHdNext )
        {
            ++cnt;
        }
    }
    UnlockHeap( heap );
    return cnt;
}

/*---------------------------------------------------------------------------*
  Name:         MEMCalcHeapSizeForUnitHeap

  Description:  メモリブロックのサイズと個数から必要なヒープのサイズを取得します。

  Arguments:    memBlockSize:  メモリブロックのサイズ(バイト値)。
                memBlockNum:   確保するメモリブロックの総数。
                alignment:     メモリブロックのアライメント。

  Returns:      必要なヒープのサイズを返します。
 *---------------------------------------------------------------------------*/
u32
MEMCalcHeapSizeForUnitHeap(
    u32     memBlockSize,
    u32     memBlockNum,
    int     alignment
)
{
    return  sizeof(MEMiHeapHead) + sizeof(MEMiUntHeapHead)              // ヒープが内部で使用するサイズ
            + (alignment - 4)                                   // アライメントの調整に必要なサイズの最大
            + memBlockNum * RoundUp( memBlockSize, alignment ); // 全ユニットが必要とするサイズ
}

} // namespace winext
} // namespace internal
} // namespace nw
