﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/mem.h>
#include "heapCommoni.h"

namespace nw {
namespace internal {
namespace winext {

/* ========================================================================
    static変数
   ======================================================================== */

/* ------------------------------------------------------------------------
    リスト関連
   ------------------------------------------------------------------------ */

static MEMList   sRootList;                      // ルートのヒープリスト
static BOOL      sRootListInitialized = FALSE;   // sRootListが初期化されていれば真
// static OSMutex   sRootMutex;                     // マルチヒープ管理時の排他制御 (24B)
static MEMSLock  sRootSLock;                     // spin lock

/* ------------------------------------------------------------------------
    フィル関連
   ------------------------------------------------------------------------ */

#if defined(_DEBUG)

static u32 sFillVals[ MEM_HEAP_FILL_MAX ] =
{
    0xC3C3C3C3, // ヒープ作成時に埋める値
    0xF3F3F3F3, // メモリブロック確保時に埋める値
    0xD3D3D3D3, // メモリブロック解放時に埋める値
};

// #if defined(_DEBUG)
#endif


/* ========================================================================
    static関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    リスト関連
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FindContainHeap_

  Description:  指定されたメモリブロックを含有するヒープをリストから再帰的に
                探し出します。

  Arguments:    pList:     リストへのポインタ
                memBlock:  メモリブロックへのポインタ

  Returns:      指定されたメモリブロックを確保したヒープが見つかれば、
                そのヒープへのポインタを返します。
                見つからなかった時は NULL を返します。
 *---------------------------------------------------------------------------*/
static MEMiHeapHead*
FindContainHeap_(
    MEMList*         pList,
    const void*      memBlock
)
{
    MEMiHeapHead* pHeapHd = NULL;
    while ( NULL != ( pHeapHd = (MEMiHeapHead*)MEMGetNextListObject(pList, pHeapHd) ) )
    {
        if ( GetUIntPtr( pHeapHd->heapStart ) <= GetUIntPtr( memBlock )
             &&  GetUIntPtr( memBlock ) < GetUIntPtr( pHeapHd->heapEnd )
        )
        {
            MEMiHeapHead* pChildHeapHd = FindContainHeap_( &pHeapHd->childList, memBlock );
            if ( pChildHeapHd )
            {
                return pChildHeapHd;
            }
            return pHeapHd;
        }
    }
    return NULL;
}


/*---------------------------------------------------------------------------*
  Name:         FindParentHeap_

  Description:  指定されたヒープの親ヒープを、引数のノード以下から検索します。
  Arguments:    pNodeHeap:
                pTargetHeap:

  Returns:      親ヒープが見つかれば、そのヒープヘッダへのポインタを返します。
                見つからなければ NULLが返ります。
 *---------------------------------------------------------------------------*/
static MEMiHeapHead*
FindParentHeap_(
    MEMiHeapHead*       pNodeHeap,
    const MEMiHeapHead* pTargetHeap
)
{
    MEMList*       pList   = &pNodeHeap->childList;
    MEMiHeapHead*  pHeapHd = NULL;

    while ( NULL != ( pHeapHd = (MEMiHeapHead*)MEMGetNextListObject(pList, pHeapHd) ) )
    {
        if ( pHeapHd == pTargetHeap )
        {
            // 子ヒープのメンバとターゲットが一致したならば親ヒープである
            return pNodeHeap;
        }

        // ターゲットのアドレスが含まれているならば更に子供を検索する。
        if ( GetUIntPtr( pHeapHd->heapStart ) <= GetUIntPtr( pTargetHeap )
             &&  GetUIntPtr( pTargetHeap ) < GetUIntPtr( pHeapHd->heapEnd )
        )
        {
            return FindParentHeap_( pHeapHd, pTargetHeap );
        }
    }
    return NULL;
}

/*---------------------------------------------------------------------------*
  Name:         FindListContainHeap_

  Description:  ヒープを含有する親ヒープを検索し、その親ヒープのリストへの
                ポインタを返します。

  Arguments:    pHeapHd:  検索対象のヒープのヘッダへのポインタ。

  Returns:      指定したヒープを含有する親ヒープが見つかれば、
                親ヒープの子リストへのポインタを返します。
                親ヒープが見つからなければルートリストへのポインタが返ります。
 *---------------------------------------------------------------------------*/
static MEMList*
FindListContainHeap_( const MEMiHeapHead* pHeapHd )
{
    MEMList* pList = &sRootList;

    MEMiHeapHead* pContainHeap = FindContainHeap_( &sRootList, pHeapHd );

    if ( pContainHeap )
    {
        pList = &pContainHeap->childList;
    }

    return pList;
}

/*---------------------------------------------------------------------------*
  Name:         ListContainsHeap_

  Description:  リスト中に指定のヒープが存在するかどうかを確認します。

  Arguments:    list    ヒープのリスト構造へのポインタ
                heap    検索するヒープへのポインタ

  Returns:      リストが指定したヒープを含有する場合にはTRUE
                リストが指定したヒープを含まない場合にはFALSE
 *---------------------------------------------------------------------------*/
static BOOL
ListContainsHeap_( MEMList* list, const MEMiHeapHead* heap )
{
    MEMiHeapHead* pHeapHd = NULL;

    while ( NULL != ( pHeapHd = (MEMiHeapHead*)MEMGetNextListObject(list, pHeapHd) ) )
    {
        if ( pHeapHd == heap )
        {
            return TRUE;
        }
    }
    return FALSE;
}


/* ========================================================================
    外部関数(非公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMiInitHeapHead

  Description:  ヒープヘッダの初期化を行います。

  Arguments:    pHeapHd:    ヒープヘッダへのポインタ。
                signature:  シグネチャ。
                heapStart:  ヒープメモリの開始アドレス。
                heapEnd:    ヒープメモリの終了アドレス +1。
                optFlag:    ヒープオプション。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMiInitHeapHead(
    MEMiHeapHead*   pHeapHd,
    u32             signature,
    void*           heapStart,
    void*           heapEnd,
    u16             optFlag
)
{
    pHeapHd->signature = signature;

    pHeapHd->heapStart = heapStart;
    pHeapHd->heapEnd   = heapEnd;

    pHeapHd->attribute.val = 0;

    SetOptForHeap( pHeapHd, optFlag );

    FillNoUseMemory(
            pHeapHd,
            heapStart,
            GetOffsetFromPtr( heapStart, heapEnd )
    );

    MEM_INIT_LIST( &pHeapHd->childList, MEMiHeapHead, link );

    // 初めて作成されるときには、ヒープリストの初期化をおこなう
    if ( ! sRootListInitialized )
    {
        MEM_INIT_LIST( &sRootList, MEMiHeapHead, link );
        // OSInitMutex( &sRootMutex );
        MEMInitSLock( &sRootSLock );
        sRootListInitialized = TRUE;
    }

    // Mutexの初期化
    // OSInitMutex( &pHeapHd->mutex );
    MEMInitSLock( &pHeapHd->slock );

    // マルチヒープのリストに追加
    // OSLockMutex( &sRootMutex );
    MEMLockSLock( &sRootSLock );
    MEMAppendListObject( FindListContainHeap_( pHeapHd ), pHeapHd );
    // OSUnlockMutex( &sRootMutex );
    MEMUnlockSLock( &sRootSLock );
}

/*---------------------------------------------------------------------------*
  Name:         MEMiFinalizeHeap

  Description:  ヒープ共通の後始末を行います。

  Arguments:    pHeapHd:  ヒープヘッダへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMiFinalizeHeap( MEMiHeapHead* pHeapHd )
{
    MEMList* pList;

    // マルチヒープのリストから削除
    // OSLockMutex( &sRootMutex );
    MEMLockSLock( &sRootSLock );

    LockHeap(pHeapHd);              // Make sure no one else is using. Note, no need to unlock later.

    pList = FindListContainHeap_( pHeapHd );
    //ASSERT( ListContainsHeap_( pList, pHeapHd ) );   // ヒープがツリー中に存在するかどうかの確認
    MEMRemoveListObject( pList, pHeapHd );
    // OSUnlockMutex( &sRootMutex );
    MEMUnlockSLock( &sRootSLock );
    pHeapHd->signature = 0;
}


/*---------------------------------------------------------------------------*
  Name:         MEMiDumpHeapHead

  Description:  ヒープヘッダの情報を表示します。

  Arguments:    pHeapHd:  ヒープヘッダへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMiDumpHeapHead( MEMiHeapHead* pHeapHd )
{
    OSReport("[OS Foundation ");

    switch ( pHeapHd->signature )
    {
    case MEMi_EXPHEAP_SIGNATURE: { OSReport("Exp");   break; }
    case MEMi_FRMHEAP_SIGNATURE: { OSReport("Frame"); break; }
    case MEMi_UNTHEAP_SIGNATURE: { OSReport("Unit");  break; }
    //default:
    //    ASSERT( FALSE );
    }

    OSReport(" Heap]\n");

    OSReport("    whole [%p - %p)\n", pHeapHd, pHeapHd->heapEnd);
}



/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMFindContainHeap

  Description:  メモリブロックを含有するヒープを検索します。

  Arguments:    memBlock:  検索対象のメモリブロック。

  Returns:      指定したメモリブロックを含むヒープが見つかれば、
                そのヒープのハンドルを返します。
                見つからなければ、HEAP_INVALID_HANDLE が返ります。
 *---------------------------------------------------------------------------*/
MEMHeapHandle
MEMFindContainHeap( const void* memBlock )
{
    MEMHeapHandle retValue;

    MEMLockSLock(&sRootSLock);
    retValue = FindContainHeap_(&sRootList, memBlock);
    MEMUnlockSLock(&sRootSLock);
    return retValue;
}


/*---------------------------------------------------------------------------*
  Name:         MEMFindParentHeap

  Description:  親ヒープを検索します。

  Arguments:    memBlock:  検索対象のメモリブロック。

  Returns:      指定したメモリブロックを含むヒープが見つかれば、
                そのヒープのハンドルを返します。
                見つからなければ、HEAP_INVALID_HANDLE が返ります。
 *---------------------------------------------------------------------------*/
MEMHeapHandle
MEMFindParentHeap( MEMHeapHandle heap )
{
    MEMiHeapHead* pHeapHd  = NULL;
    MEMHeapHandle retValue = MEM_HEAP_INVALID_HANDLE;

    MEMLockSLock(&sRootSLock);
    while ( NULL != ( pHeapHd = (MEMiHeapHead*)MEMGetNextListObject(&sRootList, pHeapHd) ) )
    {
        if ( pHeapHd == heap )
        {
            break;
        }
        // ターゲットのアドレスが含まれているならば子を検索する。
        if ( GetUIntPtr( pHeapHd->heapStart ) <= GetUIntPtr( heap )
             &&  GetUIntPtr( heap ) < GetUIntPtr( pHeapHd->heapEnd )
        )
        {
            retValue = FindParentHeap_( pHeapHd, heap );
            break;
        }
    }
    MEMUnlockSLock(&sRootSLock);
    return retValue;
}


#define _MEMSLockValue(coreid)       (coreid+1)

/*---------------------------------------------------------------------------*
  Name:         MEMInitSLock

  Description:  Initial spin lock as not taken.

  Arguments:    slock:  pointer to a MEMSLock structure.

  Returns:      none
 *---------------------------------------------------------------------------*/
void MEMInitSLock(MEMSLock *slock) {
    // put lock on own cache block
    slock->pLock  = (u32*) (((u32)&slock->space[0] + CACHE_BLOCK_SIZE-1) & ~(CACHE_BLOCK_SIZE-1));
    *slock->pLock = 0;
    slock->count  = 0;

    // Check lock addr is aligned. Lock should be on its own cache block due to Espresso workaround.
    //ASSERT(0 == ((u32)slock->pLock & ((CACHE_BLOCK_SIZE - 1))));
}


/*---------------------------------------------------------------------------*
  Name:         MEMLockSLock

  Description:  Lock the spin lock.  After the lock is obtained, the thread
                will run with highest priority until the lock is unlock.

                The spin lock is not reentrant (cannot obtain the lock more than
                once).

  Arguments:    slock:  pointer to a MEMSLock structure.

  Returns:      none
 *---------------------------------------------------------------------------*/
void MEMLockSLock(MEMSLock *slock) {}


/*---------------------------------------------------------------------------*
  Name:         MEMUnlockSLock

  Description:  Unlock the spin lock.  After the lock is obtained, if main core,
                the thread will run with highest priority until the lock is unlocked.

  Arguments:    slock:  pointer to a MEMSLock structure.

  Returns:      none
 *---------------------------------------------------------------------------*/
void MEMUnlockSLock(MEMSLock *slock) {}




#if defined(_DEBUG)

/*---------------------------------------------------------------------------*
  Name:         MEMDumpHeap

  Description:  ヒープ内部の情報を表示します。
                これはデバッグ用の関数です。

  Arguments:    heap:    フレームヒープのハンドル。

  Returns:      なし。
 *---------------------------------------------------------------------------*/

void
MEMDumpHeap( MEMHeapHandle heap )
{
    MEMiHeapHead* pHeapHd = heap;

    LockHeap( heap );
    switch ( pHeapHd->signature ) {
    case MEMi_EXPHEAP_SIGNATURE:
        {
            MEMiDumpExpHeap(heap);
        }
        break;
    case MEMi_FRMHEAP_SIGNATURE:
        {
            MEMiDumpFrmHeap(heap);
        }
        break;
    case MEMi_UNTHEAP_SIGNATURE:
        {
            MEMiDumpUnitHeap(heap);
        }
        break;
    default:
        OSReport("[OS Foundation] dump heap : unknown heap. - %p\n", heap);
    }
    UnlockHeap( heap );
}


/*---------------------------------------------------------------------------*
  Name:         MEMSetFillValForHeap

  Description:  ヒープの作成時やメモリブロックの確保・解放時にメモリに
                セットする値をセットします。
                この関数はデバッグ用の関数です。
                最終ROM版ライブラリでは常に0を返します。

  Arguments:    type:  取得する値の種類
                val:   セットする値

  Returns:      以前の、メモリブロックの確保時にメモリにセットする値を返します。
 *---------------------------------------------------------------------------*/
u32
MEMSetFillValForHeap(
    int     type,
    u32     val
)
{
    u32 oldVal;

    //ASSERT( type < MEM_HEAP_FILL_MAX );

    MEMLockSLock(&sRootSLock);

    oldVal = sFillVals[type];
    sFillVals[type] = val;

    MEMUnlockSLock(&sRootSLock);

    return oldVal;
}


/*---------------------------------------------------------------------------*
  Name:         MEMGetFillValForHeap

  Description:  ヒープの作成時やメモリブロックの確保・解放時にメモリに
                セットする値を取得します。
                この関数はデバッグ用の関数です。
                最終ROM版ライブラリでは常に0を返します。

  Arguments:    type:  取得する値の種類

  Returns:      指定された種類のメモリにセットする値を返します。
 *---------------------------------------------------------------------------*/
u32
MEMGetFillValForHeap( int type )
{
    u32 value;

    //ASSERT( type < MEM_HEAP_FILL_MAX );

    MEMLockSLock(&sRootSLock);
    value = sFillVals[ type ];
    MEMUnlockSLock(&sRootSLock);

    return value;
}

// #if defined(_DEBUG)
#endif

} // namespace winext
} // namespace internal
} // namespace nw
