﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/mem/frameHeap.h>
#include "heapCommoni.h"
#include <stdlib.h>

namespace nw {
namespace internal {
namespace winext {

/* ========================================================================
    マクロ定数
   ======================================================================== */

// アライメントの最小値
#define MIN_ALIGNMENT           4


/* ========================================================================
    static関数
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         IsValidFrmHeapHandle_

  Description:  有効なフレームヒープハンドルであるかどうかシグニチャをチェックします。

  Arguments:    handle:  ヒープハンドル

  Returns:      TRUE    有効なヒープハンドルである場合
                FALSE   有効なヒープハンドルではない場合
 *---------------------------------------------------------------------------*/
static inline BOOL
IsValidFrmHeapHandle_( MEMHeapHandle handle )
{
    if ( handle == MEM_HEAP_INVALID_HANDLE )
    {
        return FALSE;
    }

    {
        MEMiHeapHead* pHeapHd = handle;
        return pHeapHd->signature == MEMi_FRMHEAP_SIGNATURE;
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetFrmHeapHeadPtrFromHeapHead_

  Description:  ヒープヘッダへのポインタから、フレームヒープヘッダへのポインタを取得します。

  Arguments:    pHHead:  ヒープヘッダへのポインタ。

  Returns:      フレームヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiFrmHeapHead*
GetFrmHeapHeadPtrFromHeapHead_( MEMiHeapHead* pHHead )
{
    return (MEMiFrmHeapHead*)AddU32ToPtr( pHHead, sizeof(MEMiHeapHead) );
}

/*---------------------------------------------------------------------------*
  Name:         GetHeapHeadPtrFromFrmHeapHead_

  Description:  フレームヒープヘッダへのポインタから、ヒープヘッダへのポインタを取得します。

  Arguments:    pFrmHeapHd:  フレームヒープヘッダへのポインタ。

  Returns:      ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiHeapHead*
GetHeapHeadPtrFromFrmHeapHead_( MEMiFrmHeapHead* pFrmHeapHd )
{
    return (MEMiHeapHead*)SubU32ToPtr(pFrmHeapHd, sizeof(MEMiHeapHead));
}

/*---------------------------------------------------------------------------*
  Name:         InitFrameHeap_

  Description:  フレームヒープの初期化を行います。

  Arguments:    startAddress:  フレームヒープとするメモリの開始アドレス。
                endAddress:    フレームヒープとするメモリの終了アドレス +1。
                optFlag:       オプションフラグ。

  Returns:      ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static MEMiHeapHead*
InitFrameHeap_(
    void*   startAddress,
    void*   endAddress,
    u16     optFlag
)
{
    MEMiHeapHead* pHeapHd = (MEMiHeapHead*)startAddress;
    MEMiFrmHeapHead* pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_( pHeapHd );

    MEMiInitHeapHead(       // ヒープ共通初期化
        pHeapHd,
        MEMi_FRMHEAP_SIGNATURE,
        AddU32ToPtr( pFrmHeapHd, sizeof(MEMiFrmHeapHead) ),    // heapStart
        endAddress,                                        // heapEnd
        optFlag );

    pFrmHeapHd->headAllocator = pHeapHd->heapStart;
    pFrmHeapHd->tailAllocator = pHeapHd->heapEnd;

    pFrmHeapHd->pState = NULL;   // 状態保存ステート位置

    return pHeapHd;
}


/*---------------------------------------------------------------------------*
  Name:         AllocFromHead_

  Description:  ヒープの先頭からメモリブロックを確保します。
                アラインメントの指定があります。

  Arguments:    pHHead:  ヒープヘッダへのポインタ。
                size:    確保するメモリブロックのサイズ。
                alignment:  アライメント値。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
static void*
AllocFromHead_(
    MEMiFrmHeapHead*    pFrmHeapHd,
    u32                 size,
    int                 alignment
)
{
    void* newBlock = RoundUpPtr(pFrmHeapHd->headAllocator, alignment);
    void* endAddress = AddU32ToPtr(newBlock, size);

    if ( GetUIntPtr(endAddress) > GetUIntPtr(pFrmHeapHd->tailAllocator) )
    {
        return NULL;
    }

    FillAllocMemory(  // メモリ充填
        GetHeapHeadPtrFromFrmHeapHead_(pFrmHeapHd),
        pFrmHeapHd->headAllocator,
        GetOffsetFromPtr(pFrmHeapHd->headAllocator, endAddress));

    pFrmHeapHd->headAllocator = endAddress;

    return newBlock;
}

/*---------------------------------------------------------------------------*
  Name:         AllocFromTail_

  Description:  ヒープの末尾からメモリブロックを確保します。
                アラインメントの指定があります。

  Arguments:    pHHead:     ヒープヘッダへのポインタ。
                size:       確保するメモリブロックのサイズ。
                alignment:  アライメント値。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
static void*
AllocFromTail_(
    MEMiFrmHeapHead*    pFrmHeapHd,
    u32                 size,
    int                 alignment
)
{
    void* newBlock = RoundDownPtr(SubU32ToPtr(pFrmHeapHd->tailAllocator, size), alignment);

    if ( GetUIntPtr(newBlock) < GetUIntPtr(pFrmHeapHd->headAllocator) )
    {
        return NULL;
    }

    FillAllocMemory(  // メモリ充填
        GetHeapHeadPtrFromFrmHeapHead_(pFrmHeapHd),
        newBlock,
        GetOffsetFromPtr(newBlock, pFrmHeapHd->tailAllocator) );

    pFrmHeapHd->tailAllocator = newBlock;

    return newBlock;
}

/*---------------------------------------------------------------------------*
  Name:         FreeHead_

  Description:  ヒープ領域の先頭から確保したメモリブロックを一括して開放します。

  Arguments:    pHeapHd:  ヒープのヘッダへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static void
FreeHead_( MEMiHeapHead* pHeapHd )
{
    MEMiFrmHeapHead* pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_(pHeapHd);

    FillFreeMemory(
        pHeapHd,
        pHeapHd->heapStart,
        GetOffsetFromPtr(pHeapHd->heapStart, pFrmHeapHd->headAllocator) );

    pFrmHeapHd->headAllocator = pHeapHd->heapStart;
    pFrmHeapHd->pState        = NULL;
}


/*---------------------------------------------------------------------------*
  Name:         FreeTail_

  Description:  ヒープから確保した全てのメモリブロックを一括して開放します。

  Arguments:    pHeapHd:  ヒープのヘッダへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static void
FreeTail_( MEMiHeapHead* pHeapHd )
{
    MEMiFrmHeapHead* pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_(pHeapHd);

    FillFreeMemory(
        pHeapHd,
        pFrmHeapHd->tailAllocator,
        GetOffsetFromPtr(pFrmHeapHd->tailAllocator, pHeapHd->heapEnd) );

    /*
        ヒープの割り当て状態の復帰により解放したメモリブロックが復活してしまわ
        ないように、保存情報の後尾割り当てポインタを再設定しておく。
     */
    {
        MEMiFrmHeapState* pState;
        for ( pState = pFrmHeapHd->pState; pState; pState = pState->pPrevState )
        {
            pState->tailAllocator = pHeapHd->heapEnd;
        }
    }

    pFrmHeapHd->tailAllocator = pHeapHd->heapEnd;
}


/*---------------------------------------------------------------------------*
  Name:         PrintSize_

  Description:  サイズとパーセンテージを出力します。

  Arguments:    size:       対象となるサイズ。
                wholeSize:  全体のサイズ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
#if defined(_DEBUG)

static void
PrintSize_(
    u32     size,
    u32     wholeSize
)
{
    OSReport("%9d (%6.2f%%)", size, 100.0 * size / wholeSize);
}

// #if defined(_DEBUG)
#endif


/* ========================================================================
    外部関数(非公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMiGetFreeStartForFrmHeap

  Description:  フレームヒープのフリーエリアの先頭アドレスを取得します。

  Arguments:    heap: フレームヒープのハンドル。

  Returns:      フレームヒープのフリーエリアの先頭アドレスを返します。
 *---------------------------------------------------------------------------*/
void*
MEMiGetFreeStartForFrmHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    return GetFrmHeapHeadPtrFromHeapHead_(heap)->headAllocator;
}

/*---------------------------------------------------------------------------*
  Name:         MEMiGetFreeEndForFrmHeap

  Description:  フレームヒープのフリーエリアの末尾アドレスを取得します。

  Arguments:    heap: フレームヒープのハンドル。

  Returns:      フレームヒープのフリーエリアの末尾アドレス +1 を返します。
 *---------------------------------------------------------------------------*/
void*
MEMiGetFreeEndForFrmHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    return GetFrmHeapHeadPtrFromHeapHead_(heap)->tailAllocator;
}


/*---------------------------------------------------------------------------*
  Name:         MEMiDumpFrmHeap

  Description:  フレームヒープ内部の情報を表示します。
                これはデバッグ用の関数です。

  Arguments:    heap:    フレームヒープのハンドル。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
#if defined(_DEBUG)

void
MEMiDumpFrmHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    {
        MEMiHeapHead *const    pHeapHd    = heap;
        MEMiFrmHeapHead *const pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_( pHeapHd );
        const u32              heapSize   = GetOffsetFromPtr( pHeapHd->heapStart, pHeapHd->heapEnd );

        MEMiDumpHeapHead(pHeapHd);

        OSReport(  "     head [%p - %p) ", pHeapHd->heapStart, pFrmHeapHd->headAllocator);
        PrintSize_(GetOffsetFromPtr(pHeapHd->heapStart, pFrmHeapHd->headAllocator), heapSize);
        OSReport("\n     free                           ");
        PrintSize_(GetOffsetFromPtr(pFrmHeapHd->headAllocator, pFrmHeapHd->tailAllocator), heapSize);
        OSReport("\n     tail [%p - %p) ", pFrmHeapHd->tailAllocator, pHeapHd->heapEnd);
        PrintSize_(GetOffsetFromPtr(pFrmHeapHd->tailAllocator, pHeapHd->heapEnd), heapSize);
        OSReport("\n");

        if ( pFrmHeapHd->pState )
        {
            MEMiFrmHeapState* pState;

            OSReport("    state : [tag]   [head]      [tail]\n");

            for ( pState = pFrmHeapHd->pState; pState; pState = pState->pPrevState )
            {
                OSReport("            '%c%c%c%c' : %p %p\n", pState->tagName >>24, (pState->tagName >>16) & 0xFF, (pState->tagName >>8) & 0xFF, pState->tagName & 0xFF,
                                        pState->headAllocator, pState->tailAllocator);
            }
        }

        OSReport("\n");
    }
}

// #if defined(_DEBUG)
#endif



/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMCreateFrmHeapEx

  Description:  フレームヒープを作成します。

  Arguments:    startAddress: ヒープ領域の先頭アドレス。
                size:         ヒープ領域のサイズ。
                optFlag:      オプションフラグ。
                              MEM_HEAP_OPT_0_CLEAR    - メモリ確保時の0クリアフラグ
                              MEM_HEAP_OPT_DEBUG_FILL - デバッグフィルフラグ

  Returns:      関数が成功した場合、作成されたフレームヒープのハンドルが返ります。
                関数が失敗すると、MEM_INVALID_HEAP_HANDLE が返ります。

  Memo:         基本はスレッドセーフではない。
                スレッドセーフにする場合、ヒープの属性を指定する引数を追加するようにするか、
                あるいは、属性をセットする関数で制御してもらうか。
 *---------------------------------------------------------------------------*/
MEMHeapHandle
MEMCreateFrmHeapEx(
    void*   startAddress,
    u32     size,
    u16     optFlag
)
{
    void* endAddress;

    //ASSERT(startAddress != NULL);

    endAddress   = RoundDownPtr(AddU32ToPtr(startAddress, size), MIN_ALIGNMENT);
    startAddress = RoundUpPtr(startAddress, MIN_ALIGNMENT);

    if ( GetUIntPtr(startAddress) > GetUIntPtr(endAddress)
         ||  GetOffsetFromPtr(startAddress, endAddress) < sizeof(MEMiHeapHead) + sizeof(MEMiFrmHeapHead)
    )
    {
        return MEM_HEAP_INVALID_HANDLE;
    }

    {   // Frame ヒープ向け初期化
        MEMiHeapHead* pHHead = InitFrameHeap_( startAddress, endAddress, optFlag );
        return pHHead;  // ヒープヘッダへのポインタがそのままハンドル値とする
    }
}


/*---------------------------------------------------------------------------*
  Name:         MEMDestroyFrmHeap

  Description:  フレームヒープを破棄します。

  Arguments:    heap: フレームヒープのハンドル。

  Returns:      破棄したヒープが占めていた領域へのポインタを返します。
 *---------------------------------------------------------------------------*/
void*
MEMDestroyFrmHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    MEMiFinalizeHeap(heap);
    return (void*)heap;
}

/*---------------------------------------------------------------------------*
  Name:         MEMAllocFromFrmHeapEx

  Description:  フレームヒープからメモリブロックを確保します。
                メモリブロックのアライメントを指定できます。
                アライメント値を負の値で指定すると、ヒープの空き領域を後方から探します。

  Arguments:    heap:      フレームヒープのハンドル。
                size:      確保するメモリブロックのサイズ(バイト単位)。
                alignment: 確保するメモリブロックのアライメント。
                           ±4, ±8, ±16, ±32, ±64, ±128, ... の値が指定できます。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
void*
MEMAllocFromFrmHeapEx(
    MEMHeapHandle     heap,
    u32               size,
    int               alignment
)
{
    void* memory = NULL;
    MEMiFrmHeapHead* pFrmHeapHd;

    //ASSERT(IsValidFrmHeapHandle_(heap));

    // alignment のチェック
    //ASSERT(alignment % MIN_ALIGNMENT == 0);
    //ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    //ASSERT(MIN_ALIGNMENT <= abs(alignment));

    pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_(heap);

    if ( size == 0 )
    {
        size = 1;
    }

    size = RoundUp(size, MIN_ALIGNMENT);

    LockHeap( heap );

    if ( alignment >= 0 )   // ヒープ前から確保
    {
        memory = AllocFromHead_(pFrmHeapHd, size, alignment);
    }
    else                    // ヒープ後ろから確保
    {
        memory = AllocFromTail_(pFrmHeapHd, size, -alignment);
    }

    UnlockHeap( heap );

    return memory;
}

/*---------------------------------------------------------------------------*
  Name:         MEMFreeToFrmHeap

  Description:  フレームヒープへメモリブロックを返却します。

  Arguments:    heap: フレームヒープのハンドル。
                mode: メモリブロックの返却方法。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMFreeToFrmHeap(
    MEMHeapHandle    heap,
    int              mode
)
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    LockHeap( heap );

    if (mode & MEM_FRMHEAP_FREE_HEAD)
    {
        FreeHead_(heap);
    }

    if (mode & MEM_FRMHEAP_FREE_TAIL)
    {
        FreeTail_(heap);
    }

    UnlockHeap( heap );
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetAllocatableSizeForFrmHeapEx

  Description:  フレームヒープ内の割り当て可能な最大サイズを取得します。
                メモリブロックのアライメントを指定できます。

  Arguments:    heap:      フレームヒープのハンドル。
                alignment: 確保するメモリブロックのアライメント。
                           ±4, ±8, ±16, ±32, ±64, ±128, ... の値が指定できます。

  Returns:      フレームヒープ内の割り当て可能な最大サイズを返します(バイト単位)。
 *---------------------------------------------------------------------------*/
u32
MEMGetAllocatableSizeForFrmHeapEx(
    MEMHeapHandle    heap,
    int              alignment
)
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    // alignment のチェック
    //ASSERT(alignment % MIN_ALIGNMENT == 0);
    //ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    //ASSERT(MIN_ALIGNMENT <= abs(alignment));

    alignment = abs(alignment); // 念のため正数化

    LockHeap(heap);
    {
        u32 retVal;
        const MEMiFrmHeapHead* pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_(heap);
        const void*            block      = RoundUpPtr(pFrmHeapHd->headAllocator, alignment);

        if ( GetUIntPtr(block) > GetUIntPtr(pFrmHeapHd->tailAllocator) )
        {
            retVal = 0;
        }
        else
        {
            retVal = GetOffsetFromPtr( block, pFrmHeapHd->tailAllocator );
        }
        UnlockHeap(heap);
        return retVal;
    }
}


/*---------------------------------------------------------------------------*
  Name:         MEMRecordStateForFrmHeap

  Description:  フレームヒープの現在のメモリ使用状態を記録します。
                後で記録したメモリ使用状況に戻すことができます。
                状態を記録するのに20バイト使用します。

  Arguments:    heap:     フレームヒープのハンドル。
                tagName:  状態記録に付けるタグ名。

  Returns:      関数が成功した場合、TRUEが返ります。
                失敗したら、FALSEが返ります。
 *---------------------------------------------------------------------------*/
BOOL
MEMRecordStateForFrmHeap(
    MEMHeapHandle    heap,
    u32              tagName
)
{
    BOOL retVal;

    //ASSERT(IsValidFrmHeapHandle_(heap));

    LockHeap( heap );
    {
        MEMiFrmHeapHead* pFrmHeapHd  = GetFrmHeapHeadPtrFromHeapHead_(heap);
        void* oldHeadAllocator       = pFrmHeapHd->headAllocator;

        // 情報保存領域確保
        MEMiFrmHeapState* pState = (MEMiFrmHeapState*)AllocFromHead_(pFrmHeapHd, sizeof(MEMiFrmHeapState), MIN_ALIGNMENT);

        if ( ! pState )
        {
            retVal = FALSE;
            goto ret_;
        }

        // 現在の状態を格納
        pState->tagName       = tagName;
        pState->headAllocator = oldHeadAllocator;
        pState->tailAllocator = pFrmHeapHd->tailAllocator;
        pState->pPrevState    = pFrmHeapHd->pState;

        pFrmHeapHd->pState = pState;
        retVal = TRUE;
    }
ret_:
    UnlockHeap( heap );

    return retVal;
}

/*---------------------------------------------------------------------------*
  Name:         MEMFreeByStateToFrmHeap

  Description:  フレームヒープのメモリブロックを記録された状態に従って返却します。
                指定したタグ名を用いてRecordStateForFrmHeap()を呼び出す直前
                のメモリの使用状況に戻ります。
                タグ名に0を指定すると最後にRecordStateForFrmHeap()を呼び出す
                直前の状態になります。

                タグ名を指定して返却した場合、それ以後に呼び出された
                RecordStateForFrmHeap()によって記録された情報は
                無くなります。

  Arguments:    heap:     フレームヒープのハンドル。
                tagName:  状態記録に付けるタグ名。

  Returns:      関数が成功した場合、TRUEが返ります。
                失敗したら、FALSEが返ります。
 *---------------------------------------------------------------------------*/
BOOL
MEMFreeByStateToFrmHeap(
    MEMHeapHandle    heap,
    u32              tagName
)
{
    BOOL retVal;

    //ASSERT(IsValidFrmHeapHandle_(heap));

    LockHeap( heap );
    {
        MEMiFrmHeapHead*  pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_(heap);
        MEMiFrmHeapState* pState = pFrmHeapHd->pState;

        if ( tagName != 0 )   // タグ名の指定あり
        {
            for ( ; pState; pState = pState->pPrevState )
            {
                if ( pState->tagName == tagName )
                {
                    break;
                }
            }
        }

        if ( ! pState )
        {
            retVal = FALSE;
            goto ret_;
        }

        {
            void* oldHeadAllocator = pFrmHeapHd->headAllocator;
            void* oldTailAllocator = pFrmHeapHd->tailAllocator;

            pFrmHeapHd->headAllocator = pState->headAllocator;
            pFrmHeapHd->tailAllocator = pState->tailAllocator;
            pFrmHeapHd->pState = pState->pPrevState;

            FillFreeMemory( heap,
                            pFrmHeapHd->headAllocator,
                            GetOffsetFromPtr(pFrmHeapHd->headAllocator, oldHeadAllocator) );
            FillFreeMemory( heap,
                            oldTailAllocator,
                            GetOffsetFromPtr(oldTailAllocator, pFrmHeapHd->tailAllocator) );
        }

        retVal = TRUE;
    }
ret_:
    UnlockHeap( heap );
    return retVal;
}

/*---------------------------------------------------------------------------*
  Name:         MEMAdjustFrmHeap

  Description:  フレームヒープの空き領域をヒープ領域から開放し、その分ヒープ領域を
                縮小します。
                ヒープメモリの後ろから確保されたメモリブロックが存在していては
                いけません。

  Arguments:    heap:     フレームヒープのハンドル。

  Returns:      関数が成功した場合、縮小後のフレームヒープのサイズを返します
                (バイト単位)。
                失敗した場合、0を返します。
 *---------------------------------------------------------------------------*/
u32
MEMAdjustFrmHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidFrmHeapHandle_(heap));

    LockHeap( heap );
    {
        MEMiHeapHead*    pHeapHd    = heap;
        MEMiFrmHeapHead* pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_( pHeapHd );
        u32              retVal;


        // 後方に確保されたメモリブロックが存在する場合は失敗する
        if ( 0 < GetOffsetFromPtr( pFrmHeapHd->tailAllocator, pHeapHd->heapEnd ) )
        {
            retVal = 0;
            goto ret_;
        }

        pFrmHeapHd->tailAllocator = pHeapHd->heapEnd = pFrmHeapHd->headAllocator;

        retVal = GetOffsetFromPtr( heap, pHeapHd->heapEnd );
ret_:
        UnlockHeap( heap );
        return retVal;
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMResizeForMBlockFrmHeap

  Description:  フレームヒープから確保されたメモリブロックのサイズを変更します。

                サイズを変更するメモリブロックは、ヒープの空き領域の前方から確保された
                末尾のメモリブロックである必要があります。

  Arguments:    heap:      フレームヒープのハンドル。
                memBlock:  サイズを変更するメモリブロックへのポインタ。
                newSize:   新しく割り当てるサイズ(バイト単位)。
                           4未満の値を指定された場合は、4が指定されたものとして処理します。

  Returns:      関数が成功した場合、変更されたメモリブロックのサイズを返します(バイト単位)。
                関数が失敗した場合、0 が返ります。
 *---------------------------------------------------------------------------*/
u32
MEMResizeForMBlockFrmHeap(
    MEMHeapHandle    heap,
    void*            memBlock,
    u32              newSize
)
{
    MEMiHeapHead*    pHeapHd    = NULL;
    MEMiFrmHeapHead* pFrmHeapHd = NULL;

    //ASSERT(IsValidFrmHeapHandle_(heap));
    //ASSERT(memBlock == RoundDownPtr(memBlock, MIN_ALIGNMENT));  // 最低限、最小アライメントの境界にあるかチェック

    LockHeap( heap );

    pHeapHd    = heap;
    pFrmHeapHd = GetFrmHeapHeadPtrFromHeapHead_( pHeapHd );

    //ASSERT( ComparePtr(pHeapHd->heapStart, memBlock) <= 0
    //           &&  ComparePtr(pFrmHeapHd->headAllocator, memBlock) > 0 );   // メモリブロックは前方に存在すること
    //ASSERT( pFrmHeapHd->pState == NULL
    //           ||  ComparePtr(pFrmHeapHd->pState, memBlock) < 0 );          // 状態保存がメモリブロックの後方に無いこと

    /*
        newSizeを0することは認めないようにしている。
        0にしてしまうと、memBlockが指すメモリブロックが存在しなくなるため。
    */
    if ( newSize == 0 )
    {
        newSize = 1;
    }
    newSize = RoundUp( newSize, MIN_ALIGNMENT );

    {
        const u32 oldSize = GetOffsetFromPtr( memBlock, pFrmHeapHd->headAllocator );
        void* endAddress  = AddU32ToPtr( memBlock, newSize );

        if ( newSize == oldSize )  // ブロックサイズ変更なしの場合
        {
            goto ret_;
        }

        if ( newSize > oldSize )  // 拡大するとき
        {
            if ( ComparePtr( endAddress, pFrmHeapHd->tailAllocator ) > 0 )  // サイズが足りない場合
            {
                newSize = 0;
                goto ret_;
            }

            FillAllocMemory( heap, pFrmHeapHd->headAllocator, newSize - oldSize );
        }
        else                      // 縮小するとき
        {
            FillFreeMemory( heap, endAddress, oldSize - newSize );
        }

        pFrmHeapHd->headAllocator = endAddress;
    }
ret_:
    UnlockHeap( heap );
    return newSize;
}




}
}
}
