﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/mem/expHeap.h>
#include "heapCommoni.h"
#include <stdlib.h>

namespace nw {
namespace internal {
namespace winext {

/* ========================================================================
    マクロ定数
   ======================================================================== */

// フリーメモリブロックのシグネチャ
#define MBLOCK_FREE_SIGNATURE   ('FR')

// 使用メモリブロックのシグネチャ
#define MBLOCK_USED_SIGNATURE   ('UD')

// グループIDの最大値
#define MAX_GROUPID             0xff

// グループIDのデフォルト値
#define DEFAULT_GROUPID         0

// アライメントの最小値
#define MIN_ALIGNMENT           4

// デフォルトのメモリ確保モード
#define DEFAULT_ALLOC_MODE      MEM_EXPHEAP_ALLOC_MODE_FIRST

// フリーブロックとして登録する最小サイズ (ヘッダは含まないサイズ)
#define MIN_FREE_BLOCK_SIZE     4
// #define MIN_FREE_BLOCK_SIZE 16

/* ========================================================================
    構造体定義
   ======================================================================== */

typedef struct MemRegion MemRegion;

struct MemRegion
{
    void*       start;
    void*       end;
};

/* ========================================================================
    マクロ関数
   ======================================================================== */

#if defined(_DEBUG)

// ヒープの検査時の警告出力用
#define HEAP_WARNING(exp, ...) \
            (void) ((exp) && (OSReport(__VA_ARGS__), 0))

// #if defined(_DEBUG)
#endif


/* ========================================================================
    static関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    ポインタ操作
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         MaxPtr_

  Description:  ２つのポインタのうちアドレスの大きなものを取得します

  Arguments:    a, b    比較対象となるポインタ

  Returns:      a   aのアドレスの方が大きい場合
                b   bのアドレスの方が大きい場合
 *---------------------------------------------------------------------------*/
static inline void*
MaxPtr_( void* a, void* b )
{
    return GetUIntPtr(a) > GetUIntPtr(b) ? a: b;
}

/*---------------------------------------------------------------------------*
  Name:         IsValidExpHeapHandle_

  Description:  有効な拡張ヒープハンドルであるかどうかシグニチャをチェックします。

  Arguments:    handle:  ヒープハンドル

  Returns:      TRUE    有効なヒープハンドルである場合
                FALSE   有効なヒープハンドルではない場合
 *---------------------------------------------------------------------------*/
static inline BOOL
IsValidExpHeapHandle_( MEMHeapHandle handle )
{
    if ( handle == MEM_HEAP_INVALID_HANDLE )
    {
        return FALSE;
    }

    {
        MEMiHeapHead* pHeapHd = handle;
        return pHeapHd->signature == MEMi_EXPHEAP_SIGNATURE;
    }
}

/*---------------------------------------------------------------------------*
  Name:         GetExpHeapHeadPtrFromHeapHead_

  Description:  ヒープヘッダへのポインタから、拡張ヒープヘッダへのポインタを取得します。

  Arguments:    pHHead:  ヒープヘッダへのポインタ。

  Returns:      拡張ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiExpHeapHead*
GetExpHeapHeadPtrFromHeapHead_( MEMiHeapHead* pHHead )
{
    return (MEMiExpHeapHead*)AddU32ToPtr( pHHead, sizeof(MEMiHeapHead) );
}

/*---------------------------------------------------------------------------*
  Name:         GetHeapHeadPtrFromExpHeapHead_

  Description:  拡張ヒープヘッダへのポインタから、ヒープヘッダへのポインタを取得します。

  Arguments:    pEHHead:  拡張ヒープヘッダへのポインタ。

  Returns:      ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiHeapHead*
GetHeapHeadPtrFromExpHeapHead_( MEMiExpHeapHead* pEHHead )
{
    return (MEMiHeapHead*)SubU32ToPtr( pEHHead, sizeof(MEMiHeapHead) );
}

/*---------------------------------------------------------------------------*
  Name:         GetExpHeapHeadPtrFromHandle_

  Description:  拡張ヒープハンドルから、拡張ヒープヘッダへのポインタを取得します。

  Arguments:    heap:  拡張ヒープハンドル。

  Returns:      拡張ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiExpHeapHead*
GetExpHeapHeadPtrFromHandle_( MEMHeapHandle heap )
{
    return (MEMiExpHeapHead*)GetExpHeapHeadPtrFromHeapHead_( heap );
}

/*---------------------------------------------------------------------------*
  Name:         GetMemPtrForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体へのポインタから、
                メモリブロックへのポインタを取得します。

  Arguments:    pMBlkHd:  MEMiExpHeapMBlockHead構造体へのポインタ。

  Returns:      メモリブロックへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline void*
GetMemPtrForMBlock_( MEMiExpHeapMBlockHead* pMBlkHd )
{
    return AddU32ToPtr( pMBlkHd, sizeof(MEMiExpHeapMBlockHead) );
}

static inline const void*
GetMemCPtrForMBlock_( const MEMiExpHeapMBlockHead* pMBlkHd )
{
    return AddU32ToCPtr( pMBlkHd, sizeof(MEMiExpHeapMBlockHead) );
}

/*---------------------------------------------------------------------------*
  Name:         GetMBlockHeadPtr_

  Description:  メモリブロックへのポインタから、
                MEMiExpHeapMBlockHead構造体へのポインタを取得します。
                メモリブロックへのポインタを取得します。

  Arguments:    memBlock:  メモリブロックへのポインタ。

  Returns:      MEMiExpHeapMBlockHead構造体へのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiExpHeapMBlockHead*
GetMBlockHeadPtr_( void* memBlock )
{
    return (MEMiExpHeapMBlockHead*)SubU32ToPtr(memBlock, sizeof(MEMiExpHeapMBlockHead));
}

static inline const MEMiExpHeapMBlockHead*
GetMBlockHeadCPtr_( const void* memBlock )
{
    return (const MEMiExpHeapMBlockHead*)SubU32ToCPtr(memBlock, sizeof(MEMiExpHeapMBlockHead));
}

/*---------------------------------------------------------------------------*
  Name:         GetMBlockEndAddr_

  Description:  メモリブロックへのポインタから、
                メモリの終端アドレスを取得します

  Arguments:    pMBHead:  MEMiExpHeapMBlockHead構造体へのポインタ。

  Returns:      メモリブロックの終端アドレスを返します。
 *---------------------------------------------------------------------------*/
static inline void*
GetMBlockEndAddr_( MEMiExpHeapMBlockHead* pMBHead )
{
    return AddU32ToPtr( GetMemPtrForMBlock_(pMBHead), pMBHead->blockSize );
}

/* ------------------------------------------------------------------------
    MEMiExpHeapMBlockHead 構造体アクセス関数
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         GetAlignmentForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のアライメント値を取得します。

  Arguments:    pMBlkHd:  MEMiExpHeapMBlockHead構造体へのポインタ。

  Returns:      MEMiExpHeapMBlockHead構造体のアライメント値を返します。
 *---------------------------------------------------------------------------*/
static inline u32
GetAlignmentForMBlock_( const MEMiExpHeapMBlockHead* pMBlkHd )
{
    return pMBlkHd->attribute.fields.alignment;
}

/*---------------------------------------------------------------------------*
  Name:         SetAlignmentForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のアライメント値をセットします。

  Arguments:    pMBlkHd:    MEMiExpHeapMBlockHead構造体へのポインタ。
                alignment:  セットするアライメント値

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
SetAlignmentForMBlock_(
    MEMiExpHeapMBlockHead*    pMBlkHd,
    u16                       alignment
)
{
    pMBlkHd->attribute.fields.alignment = alignment;
}


/*---------------------------------------------------------------------------*
  Name:         GetGroupIDForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のグループIDを取得します

  Arguments:    pMBlkHd:    MEMiExpHeapMBlockHead構造体へのポインタ。

  Returns:      グループIDの値を返します。
 *---------------------------------------------------------------------------*/
static inline u16
GetGroupIDForMBlock_( const MEMiExpHeapMBlockHead* pMBHead )
{
    return pMBHead->attribute.fields.groupID;
}

/*---------------------------------------------------------------------------*
  Name:         SetGroupIDForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のグループIDの値をセットします。

  Arguments:    pMBlkHd:    MEMiExpHeapMBlockHead構造体へのポインタ。
                id:         セットするグループID

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
SetGroupIDForMBlock_(
    MEMiExpHeapMBlockHead*    pMBHead,
    u16                       id
)
{
    pMBHead->attribute.fields.groupID = (u8)id;
}


/*---------------------------------------------------------------------------*
  Name:         GetAllocDirForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のメモリアロケートの方向を取得します

  Arguments:    pMBlkHd:    MEMiExpHeapMBlockHead構造体へのポインタ。

  Returns:      メモリアロケートの方向を返します。
 *---------------------------------------------------------------------------*/
static inline u16
GetAllocDirForMBlock_( const MEMiExpHeapMBlockHead* pMBHead )
{
    return pMBHead->attribute.fields.allocDir;
}


/*---------------------------------------------------------------------------*
  Name:         SetAllocDirForMBlock_

  Description:  MEMiExpHeapMBlockHead構造体のメモリアロケートの方向をセットします。

  Arguments:    pMBlkHd:    MEMiExpHeapMBlockHead構造体へのポインタ。
                mode:       セットするメモリアロケート方向

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
SetAllocDirForMBlock_(
    MEMiExpHeapMBlockHead*    pMBHead,
    u16                       mode
)
{
    pMBHead->attribute.fields.allocDir = mode;
}


/* ------------------------------------------------------------------------
    MEMiExpHeapHead 構造体アクセス関数
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         GetAllocMode_

  Description:  拡張ヒープのメモリ確保モードを取得します。

  Arguments:    pEHHead:  拡張ヒープヘッダへのポインタ。

  Returns:      拡張ヒープのメモリ確保モードを返します。
 *---------------------------------------------------------------------------*/
static inline u16
GetAllocMode_( MEMiExpHeapHead* pEHHead )
{
    return pEHHead->feature.fields.allocMode;
}

/*---------------------------------------------------------------------------*
  Name:         SetAllocMode_

  Description:  拡張ヒープのメモリ確保モードをセットします。

  Arguments:    pEHHead:  拡張ヒープヘッダへのポインタ。
                mode:     メモリ確保モード。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
SetAllocMode_(
    MEMiExpHeapHead*      pEHHead,
    u16                   mode
)
{
    pEHHead->feature.fields.allocMode = mode;
}


/*---------------------------------------------------------------------------*
  Name:         GetRegionOfMBlock_

  Description:  MEMiExpHeapMBlockHead構造体からリージョン情報を生成します。

  Arguments:    region  生成するリージョン情報へのポインタ
                block   メモリブロックヘッダへのポインタ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static void
GetRegionOfMBlock_(
    MemRegion*                region,
    MEMiExpHeapMBlockHead*    block
)
{
    region->start = SubU32ToPtr( block, GetAlignmentForMBlock_(block) );
    region->end   = GetMBlockEndAddr_(block);
}


/* ------------------------------------------------------------------------
    メモリブロックリスト操作
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         RemoveMBlock_

  Description:  メモリブロックリストから指定したメモリブロックを削除します。

  Arguments:    list    ブロックリストへのポインタ
                block   削除するメモリブロックへのポインタ

  Returns:      一つ前のメモリブロックへのポインタ
 *---------------------------------------------------------------------------*/
static MEMiExpHeapMBlockHead*
RemoveMBlock_(
    MEMiExpMBlockList*       list,
    MEMiExpHeapMBlockHead*   block
)
{
    MEMiExpHeapMBlockHead *const prev = block->pMBHeadPrev;
    MEMiExpHeapMBlockHead *const next = block->pMBHeadNext;

    // 前参照リンク
    if ( prev )
    {
        prev->pMBHeadNext = next;
    }
    else
    {
        list->head = next;
    }

    // 次参照リンク
    if ( next )
    {
        next->pMBHeadPrev = prev;
    }
    else
    {
        list->tail = prev;
    }

    return prev;
}

/*---------------------------------------------------------------------------*
  Name:         InsertMBlock_

  Description:  メモリブロックリストへメモリブロックを挿入します。

  Arguments:    list    メモリブロックリストへのポインタ
                target  挿入するメモリブロックへのポインタ
                prev    メモリブロックの挿入位置となるブロックへのポインタ

  Returns:      targetへのポインタ
 *---------------------------------------------------------------------------*/
static MEMiExpHeapMBlockHead*
InsertMBlock_(
    MEMiExpMBlockList*       list,
    MEMiExpHeapMBlockHead*   target,
    MEMiExpHeapMBlockHead*   prev
)
{
    MEMiExpHeapMBlockHead* next;

    // 前参照リンク
    target->pMBHeadPrev = prev;
    if ( prev )
    {
        next = prev->pMBHeadNext;
        prev->pMBHeadNext = target;
    }
    else
    {
        next = list->head;
        list->head = target;
    }

    // 次参照リンク
    target->pMBHeadNext = next;
    if ( next )
    {
        next->pMBHeadPrev = target;
    }
    else
    {
        list->tail = target;
    }

    return target;
}

/*---------------------------------------------------------------------------*
  Name:         AppendMBlock_

  Description:  メモリブロックをリストの最後に追加します。

  Arguments:    link:   追加するリスト
                block:  追加するメモリブロック

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
AppendMBlock_(
    MEMiExpMBlockList*       list,
    MEMiExpHeapMBlockHead*   block
)
{
    (void)InsertMBlock_( list, block, list->tail );
}

/*---------------------------------------------------------------------------*
  Name:         InitMBlock_

  Description:  メモリブロックの初期化を行います。

  Arguments:    pRegion:    メモリブロックとするリージョンを表す構造体へのポインタ。
                signature:  メモリブロックのシグネチャ。

  Returns:      初期化したメモリブロックへのポインタを返します。
 *---------------------------------------------------------------------------*/
static MEMiExpHeapMBlockHead*
InitMBlock_(
    MemRegion*              pRegion,
    u16                     signature
)
{
    MEMiExpHeapMBlockHead* block = (MEMiExpHeapMBlockHead*)pRegion->start;

    block->signature = signature;   /* シグニチャを埋め込む */
    block->attribute.val = 0;       /* 属性パラメータをクリア */

    // メモリブロック開始位置から次のリージョンまでのサイズをblockSizeとして設定
    block->blockSize = GetOffsetFromPtr( GetMemPtrForMBlock_(block), pRegion->end );

    block->pMBHeadPrev = NULL;
    block->pMBHeadNext = NULL;

    return block;
}

/*---------------------------------------------------------------------------*
  Name:         InitFreeMBlock_

  Description:  メモリブロックをフリーブロック用に初期化を行います。

  Arguments:    pRegion:    メモリブロックとするリージョンを表す構造体へのポインタ。

  Returns:      初期化したメモリブロックへのポインタを返します。
 *---------------------------------------------------------------------------*/
static inline MEMiExpHeapMBlockHead*
InitFreeMBlock_( MemRegion* pRegion )
{
    return InitMBlock_( pRegion, MBLOCK_FREE_SIGNATURE );
}

/*---------------------------------------------------------------------------*
  Name:         InitExpHeap_

  Description:  拡張ヒープの初期化を行います。

  Arguments:    startAddress:  拡張ヒープとするメモリの開始アドレス。
                endAddress:    拡張ヒープとするメモリの終了アドレス +1。
                optFlag:       オプションフラグ。

  Returns:      ヒープヘッダへのポインタを返します。
 *---------------------------------------------------------------------------*/
static MEMiHeapHead*
InitExpHeap_(
    void*   startAddress,
    void*   endAddress,
    u16     optFlag
)
{
    MEMiHeapHead*    pHeapHd    = (MEMiHeapHead*)startAddress;
    MEMiExpHeapHead* pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_( pHeapHd );

    MEMiInitHeapHead(           // ヒープ共通初期化
        pHeapHd,
        MEMi_EXPHEAP_SIGNATURE,
        AddU32ToPtr( pExpHeapHd, sizeof(MEMiExpHeapHead) ),    // heapStart
        endAddress,                                            // heapEnd
        optFlag );

    pExpHeapHd->groupID = DEFAULT_GROUPID;      // グループID
    pExpHeapHd->feature.val = 0;
    SetAllocMode_( pExpHeapHd, DEFAULT_ALLOC_MODE );

    // フリーのブロックを作る
    {
        MEMiExpHeapMBlockHead* pMBHead;
        MemRegion region;

        region.start = pHeapHd->heapStart;
        region.end   = pHeapHd->heapEnd;
        pMBHead = InitFreeMBlock_( &region );

        // ブロックリスト
        pExpHeapHd->mbFreeList.head = pMBHead;
        pExpHeapHd->mbFreeList.tail = pMBHead;
        pExpHeapHd->mbUsedList.head = NULL;
        pExpHeapHd->mbUsedList.tail = NULL;

        return pHeapHd;
    }
}

/*---------------------------------------------------------------------------*
  Name:         AllocUsedBlockFromFreeBlock_

  Description:  フリーブロックの中から新しいメモリブロックを確保します。

  Arguments:    pEHHead:      拡張ヒープヘッダへのポインタ。
                pMBHeadFree:  フリーブロックヘッダへのポインタ。
                mblock:       確保するメモリブロックのアドレス。
                size:         確保するメモリブロックのサイズ。
                direction:    確保方向。

  Returns:      確保したメモリブロックの先頭のポインタを返します。
 *---------------------------------------------------------------------------*/
static void*
AllocUsedBlockFromFreeBlock_(
    MEMiExpHeapHead*          pEHHead,
    MEMiExpHeapMBlockHead*    pMBHeadFree,
    void*                     mblock,
    u32                       size,
    u16                       direction
)
{
    MemRegion freeRgnT;
    MemRegion freeRgnB;
    MEMiExpHeapMBlockHead* pMBHeadFreePrev;

    GetRegionOfMBlock_( &freeRgnT, pMBHeadFree );
    freeRgnB.end   = freeRgnT.end;
    freeRgnB.start = AddU32ToPtr( mblock, size );
    freeRgnT.end   = SubU32ToPtr( mblock, sizeof(MEMiExpHeapMBlockHead) );

    pMBHeadFreePrev = RemoveMBlock_( &pEHHead->mbFreeList, pMBHeadFree );  // 一旦フリーブロックを削除

    // フリーブロック作る余裕が無い場合
    if ( ( GetOffsetFromPtr(freeRgnT.start, freeRgnT.end) < sizeof(MEMiExpHeapMBlockHead) + MIN_FREE_BLOCK_SIZE ) ||
         ( direction == MEM_EXPHEAP_ALLOC_DIR_FRONT && !pEHHead->feature.fields.useMarginOfAlign                ) )
    {
        freeRgnT.end = freeRgnT.start;  // 使用ブロックのアライメント値に含める
    }
    else
    {
        pMBHeadFreePrev = InsertMBlock_( &pEHHead->mbFreeList, InitFreeMBlock_(&freeRgnT), pMBHeadFreePrev );
    }

    // フリーブロック作る余裕が無い場合
    if ( ( GetOffsetFromPtr(freeRgnB.start, freeRgnB.end) < sizeof(MEMiExpHeapMBlockHead) + MIN_FREE_BLOCK_SIZE ) ||
         ( direction == MEM_EXPHEAP_ALLOC_DIR_REAR && !pEHHead->feature.fields.useMarginOfAlign                 ) )
    {
        freeRgnB.start= freeRgnB.end;   // 使用ブロックに含める
    }
    else
    {
        (void)InsertMBlock_( &pEHHead->mbFreeList, InitFreeMBlock_(&freeRgnB), pMBHeadFreePrev );
    }

    // デバグ用メモリ充填
    FillAllocMemory(
            GetHeapHeadPtrFromExpHeapHead_(pEHHead),
            freeRgnT.end,
            GetOffsetFromPtr(freeRgnT.end, freeRgnB.start)
    );

    // 新規ブロック作成
    {
        MEMiExpHeapMBlockHead* pMBHeadNewUsed;
        MemRegion region;

        region.start = SubU32ToPtr( mblock, sizeof(MEMiExpHeapMBlockHead) );
        region.end   = freeRgnB.start;

        pMBHeadNewUsed = InitMBlock_(&region, MBLOCK_USED_SIGNATURE );
        SetAllocDirForMBlock_( pMBHeadNewUsed, direction );
        SetAlignmentForMBlock_( pMBHeadNewUsed, (u16)GetOffsetFromPtr(freeRgnT.end, pMBHeadNewUsed) );
        SetGroupIDForMBlock_( pMBHeadNewUsed, pEHHead->groupID );
        AppendMBlock_( &pEHHead->mbUsedList, pMBHeadNewUsed );
    }

    return mblock;
}

/*---------------------------------------------------------------------------*
  Name:         AllocFromHead_

  Description:  ヒープの先頭からメモリブロックを確保します。

  Arguments:    pHeapHd:    ヒープヘッダへのポインタ。
                size:       確保するメモリブロックのサイズ。
                alignment:  アライメント値。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
static void*
AllocFromHead_(
    MEMiHeapHead*   pHeapHd,
    u32             size,
    int             alignment
)
{
    MEMiExpHeapHead* pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_(pHeapHd);

    // 最初にみつかったものを割り当てるか
    const BOOL bAllocFirst = GetAllocMode_(pExpHeapHd) == MEM_EXPHEAP_ALLOC_MODE_FIRST;

    MEMiExpHeapMBlockHead* pMBlkHd      = NULL;
    MEMiExpHeapMBlockHead* pMBlkHdFound = NULL;
    u32 foundSize = 0xffffffff;
    void* foundMBlock = NULL;

    // フリーブロック検索
    for ( pMBlkHd = pExpHeapHd->mbFreeList.head; pMBlkHd; pMBlkHd = pMBlkHd->pMBHeadNext )
    {
        void *const mblock    = GetMemPtrForMBlock_(pMBlkHd);
        void *const reqMBlock = RoundUpPtr( mblock, alignment );
        const u32 offset      = GetOffsetFromPtr( mblock, reqMBlock );    // 発生したずれ

        if ( pMBlkHd->blockSize >= size + offset &&
             foundSize > pMBlkHd->blockSize      )
        {
            pMBlkHdFound  = pMBlkHd;
            foundSize     = pMBlkHd->blockSize;
            foundMBlock   = reqMBlock;

            if ( bAllocFirst || foundSize == size )
            {
                break;
            }
        }
    }

    if ( ! pMBlkHdFound ) // 条件に合うブロックが見つからない
    {
        return NULL;
    }

    return AllocUsedBlockFromFreeBlock_( // 発見したフリーブロックから領域確保
                    pExpHeapHd,
                    pMBlkHdFound,
                    foundMBlock,
                    size,
                    MEM_EXPHEAP_ALLOC_DIR_FRONT
           );
}

/*---------------------------------------------------------------------------*
  Name:         AllocFromTail_

  Description:  ヒープの末尾からメモリブロックを確保します。

  Arguments:    pHeapHd:    ヒープヘッダへのポインタ。
                size:       確保するメモリブロックのサイズ。
                alignment:  アライメント値。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
static void*
AllocFromTail_(
    MEMiHeapHead*   pHeapHd,
    u32             size,
    int             alignment
)
{
    MEMiExpHeapHead* pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_(pHeapHd);

    // 最初にみつかったものを割り当てるか
    const BOOL bAllocFirst = GetAllocMode_(pExpHeapHd) == MEM_EXPHEAP_ALLOC_MODE_FIRST;

    MEMiExpHeapMBlockHead* pMBlkHd      = NULL;
    MEMiExpHeapMBlockHead* pMBlkHdFound = NULL;
    u32 foundSize = 0xffffffff;
    void* foundMBlock = NULL;

    // フリーブロック検索
    for ( pMBlkHd = pExpHeapHd->mbFreeList.tail; pMBlkHd; pMBlkHd = pMBlkHd->pMBHeadPrev )
    {
        void *const mblock    = GetMemPtrForMBlock_(pMBlkHd);
        void *const mblockEnd = AddU32ToPtr( mblock, pMBlkHd->blockSize );
        void *const reqMBlock = RoundDownPtr( SubU32ToPtr(mblockEnd, size), alignment );  // アライン済みのアドレス

        if ( ComparePtr( reqMBlock, mblock ) >= 0
             &&  foundSize > pMBlkHd->blockSize )
        {
            pMBlkHdFound = pMBlkHd;
            foundSize    = pMBlkHd->blockSize;
            foundMBlock  = reqMBlock;

            if ( bAllocFirst || foundSize == size )
            {
                break;
            }
        }
    }

    if ( ! pMBlkHdFound ) // 条件に合うブロックが見つからない
    {
        return NULL;
    }

    return AllocUsedBlockFromFreeBlock_( // 発見したフリーブロックから領域確保
                    pExpHeapHd,
                    pMBlkHdFound,
                    foundMBlock,
                    size,
                    MEM_EXPHEAP_ALLOC_DIR_REAR
           );
}

/*---------------------------------------------------------------------------*
  Name:         RecycleRegion_

  Description:  空きリージョンをフリーメモリブロックへ組み入れます。
                フリーブロックと隣接している場合は、フリーブロックを拡張します。
                フリーブロックと隣接しておらず、かつフリーブロックとするほどの
                サイズが無い場合は、この空きリージョンのサイズを後方に隣接する
                使用済みブロックのアライメント値として設定します。
                空きリージョンの後方に隣接する使用済みブロックが無い場合は、
                関数は失敗します。

  Arguments:    pEHHead:  拡張ヒープヘッダへのポインタ。
                pRegion:  空きリージョンへのポインタ。

  Returns:      関数が成功すれば TRUE を返します。
                失敗すれば FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
RecycleRegion_(
    MEMiExpHeapHead*    pEHHead,
    const MemRegion*    pRegion
)
{
    MEMiExpHeapMBlockHead* pBlkPrFree  = NULL;   // 直前フリーブロック
    MemRegion freeRgn = *pRegion;

    // 指定エリアに隣接したフリーエリアを検索
    {
        MEMiExpHeapMBlockHead* pBlk;

        for ( pBlk = pEHHead->mbFreeList.head; pBlk; pBlk = pBlk->pMBHeadNext )
        {
            if ( pBlk < pRegion->start )
            {
                pBlkPrFree = pBlk;
                continue;
            }

            if ( pBlk == pRegion->end )   // 後方に隣接するブロックか?
            {
                // 空きリージョンを結合
                freeRgn.end = GetMBlockEndAddr_(pBlk);
                (void)RemoveMBlock_( &pEHHead->mbFreeList, pBlk );

                // ヘッダ部をNoUseで埋める
                FillNoUseMemory( GetHeapHeadPtrFromExpHeapHead_(pEHHead), pBlk, sizeof(MEMiExpHeapMBlockHead) );
            }
            break;
        }
    }

    // 直前のフリーブロックが前方に隣接するブロックか?
    if ( pBlkPrFree && GetMBlockEndAddr_(pBlkPrFree) == pRegion->start )
    {
        // 空きリージョンを結合
        freeRgn.start = pBlkPrFree;
        pBlkPrFree = RemoveMBlock_(&pEHHead->mbFreeList, pBlkPrFree);
    }

    if ( GetOffsetFromPtr(freeRgn.start, freeRgn.end) < sizeof(MEMiExpHeapMBlockHead) ) // ブロックになれない大きさ
    {
        return FALSE;   // ResizeForMBlockExpHeap()で小さいサイズを縮小しようとしていて、
                        // かつ後ろにフリーブロックが無い場合にここに到達
    }

    // デバグ用メモリ充填
    FillFreeMemory( GetHeapHeadPtrFromExpHeapHead_(pEHHead), pRegion->start, GetOffsetFromPtr(pRegion->start, pRegion->end) );

    (void)InsertMBlock_(
                &pEHHead->mbFreeList,
                InitFreeMBlock_(&freeRgn),   // フリーブロック作成
                pBlkPrFree
          );

    return TRUE;
}





/* ========================================================================
    動作チェック用関数
   ======================================================================== */
#if defined(_DEBUG)

/*---------------------------------------------------------------------------*
  Name:         CheckMBlock_

  Description:  メモリブロックのヘッダの内容が妥当であるかチェックします。

  Arguments:    pMBHead:    チェックするメモリブロックのヘッダへのポインタ。
                pHeapHd:    拡張ヒープのヘッダへのポインタ。
                signature:  メモリブロックのシグネチャ。
                heapType:   メモリブロックのタイプ(使用 or フリー)
                flag:       フラグ。

  Returns:      メモリブロックのヘッダの内容が妥当なら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
CheckMBlock_(
    const MEMiExpHeapMBlockHead*    pMBHead,
    MEMiHeapHead*                   pHeapHd,
    u16                             signature,
    const char*                     heapType,
    u32                             flag
)
{
    const BOOL bPrint = ( 0 != (flag & MEM_HEAP_ERROR_PRINT) );
    const void *const memBlock = GetMemCPtrForMBlock_(pMBHead);

    if ( pHeapHd )
    {
        if ( GetUIntPtr(pMBHead) < GetUIntPtr(pHeapHd->heapStart)
             || GetUIntPtr(memBlock) > GetUIntPtr(pHeapHd->heapEnd)
        )
        {
            HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Bad %s memory block address. - address %08X, heap area [%08X - %08X)\n",
                         heapType, memBlock, pHeapHd->heapStart, pHeapHd->heapEnd);
            return FALSE;
        }
    }

    if ( pMBHead->signature != signature )    // シグネチャが異なる?
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Bad %s memory block signature. - address %08X, signature %04X\n",
                     heapType, memBlock, pMBHead->signature);
        return FALSE;
    }

    if ( pMBHead->blockSize >= 0x08000000 )   // サイズが大きすぎる?(128MB以上)
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Too large %s memory block. - address %08X, block size %08X\n",
                     heapType, memBlock, pMBHead->blockSize);
        return FALSE;
    }

    if ( pHeapHd )
    {
        if ( GetUIntPtr(memBlock) + pMBHead->blockSize > GetUIntPtr(pHeapHd->heapEnd) )
        {
            HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " wrong size %s memory block. - address %08X, block size %08X\n",
                         heapType, memBlock, pMBHead->blockSize);
            return FALSE;
        }
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         CheckUsedMBlock_

  Description:  使用メモリブロックのヘッダの内容が妥当であるかチェックします。

  Arguments:    pMBHead:  チェックするメモリブロックのヘッダへのポインタ。
                pHeapHd:  拡張ヒープのヘッダへのポインタ。
                flag:     フラグ。

  Returns:      メモリブロックのヘッダの内容が妥当なら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static inline BOOL
CheckUsedMBlock_(
    const MEMiExpHeapMBlockHead*    pMBHead,
    MEMiHeapHead*                   pHeapHd,
    u32                             flag
)
{
    if ( pHeapHd )
    {
        MEMiExpHeapHead* pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_(pHeapHd);
        MEMiExpHeapMBlockHead* pMBlkHd      = NULL;
        for ( pMBlkHd = pExpHeapHd->mbUsedList.head; pMBlkHd; pMBlkHd = pMBlkHd->pMBHeadNext )
        {
            if ( pMBHead == pMBlkHd )
            {
                break;
            }
        }
        if ( pMBlkHd == NULL )
        {
            return FALSE;
        }
    }
    return CheckMBlock_( pMBHead, pHeapHd, MBLOCK_USED_SIGNATURE, "used", flag );
}

/*---------------------------------------------------------------------------*
  Name:         CheckFreeMBlock_

  Description:  フリーメモリブロックのヘッダの内容が妥当であるかチェックします。

  Arguments:    pMBHead:  チェックするメモリブロックのヘッダへのポインタ。
                pHeapHd:  拡張ヒープのヘッダへのポインタ。
                flag:     フラグ。

  Returns:      メモリブロックのヘッダの内容が妥当なら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static inline BOOL
CheckFreeMBlock_(
    const MEMiExpHeapMBlockHead*    pMBHead,
    MEMiHeapHead*                   pHeapHd,
    u32                             flag
)
{
    return CheckMBlock_( pMBHead, pHeapHd, MBLOCK_FREE_SIGNATURE, "free", flag );
}


/*---------------------------------------------------------------------------*
  Name:         CheckMBlockPrevPtr_

  Description:  メモリブロックの前へのリンクが正しいかチェックします。

  Arguments:    pMBHead:      チェックするメモリブロックのヘッダへのポインタ。
                pMBHeadPrev:  チェックするメモリブロックの前のメモリブロックのヘッダへのポインタ。
                flag:         フラグ。

  Returns:      メモリブロックの前へのリンクが正しいなら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
CheckMBlockPrevPtr_(
    const MEMiExpHeapMBlockHead*    pMBHead,
    const MEMiExpHeapMBlockHead*    pMBHeadPrev,
    u32                             flag
)
{
    const BOOL bPrint = ( 0 != (flag & MEM_HEAP_ERROR_PRINT) );

    if ( pMBHead->pMBHeadPrev != pMBHeadPrev )
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Wrong link memory block. - address %08X, previous address %08X != %08X\n",
                     GetMemCPtrForMBlock_(pMBHead), pMBHead->pMBHeadPrev, pMBHeadPrev);
        return FALSE;
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         CheckMBlockNextPtr_

  Description:  メモリブロックの次へのリンクが正しいかチェックします。

  Arguments:    pMBHead:      チェックするメモリブロックのヘッダへのポインタ。
                pMBHeadNext:  チェックするメモリブロックの次のメモリブロックのヘッダへのポインタ。
                flag:         フラグ。

  Returns:      メモリブロックの次へのリンクが正しいなら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
CheckMBlockNextPtr_(
    const MEMiExpHeapMBlockHead*   pMBHead,
    const MEMiExpHeapMBlockHead*   pMBHeadNext,
    u32                            flag
)
{
    const BOOL bPrint = ( 0 != (flag & MEM_HEAP_ERROR_PRINT) );

    if (pMBHead->pMBHeadNext != pMBHeadNext)
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Wrong link memory block. - address %08X, next address %08X != %08X\n",
                     GetMemCPtrForMBlock_(pMBHead), pMBHead->pMBHeadNext, pMBHeadNext);
        return FALSE;
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         CheckMBlockLinkTail_

  Description:  メモリブロックのポインタがメモリブロックリストの先頭/末尾であるかチェックします。

  Arguments:    pMBHead:      チェックするメモリブロックのヘッダへのポインタ。
                pMBHeadTail:  メモリブロックリストの先頭/末尾メモリブロックのヘッダへのポインタ。
                headType:     先頭か末尾を表す文字列。
                flag:         フラグ。

  Returns:      メモリブロックのポインタがメモリブロックリストの先頭/末尾であるなら TRUE、
                そうでないなら FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
CheckMBlockLinkTail_(
    const MEMiExpHeapMBlockHead*    pMBHead,
    const MEMiExpHeapMBlockHead*    pMBHeadTail,
    const char*                     heapType,
    u32                             flag
)
{
    const BOOL bPrint = 0 != (flag & MEM_HEAP_ERROR_PRINT);

    if ( pMBHead != pMBHeadTail )
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Wrong memory brock list %s pointer. - address %08X, %s address %08X != %08X\n",
                     heapType, GetMemCPtrForMBlock_(pMBHead), heapType, pMBHead, pMBHeadTail);
        return FALSE;
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         IsValidUsedMBlock_

  Description:  使用メモリブロックの妥当性をチェックします。

  Arguments:    memBlock:  チェックするメモリブロック。
                heap:      メモリブロックを含有する拡張ヒープのハンドル。
                           NULLを指定すると、メモリブロックがヒープに含まれているかの
                           チェックは行いません。

  Returns:      指定したメモリブロックが問題なければ TRUE を返します。
                問題があれば FALSE を返します。
 *---------------------------------------------------------------------------*/
static BOOL
IsValidUsedMBlock_(
    const void*       memBlock,
    MEMHeapHandle     heap
)
{
    MEMiHeapHead* pHeapHd = heap;
    BOOL ret;

    if ( ! memBlock)
    {
        return FALSE;
    }

    if ( heap ) { LockHeap( heap );   }

    ret = CheckUsedMBlock_(GetMBlockHeadCPtr_(memBlock), pHeapHd, 0);

    if ( heap ) { UnlockHeap( heap ); }

    return ret;
}

// #if defined(_DEBUG)
#endif



/* ========================================================================
    外部関数(非公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMiDumpExpHeap

  Description:  拡張ヒープ内部の情報を表示します。
                これはデバッグ用の関数です。

  Arguments:    heap:    拡張ヒープのハンドル。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
#if defined(_DEBUG)

void
MEMiDumpExpHeap( MEMHeapHandle heap )
{
    //ASSERT(IsValidExpHeapHandle_(heap));

    {
        u32  usedSize = 0;
        u32  usedCnt = 0;
        u32  freeSize = 0;
        u32  freeCnt = 0;

        MEMiHeapHead* pHeapHd = heap;
        MEMiExpHeapHead* pExpHeapHd = GetExpHeapHeadPtrFromHandle_(pHeapHd);

        MEMiDumpHeapHead(pHeapHd);

        OSReport("     attr  address:   size    gid aln   prev_ptr next_ptr\n");   // ヘッダー行

        // ---------------- UsedBlock のダンプ ----------------
        OSReport("    (Used Blocks)\n" );

        if ( pExpHeapHd->mbUsedList.head == NULL )
        {
            OSReport("     NONE\n");
        }
        else
        {
            MEMiExpHeapMBlockHead* pMBHead;

            for ( pMBHead = pExpHeapHd->mbUsedList.head; pMBHead; pMBHead = pMBHead->pMBHeadNext )
            {
                if ( pMBHead->signature != MBLOCK_USED_SIGNATURE )
                {
                    OSReport("    xxxxx %08x: --------  --- ---  (-------- --------)\nabort\n", pMBHead);
                    break;
                }

                OSReport("    %s %08x: %8d  %3d %3d  (%08x %08x)\n",
                    GetAllocDirForMBlock_(pMBHead) == MEM_EXPHEAP_ALLOC_DIR_REAR ? " rear" : "front",
                    GetMemPtrForMBlock_(pMBHead),
                    pMBHead->blockSize,
                    GetGroupIDForMBlock_( pMBHead ),
                    GetAlignmentForMBlock_( pMBHead ),
                    pMBHead->pMBHeadPrev ? GetMemPtrForMBlock_(pMBHead->pMBHeadPrev): NULL,
                    pMBHead->pMBHeadNext ? GetMemPtrForMBlock_(pMBHead->pMBHeadNext): NULL
                );

                // ---- 使用量
                usedSize += sizeof(MEMiExpHeapMBlockHead) + pMBHead->blockSize + GetAlignmentForMBlock_(pMBHead);

                usedCnt ++;
            }
        }

        // ---------------- FreeBlock のダンプ ----------------
        OSReport("    (Free Blocks)\n" );

        if ( pExpHeapHd->mbFreeList.head == NULL )
        {
            OSReport("     NONE\n" );
        }
        else
        {
            MEMiExpHeapMBlockHead* pMBHead;

            for ( pMBHead = pExpHeapHd->mbFreeList.head; pMBHead; pMBHead = pMBHead->pMBHeadNext )
            {
                if ( pMBHead->signature != MBLOCK_FREE_SIGNATURE )
                {
                    OSReport("    xxxxx %08x: --------  --- ---  (-------- --------)\nabort\n", pMBHead);
                    break;
                }

                OSReport("    %s %08x: %8d  %3d %3d  (%08x %08x)\n",
                    " free",
                    GetMemPtrForMBlock_(pMBHead),
                    pMBHead->blockSize,
                    GetGroupIDForMBlock_( pMBHead ),
                    GetAlignmentForMBlock_( pMBHead ),
                    pMBHead->pMBHeadPrev ? GetMemPtrForMBlock_(pMBHead->pMBHeadPrev): NULL,
                    pMBHead->pMBHeadNext ? GetMemPtrForMBlock_(pMBHead->pMBHeadNext): NULL
                );

                freeCnt ++;
            }
        }

        OSReport("\n");

        {
            u32 heapSize  = GetOffsetFromPtr(pHeapHd->heapStart, pHeapHd->heapEnd); // ヒープサイズ(データ領域のサイズ)
            OSReport("    %d / %d bytes (%6.2f%%) used (U:%d F:%d)\n",
                               usedSize, heapSize, 100.0 * usedSize / heapSize, usedCnt, freeCnt);
        }

        OSReport("\n");
    }
}

// #if defined(_DEBUG)
#endif



/* ========================================================================
    外部関数(公開)
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMCreateExpHeapEx

  Description:  拡張ヒープを作成します。

  Arguments:    startAddress: ヒープ領域の先頭アドレス。
                size:         ヒープ領域のサイズ。
                optFlag:      オプションフラグ。

  Returns:      関数が成功した場合、作成された拡張ヒープのハンドルが返ります。
                関数が失敗すると、MEM_HEAP_INVALID_HANDLE が返ります。
 *---------------------------------------------------------------------------*/
MEMHeapHandle
MEMCreateExpHeapEx(
    void*   startAddress,
    u32     size,
    u16     optFlag
)
{
    void* endAddress;

    //ASSERT(startAddress != NULL);

    endAddress   = RoundDownPtr(AddU32ToPtr(startAddress, size), MIN_ALIGNMENT);
    startAddress = RoundUpPtr(startAddress, MIN_ALIGNMENT);

    if ( GetUIntPtr(startAddress) > GetUIntPtr(endAddress)
         ||  GetOffsetFromPtr(startAddress, endAddress) <
                    sizeof(MEMiHeapHead) + sizeof(MEMiExpHeapHead) + sizeof(MEMiExpHeapMBlockHead) + MIN_ALIGNMENT
    )
    {
        return MEM_HEAP_INVALID_HANDLE;
    }

    {   // Expヒープ向け初期化
        MEMiHeapHead* pHeapHd = InitExpHeap_( startAddress, endAddress, optFlag );
        return pHeapHd;  // ヒープヘッダへのポインタがそのままハンドル値とする
    }
}

/*---------------------------------------------------------------------------*
  Name:         MEMDestroyExpHeap

  Description:  拡張ヒープを破棄します。

  Arguments:    heap: 拡張ヒープのハンドル。

  Returns:      破棄したヒープが占めていた領域へのポインタを返します。
 *---------------------------------------------------------------------------*/
void*
MEMDestroyExpHeap( MEMHeapHandle heap )
{
    //ASSERT( IsValidExpHeapHandle_(heap) );

    MEMiFinalizeHeap( heap );
    return (void*)heap;
}

/*---------------------------------------------------------------------------*
  Name:         MEMAllocFromExpHeapEx

  Description:  拡張ヒープからメモリブロックを確保します。
                メモリブロックのアライメントを指定できます。
                アライメント値を負の値で指定すると、ヒープの空き領域を後方から探します。

  Arguments:    heap:      拡張ヒープのハンドル。
                size:      確保するメモリブロックのサイズ(バイト単位)。
                alignment: 確保するメモリブロックのアライメント。
                           ±4, ±8, ±16, ±32, ±64, ±128, ..., ±256MB の値が指定できます。

  Returns:      メモリブロックの確保が成功した場合、確保したメモリブロックへの
                ポインタが返ります。
                失敗した場合、NULLが返ります。
 *---------------------------------------------------------------------------*/
void*
MEMAllocFromExpHeapEx(
    MEMHeapHandle   heap,
    u32             size,
    int             alignment
)
{
    void* memory = NULL;

    //ASSERT( IsValidExpHeapHandle_(heap) );

    // alignment のチェック
    //ASSERT(alignment % MIN_ALIGNMENT == 0);
    //ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    //ASSERT(MIN_ALIGNMENT <= abs(alignment));
    //ASSERT( (-256 * 1024 * 1024 <= alignment) && (alignment <= 256 * 1024 * 1024) );

    if ( size == 0 )
    {
        size = 1;
    }

    size = RoundUp( size, MIN_ALIGNMENT );    // 実際に確保するサイズ

    LockHeap( heap );

    if ( alignment >= 0 )     // 前から確保
    {
        memory = AllocFromHead_( heap, size, alignment );
    }
    else                    // 後ろから確保
    {
        memory = AllocFromTail_( heap, size, -alignment );
    }

    UnlockHeap( heap );

    return memory;
}

/*---------------------------------------------------------------------------*
  Name:         MEMResizeForMBlockExpHeap

  Description:  拡張ヒープから確保されたメモリブロックのサイズを変更します。

  Arguments:    heap:     拡張ヒープのハンドル。
                memBlock: サイズを変更するメモリブロックへのポインタ。
                size:     新しく割り当てるサイズ(バイト単位)。

  Returns:      関数が成功した場合、変更されたメモリブロックのサイズを返します(バイト単位)。
                関数が失敗した場合、0 が返ります。
 *---------------------------------------------------------------------------*/
u32
MEMResizeForMBlockExpHeap(
    MEMHeapHandle   heap,
    void*           memBlock,
    u32             size
)
{
    MEMiExpHeapHead       *pEHHead;
    MEMiExpHeapMBlockHead *pMBHead;

    //ASSERT(IsValidExpHeapHandle_(heap));
    //ASSERT(memBlock != NULL);

    LockHeap(heap);

    pEHHead = GetExpHeapHeadPtrFromHandle_(heap);
    pMBHead = GetMBlockHeadPtr_(memBlock);

    size = RoundUp( size, MIN_ALIGNMENT );
    if ( size == pMBHead->blockSize )  // ブロックサイズ変更なしの場合
    {
        return size;
    }


    //ASSERT(IsValidUsedMBlock_(memBlock, heap));

    // 新しいエリアが拡大するとき
    if ( size > pMBHead->blockSize )
    {
        void* crUsedEnd = GetMBlockEndAddr_(pMBHead);   // 現使用ブロックのend address
        MEMiExpHeapMBlockHead* block;

        // 次のフリーブロックを探す
        for ( block = pEHHead->mbFreeList.head; block; block = block->pMBHeadNext )
        {
            if ( block == crUsedEnd )
            {
                break;
            }
        }

        // 次のフリーブロックがないか、サイズが足りない場合
        if ( ! block || size > pMBHead->blockSize + sizeof(MEMiExpHeapMBlockHead) + block->blockSize)
        {
            UnlockHeap( heap );
            return 0;
        }

        {
            MemRegion         rgnNewFree;
            void              *oldFreeStart;
            MEMiExpHeapMBlockHead *nextBlockPrev;

            // フリーブロックのリージョンを取得し、フリーブロックを一旦外す
            GetRegionOfMBlock_( &rgnNewFree, block );
            nextBlockPrev = RemoveMBlock_( &pEHHead->mbFreeList, block );

            oldFreeStart = rgnNewFree.start;
            rgnNewFree.start = AddU32ToPtr( memBlock, size ); // 新しくフリーとなるリージョン位置

            // 余りがメモリブロックサイズ未満なら
            if ( GetOffsetFromPtr(rgnNewFree.start, rgnNewFree.end) < sizeof(MEMiExpHeapMBlockHead) )
            {
                rgnNewFree.start = rgnNewFree.end;  // 使用ブロックに吸収
            }

            pMBHead->blockSize = GetOffsetFromPtr(memBlock, rgnNewFree.start);  // 対象ブロックサイズ変更

            // 余りがメモリブロックサイズ以上なら
            if ( GetOffsetFromPtr(rgnNewFree.start, rgnNewFree.end) >= sizeof(MEMiExpHeapMBlockHead) )
            {
                (void)InsertMBlock_(&pEHHead->mbFreeList, InitFreeMBlock_(&rgnNewFree), nextBlockPrev);   // 新しくフリーブロックを作る
            }

            FillAllocMemory(  // 拡張した部分フィル
                heap,
                oldFreeStart,
                GetOffsetFromPtr(oldFreeStart, rgnNewFree.start));
        }
    }
    // 新しいエリアが縮小するとき
    else
    {
        MemRegion rgnNewFree;
        const u32 oldBlockSize = pMBHead->blockSize;

        rgnNewFree.start = AddU32ToPtr(memBlock, size); // 新しくフリーとなるリージョン位置
        rgnNewFree.end   = GetMBlockEndAddr_(pMBHead);   // 現使用ブロックのend address

        pMBHead->blockSize = size;  // 対象ブロックサイズ変更

        if ( ! RecycleRegion_(pEHHead, &rgnNewFree) )    // フリーリストに返してみる
        {
            pMBHead->blockSize = oldBlockSize;  // 失敗したら、元に戻す
        }
    }

    UnlockHeap( heap );

    return pMBHead->blockSize;
}

/*---------------------------------------------------------------------------*
  Name:         MEMFreeToExpHeap

  Description:  拡張ヒープへメモリブロックを返却します。

  Arguments:    heap:     拡張ヒープのハンドル。
                memBlock: 返却するメモリブロックへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMFreeToExpHeap(
    MEMHeapHandle   heap,
    void*           memBlock
)
{
    //ASSERT(IsValidExpHeapHandle_(heap));
    if ( memBlock == NULL )
    {
        return;
    }

    LockHeap( heap );
    {
        MEMiHeapHead          *pHeapHd    = heap;
        MEMiExpHeapHead       *pExpHeapHd = GetExpHeapHeadPtrFromHandle_( pHeapHd );
        MEMiExpHeapMBlockHead *pMBHead    = GetMBlockHeadPtr_( memBlock );
        MemRegion             region;

        //ASSERT(IsValidUsedMBlock_(memBlock, heap));

        // このヒープの中に入っているか
        //ASSERT( pHeapHd->heapStart <= memBlock && memBlock < pHeapHd->heapEnd );

        GetRegionOfMBlock_( &region, pMBHead );
        (void)RemoveMBlock_( &pExpHeapHd->mbUsedList, pMBHead );   // 使用リストからはずす
        (void)RecycleRegion_( pExpHeapHd, &region );   // 指定アドレスから指定サイズをフリーに加える
    }
    UnlockHeap( heap );
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetTotalFreeSizeForExpHeap

  Description:  拡張ヒープの空き領域のサイズの合計を取得します。

  Arguments:    heap:     拡張ヒープのハンドル。

  Returns:      拡張ヒープの空き領域のサイズの合計を返します(バイト単位)。
 *---------------------------------------------------------------------------*/
u32
MEMGetTotalFreeSizeForExpHeap( MEMHeapHandle heap )
{
    u32     sumSize  = 0;

    //ASSERT(IsValidExpHeapHandle_(heap));

    LockHeap( heap );

    {
        MEMiExpHeapHead       *pEHHead = GetExpHeapHeadPtrFromHandle_(heap);
        MEMiExpHeapMBlockHead *pMBHead;

        for ( pMBHead = pEHHead->mbFreeList.head; pMBHead; pMBHead = pMBHead->pMBHeadNext )
        {
            sumSize += pMBHead->blockSize;
        }
    }

    UnlockHeap( heap );
    return sumSize;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetAllocatableSizeForExpHeapEx

  Description:  拡張ヒープ内の割り当て可能な最大サイズを取得します。
                メモリブロックのアライメントを指定できます。

  Arguments:    heap:      拡張ヒープのハンドル。
                alignment: 確保するメモリブロックのアライメント。
                           ±4, ±8, ±16, ±32, ±64, ±128, ..., ±256MB の値が指定できます。

  Returns:      拡張ヒープ内の割り当て可能な最大サイズを返します(バイト単位)。
 *---------------------------------------------------------------------------*/
u32
MEMGetAllocatableSizeForExpHeapEx(
    MEMHeapHandle   heap,
    int             alignment
)
{
    //ASSERT(IsValidExpHeapHandle_(heap));

    // alignment のチェック
    //ASSERT(alignment % MIN_ALIGNMENT == 0);
    //ASSERT((abs(alignment) & (abs(alignment) - 1)) == 0);
    //ASSERT(MIN_ALIGNMENT <= abs(alignment));

    alignment = abs(alignment); // 念のため正数化

    LockHeap( heap );

    {
        MEMiExpHeapHead       *pEHHead  = GetExpHeapHeadPtrFromHandle_(heap);
        MEMiExpHeapMBlockHead *pMBlkHd;
        u32                   maxSize   = 0;
        u32                   offsetMin = 0xFFFFFFFF;

        for ( pMBlkHd = pEHHead->mbFreeList.head; pMBlkHd; pMBlkHd = pMBlkHd->pMBHeadNext )
        {
            // アライメントを考慮したメモリブロック位置
            void* baseAddress = RoundUpPtr(GetMemPtrForMBlock_(pMBlkHd), alignment);

            if ( GetUIntPtr(baseAddress) < GetUIntPtr(GetMBlockEndAddr_(pMBlkHd)) )
            {
                const u32 blockSize = GetOffsetFromPtr(baseAddress, GetMBlockEndAddr_(pMBlkHd));
                // アライメントによる空きエリア
                const u32 offset = GetOffsetFromPtr(GetMemPtrForMBlock_(pMBlkHd), baseAddress);

                /*
                    サイズが大きい場合、あるいはサイズが同じでも無駄な空間がより小さい場合は、
                    メモリブロックを更新
                 */
                if ( maxSize < blockSize
                     ||  (maxSize == blockSize && offsetMin > offset)
                )
                {
                    maxSize = blockSize;
                    offsetMin= offset;
                }
            }
        }

        UnlockHeap( heap );

        return maxSize;
    }
}


/*---------------------------------------------------------------------------*
  Name:         MEMiIsEmptyExpHeap

  Description:  拡張ヒープに確保されたブロックが存在するかどうか調べます。

  Arguments:    heap:  拡張ヒープのハンドル。

  Returns:      存在しないならばTRUE、
                存在するならば  FALSE
 *---------------------------------------------------------------------------*/
BOOL
MEMiIsEmptyExpHeap( MEMHeapHandle heap )
{
    MEMiExpHeapHead     *pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_( heap );
    BOOL                ret;

    LockHeap( heap );

    ret = (pExpHeapHd->mbFreeList.head == NULL);

    UnlockHeap( heap );

    return ret;
}

/*---------------------------------------------------------------------------*
  Name:         MEMSetAllocModeForExpHeap

  Description:  拡張ヒープのメモリ確保モードをセットします。

  Arguments:    heap:  拡張ヒープのハンドル。
                mode:  メモリ確保モード。

  Returns:      以前の拡張ヒープのメモリ確保モードを返します。
 *---------------------------------------------------------------------------*/
u16
MEMSetAllocModeForExpHeap(
    MEMHeapHandle     heap,
    u16               mode
)
{
    u16  beforeMode;

    //ASSERT(IsValidExpHeapHandle_(heap));

    LockHeap(heap);
    {
        MEMiExpHeapHead *const pEHHead = GetExpHeapHeadPtrFromHandle_(heap);
        beforeMode = GetAllocMode_(pEHHead);
        SetAllocMode_(pEHHead, mode);
    }
    UnlockHeap(heap);

    return beforeMode;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetAllocModeForExpHeap

  Description:  拡張ヒープのメモリ確保モードを取得します。

  Arguments:    heap:    拡張ヒープのハンドル。

  Returns:      拡張ヒープのメモリ確保モードを返します。
 *---------------------------------------------------------------------------*/
u16
MEMGetAllocModeForExpHeap( MEMHeapHandle heap )
{
    u16 retValue;

    //ASSERT(IsValidExpHeapHandle_(heap));
    LockHeap(heap);
    retValue = GetAllocMode_(GetExpHeapHeadPtrFromHandle_(heap));
    UnlockHeap(heap);
    return retValue;
}


/*---------------------------------------------------------------------------*
  Name:         MEMUseMarginOfAlignmentForExpHeap

  Description:  アライメントの際に発生する隙間の領域を再利用するかどうかを
                設定します。
                デフォルトではFALSEに設定されています。

                TRUEに設定した場合には小さなメモリ領域が有効に使用できる可能性がありますが、
                大量のフリーブロックが生成されメモリ確保時のパフォーマンスが悪化する
                危険性があります。


  Arguments:    heap    拡張ヒープのハンドル
                reuse   TRUEの場合、アライメントで発生する領域を再利用します。
                        FALSEの場合は再利用しません。

  Returns:      以前の設定値が返ります。
 *---------------------------------------------------------------------------*/
BOOL
MEMUseMarginOfAlignmentForExpHeap( MEMHeapHandle heap, BOOL reuse )
{
    MEMiExpHeapHead *const pEHHead = GetExpHeapHeadPtrFromHandle_(heap);
    BOOL before;

    //ASSERT( IsValidExpHeapHandle_(heap) );
    LockHeap(heap);

    before = pEHHead->feature.fields.useMarginOfAlign;
    pEHHead->feature.fields.useMarginOfAlign = reuse;

    UnlockHeap(heap);
    return before;
}


/*---------------------------------------------------------------------------*
  Name:         MEMSetGroupIDForExpHeap

  Description:  拡張ヒープのグループIDをセットします。

  Arguments:    heap:    拡張ヒープのハンドル。
                groupID: セットするグループID値。

  Returns:      以前のグループID値が返ります。
 *---------------------------------------------------------------------------*/
u16
MEMSetGroupIDForExpHeap(
    MEMHeapHandle   heap,
    u16             groupID
)
{
    u16  beforeGroupID;

    //ASSERT( IsValidExpHeapHandle_(heap) );
    //ASSERT( groupID <= MAX_GROUPID      );

    LockHeap(heap);
    {
        MEMiExpHeapHead* pEHHead = GetExpHeapHeadPtrFromHandle_(heap);
        beforeGroupID = pEHHead->groupID;
        pEHHead->groupID = groupID;

    }
    UnlockHeap(heap);

    return beforeGroupID;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetGroupIDForExpHeap

  Description:  拡張ヒープのグループIDを取得します。

  Arguments:    heap:    拡張ヒープのハンドル。

  Returns:      グループID値が返ります。
 *---------------------------------------------------------------------------*/
u16
MEMGetGroupIDForExpHeap( MEMHeapHandle heap )
{
    u16 retValue;

    //ASSERT(IsValidExpHeapHandle_(heap));
    LockHeap(heap);
    retValue = GetExpHeapHeadPtrFromHandle_(heap)->groupID;
    UnlockHeap(heap);

    return retValue;
}

/*---------------------------------------------------------------------------*
  Name:         MEMVisitAllocatedForExpHeap

  Description:  拡張ヒープから割り当てられたメモリブロック全てに対して、
                ユーザが指定した関数を呼ばせます。
                visitor関数で呼ばれるメモリブロックの順番は、確保した順番になります。

                visitor の型 HeapVisitor は次のように定義されています。

                    typedef void (*HeapVisitor)(
                                    void*               memBlock,
                                    MEMHeapHandle    heap,
                                    u32                 userParam);

                                        memBlock:   メモリブロックへのポインタ。
                                        heap:       メモリブロックを含有するヒープ。
                                        userParam:  ユーザー用パラメータ。

  Arguments:    heap:       拡張ヒープのハンドル。
                visitor:    各メモリブロックに対して呼ばせる関数。
                userParam:  visitor関数に渡すユーザ指定のパラメータ

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMVisitAllocatedForExpHeap(
    MEMHeapHandle      heap,
    MEMHeapVisitor     visitor,
    u32                userParam
)
{
    //ASSERT(IsValidExpHeapHandle_(heap));
    //ASSERT(visitor != NULL);

    LockHeap( heap );
    {
        MEMiExpHeapMBlockHead* pMBHead = GetExpHeapHeadPtrFromHandle_(heap)->mbUsedList.head;

        while ( pMBHead )
        {
            MEMiExpHeapMBlockHead* pMBHeadNext = pMBHead->pMBHeadNext;
            (*visitor)(GetMemPtrForMBlock_(pMBHead), heap, userParam);
            pMBHead = pMBHeadNext;
        }
    }
    UnlockHeap( heap );
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetSizeForMBlockExpHeap

  Description:  拡張ヒープから確保されたメモリブロックのサイズを取得します。

  Arguments:    memBlock:  サイズを取得するメモリブロックへのポインタ。

  Returns:      指定したメモリブロックのサイズを返します(バイト単位)。
 *---------------------------------------------------------------------------*/
u32
MEMGetSizeForMBlockExpHeap( const void* memBlock )
{
    //ASSERT(IsValidUsedMBlock_(memBlock, NULL));

    return GetMBlockHeadCPtr_( memBlock )->blockSize;
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetGroupIDForMBlockExpHeap

  Description:  拡張ヒープから確保されたメモリブロックのグループIDを取得します。

  Arguments:    memBlock:  グループIDを取得するメモリブロックへのポインタ。

  Returns:      指定したメモリブロックのグループIDが返ります。
 *---------------------------------------------------------------------------*/
u16
MEMGetGroupIDForMBlockExpHeap( const void* memBlock )
{
    //ASSERT(IsValidUsedMBlock_(memBlock, NULL));

    return GetGroupIDForMBlock_( GetMBlockHeadCPtr_(memBlock) );
}

/*---------------------------------------------------------------------------*
  Name:         MEMGetAllocDirForMBlockExpHeap

  Description:  拡張ヒープから確保されたメモリブロックの確保方向を取得します。

  Arguments:    memBlock:  メモリブロックへのポインタ。

  Returns:      指定したメモリブロックの確保方向が返ります。
 *---------------------------------------------------------------------------*/
u16
MEMGetAllocDirForMBlockExpHeap( const void* memBlock )
{
    //ASSERT(IsValidUsedMBlock_( memBlock, NULL ));

    return GetAllocDirForMBlock_( GetMBlockHeadCPtr_(memBlock) );
}


/*---------------------------------------------------------------------------*
  Name:         MEMAdjustExpHeap

  Description:  拡張ヒープの空き領域を解放し、拡張ヒープが使用するメモリ領域を縮小します。
                ヒープメモリの後ろから確保されたメモリブロックが存在していてはいけません。

  Arguments:    heap:      拡張ヒープのハンドル。

  Returns:      関数が成功した場合、縮小後の拡張ヒープの全体サイズを返します。
                (ヘッダ部分も含む)
                失敗した場合、0を返します。

 *---------------------------------------------------------------------------*/
u32
MEMAdjustExpHeap( MEMHeapHandle heap )
{
    //ASSERT( IsValidExpHeapHandle_(heap) );

    {
        MEMiHeapHead            *pHeapHd    = heap;
        MEMiExpHeapHead         *pExpHeapHd = GetExpHeapHeadPtrFromHeapHead_( heap );
        MEMiExpHeapMBlockHead   *pMBlkHd;
        u32                     retVal;

        LockHeap( heap );

        pMBlkHd  = pExpHeapHd->mbFreeList.tail;

        // 空き領域が無い場合は失敗
        if ( pMBlkHd == NULL )
        {
            retVal = 0;
            goto ret_;
        }

        {
            void * const pMBlk      = GetMemPtrForMBlock_( pMBlkHd );
            void * const pMBlkEnd   = AddU32ToPtr( pMBlk, pMBlkHd->blockSize );
            u32 blockSize;

            // 末尾から確保されたメモリブロックが存在する場合は失敗
            if ( pMBlkEnd != MEMGetHeapEndAddress( heap ) )
            {
                retVal = 0;
                goto ret_;
            }

            // 解放されたフリーブロックをフリーリストから削除
            (void)RemoveMBlock_( &pExpHeapHd->mbFreeList, pMBlkHd );

            blockSize = pMBlkHd->blockSize + sizeof( MEMiExpHeapMBlockHead );
            pHeapHd->heapEnd = SubU32ToPtr( pHeapHd->heapEnd, blockSize );

            retVal = GetOffsetFromPtr( pHeapHd, pHeapHd->heapEnd );
        }
ret_:
        UnlockHeap( heap );

        return retVal;
    }
}


#if defined(_DEBUG)

/*---------------------------------------------------------------------------*
  Name:         MEMCheckExpHeap

  Description:  拡張ヒープが破壊されていないかどうかをチェックします。

  Arguments:    heap:     拡張ヒープのハンドル。
                optFlag:  フラグ。

  Returns:      ヒープが正常だった場合 TRUE を返します。
                ヒープにエラーがあった場合、FALSE を返します。
 *---------------------------------------------------------------------------*/
BOOL
MEMCheckExpHeap(
    MEMHeapHandle     heap,
    u32               optFlag
)
{
    const BOOL bPrint = 0 != (optFlag & MEM_HEAP_ERROR_PRINT);
    u32        totalBytes  = 0;
    BOOL       retVal;

    if ( ! IsValidExpHeapHandle_(heap) )
    {
        HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Invalid heap handle. - %08X\n", heap);
        return FALSE;
    }

    LockHeap( heap );
    {
        MEMiHeapHead *const    pHeapHd     = heap;
        MEMiExpHeapHead *const pExpHeapHd  = GetExpHeapHeadPtrFromHeapHead_(pHeapHd);
        MEMiExpHeapMBlockHead* pMBHead     = NULL;
        MEMiExpHeapMBlockHead* pMBHeadPrev = NULL;

        // 使用ブロック
        for ( pMBHead = pExpHeapHd->mbUsedList.head; pMBHead; pMBHeadPrev = pMBHead, pMBHead = pMBHead->pMBHeadNext )
        {
            if ( ! CheckUsedMBlock_(pMBHead, pHeapHd, optFlag)
              || ! CheckMBlockPrevPtr_(pMBHead, pMBHeadPrev, optFlag)   // 前ブロックへのポインタが1ループ前のメモリブロックのポインタと同じでない?
            )
            {
                retVal = FALSE;
                goto ret_;
            }

            // 占有サイズを積算
            totalBytes += sizeof(MEMiExpHeapMBlockHead) + pMBHead->blockSize + GetAlignmentForMBlock_(pMBHead);
        }

        if ( ! CheckMBlockLinkTail_(pMBHeadPrev, pExpHeapHd->mbUsedList.tail, "tail", optFlag))  // 末尾ブロックが最後のブロックへのポインタを指しているか?
        {
            retVal = FALSE;
            goto ret_;
        }

        // フリーブロック
        pMBHead = NULL;
        pMBHeadPrev = NULL;
        for ( pMBHead = pExpHeapHd->mbFreeList.head; pMBHead; pMBHeadPrev = pMBHead, pMBHead = pMBHead->pMBHeadNext )
        {
            if ( ! CheckFreeMBlock_(pMBHead, pHeapHd, optFlag)
                 || ! CheckMBlockPrevPtr_(pMBHead, pMBHeadPrev, optFlag)   // 前ブロックへのポインタが1ループ前のメモリブロックのポインタと同じでない?
            )
            {
                retVal = FALSE;
                goto ret_;
            }

            // 占有サイズを積算
            totalBytes += sizeof(MEMiExpHeapMBlockHead) + pMBHead->blockSize;
        }

        if ( ! CheckMBlockLinkTail_(pMBHeadPrev, pExpHeapHd->mbFreeList.tail, "tail", optFlag) )  // 末尾ブロックが最後のブロックへのポインタを指しているか?
        {
            retVal = FALSE;
            goto ret_;
        }

        // 全体の結果表示
        if ( totalBytes != GetOffsetFromPtr(pHeapHd->heapStart, pHeapHd->heapEnd) )
        {
            HEAP_WARNING(bPrint, "[OS Foundation " "Exp" " Heap]" " Incorrect total memory block size. - heap size %08X, sum size %08X\n",
                GetOffsetFromPtr(pHeapHd->heapStart, pHeapHd->heapEnd), totalBytes);
            retVal = FALSE;
            goto ret_;
        }
        retVal = TRUE;
    }
ret_:
    UnlockHeap( heap );
    return retVal;
}


/*---------------------------------------------------------------------------*
  Name:         MEMCheckForMBlockExpHeap

  Description:  拡張ヒープのメモリブロックが破壊されていないかどうかをチェックします。

  Arguments:    memBlock:  チェックするメモリブロックへのポインタ。
                heap:      拡張ヒープのハンドル。
                optFlag:   フラグ。

  Returns:      メモリブロックが正常だった場合 TRUE を返します。
                メモリブロックにエラーがあった場合、FALSE を返します。
 *---------------------------------------------------------------------------*/
BOOL
MEMCheckForMBlockExpHeap(
    const void*     memBlock,
    MEMHeapHandle   heap,
    u32             optFlag
)
{
    const MEMiExpHeapMBlockHead* pMBHead = NULL;
    MEMiHeapHead *const          pHeapHd = heap;
    BOOL                         result  = TRUE;

    if ( ! memBlock )
    {
        return FALSE;
    }

    pMBHead = GetMBlockHeadCPtr_( memBlock );

    if (heap) LockHeap(heap);

    if ( ! CheckUsedMBlock_( pMBHead, pHeapHd, optFlag ) )
    {
        result = FALSE;
        goto done;
    }

    if ( pMBHead->pMBHeadPrev )
    {
        if ( ! CheckUsedMBlock_(pMBHead->pMBHeadPrev, pHeapHd, optFlag)        // 前ブロックのシグネチャとサイズのチェック
             || ! CheckMBlockNextPtr_(pMBHead->pMBHeadPrev, pMBHead, optFlag)  // 前ブロックの次ブロックへのポインタが自分を指しているか?
        )
        {
            result = FALSE;
            goto done;
        }
    }
    else
    {
        if ( pHeapHd )
        {
            // prevが NULL のときは、リストのheadポインタは自分を指しているはず
            if ( ! CheckMBlockLinkTail_(pMBHead, GetExpHeapHeadPtrFromHeapHead_(pHeapHd)->mbUsedList.head, "head", optFlag) )
            {
                result = FALSE;
                goto done;
            }
        }
    }

    if ( pMBHead->pMBHeadNext )
    {
        if ( ! CheckUsedMBlock_(pMBHead->pMBHeadNext, pHeapHd, optFlag)        // 次ブロックのシグネチャとサイズのチェック
             || ! CheckMBlockPrevPtr_(pMBHead->pMBHeadNext, pMBHead, optFlag)  // 次ブロックの前ブロックへのポインタが自分を指しているか?
        )
        {
            result = FALSE;
            goto done;
        }
    }
    else
    {
        if ( pHeapHd )
        {
            // nextが NULL のときは、リストのtailポインタは自分を指しているはず
            if ( ! CheckMBlockLinkTail_(pMBHead, GetExpHeapHeadPtrFromHeapHead_(pHeapHd)->mbUsedList.tail, "tail", optFlag) )
            {
                result = FALSE;
                goto done;
            }
        }
    }

done:
    if (heap) UnlockHeap(heap);
    return result;
}

// #if defined(_DEBUG)
#endif


}
}
}
