﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <winext/cafe/mem/arena.h>
#include <winext/cafe/mem/heapCommon.h>
#include <winext/cafe/mem/defaultHeap.h>
#include <winext/cafe/mem/expHeap.h>
#include <winext/cafe/mem/frameHeap.h>
#include <winext/cafe/mem/unitHeap.h>
#include <winext/cafe/mem/allocator.h>

namespace nw {
namespace internal {
namespace winext {

/* ========================================================================
    static関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    Exp Heap 用
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         AllocatorAllocForExpHeap_

  Description:  拡張ヒープからのメモリ取得関数

  Arguments:    pAllocator  アロケータへのポインタ
                size        取得するメモリサイズ

  Returns:      拡張ヒープから取得したメモリ領域へのポインタ
 *---------------------------------------------------------------------------*/
static void*
AllocatorAllocForExpHeap_(
    MEMAllocator*   pAllocator,
    u32             size
)
{
    MEMHeapHandle const heap = (MEMHeapHandle)pAllocator->pHeap;
    int const alignment   = (int)pAllocator->heapParam1;
    return MEMAllocFromExpHeapEx( heap, size, alignment );
}

/*---------------------------------------------------------------------------*
  Name:         AllocatorFreeForExpHeap_

  Description:  拡張ヒープへのメモリ解放関数

  Arguments:    pAllocator  アロケータへのポインタ
                memBlock    解放数メモリブロックへのポインタ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static void
AllocatorFreeForExpHeap_(
    MEMAllocator*   pAllocator,
    void*           memBlock
)
{
    MEMHeapHandle const heap = (MEMHeapHandle)pAllocator->pHeap;
    MEMFreeToExpHeap( heap, memBlock );
}

/* ------------------------------------------------------------------------
    Frame Heap 用
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         AllocatorAllocForFrmHeap_

  Description:  フレームヒープからのメモリ取得関数

  Arguments:    pAllocator  アロケータへのポインタ
                size        取得するメモリサイズ

  Returns:      フレームヒープから取得したメモリ領域へのポインタ
 *---------------------------------------------------------------------------*/
static void*
AllocatorAllocForFrmHeap_(
    MEMAllocator*   pAllocator,
    u32             size
)
{
    MEMHeapHandle const heap = (MEMHeapHandle)pAllocator->pHeap;
    int const alignment   = (int)pAllocator->heapParam1;
    return MEMAllocFromFrmHeapEx( heap, size, alignment );
}

/*---------------------------------------------------------------------------*
  Name:         AllocatorFreeForFrmHeap_

  Description:  フレームヒープへのメモリ解放関数

  Arguments:    pAllocator  アロケータへのポインタ
                memBlock    解放数メモリブロックへのポインタ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static void
AllocatorFreeForFrmHeap_(
    MEMAllocator*     pAllocator,
    void*             memBlock
)
{
    /*
        フレームヒープではメモリブロック単位のメモリ解放ができないため、
        この実装では何もしないことにする。
    */
}


/* ------------------------------------------------------------------------
    Unit Heap 用
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         AllocatorAllocForUnitHeap_

  Description:  ユニットヒープからのメモリ取得関数

  Arguments:    pAllocator  アロケータへのポインタ
                size        取得するメモリサイズ

  Returns:      ユニットヒープから取得したメモリ領域へのポインタ
 *---------------------------------------------------------------------------*/
static void*
AllocatorAllocForUnitHeap_(
    MEMAllocator*   pAllocator,
    u32             size
)
{
    /*
        ユニットヒープのメモリブロックサイズを超えるサイズの割り当てはできないので
        NULLを返す。
    */
    MEMHeapHandle const heap = (MEMHeapHandle)pAllocator->pHeap;

    if ( size > MEMGetMemBlockSizeForUnitHeap(heap) )
    {
        return NULL;
    }

    return MEMAllocFromUnitHeap(heap);
}

/*---------------------------------------------------------------------------*
  Name:         AllocatorFreeForUnitHeap_

  Description:  ユニットヒープへのメモリ解放関数

  Arguments:    pAllocator  アロケータへのポインタ
                memBlock    解放数メモリブロックへのポインタ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static void
AllocatorFreeForUnitHeap_(
    MEMAllocator*   pAllocator,
    void*           memBlock
)
{
    MEMHeapHandle const heap = (MEMHeapHandle)pAllocator->pHeap;
    MEMFreeToUnitHeap( heap, memBlock );
}


/* ------------------------------------------------------------------------
    OS heap 用
   ------------------------------------------------------------------------ */
/*---------------------------------------------------------------------------*
  Name:         AllocatorAllocForDefaultHeap_

  Description:

  Arguments:    pAllocator  アロケータへのポインタ
                size        取得するメモリサイズ

  Returns:
 *---------------------------------------------------------------------------*/
static void*
AllocatorAllocForDefaultHeap_(
    MEMAllocator*   pAllocator,
    u32             size
)
{
    return MEMAllocFromDefaultHeap(size);
}

/*---------------------------------------------------------------------------*
  Name:         AllocatorFreeForDefaultHeap_

  Description:

  Arguments:    pAllocator  アロケータへのポインタ
                memBlock    解放数メモリブロックへのポインタ

  Returns:
 *---------------------------------------------------------------------------*/
static void
AllocatorFreeForDefaultHeap_(
    MEMAllocator*   pAllocator,
    void*           memBlock
)
{
    MEMFreeToDefaultHeap(memBlock);
}



/* ========================================================================
    外部関数
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         MEMAllocFromAllocator

  Description:  アロケータからメモリブロック確保します。

                実際には、どのように確保されるかどうかはアロケータと
                それに関連付けられているメモリマネージャの実装によります。

  Arguments:    pAllocator:  Allocator構造体のアドレス。
                size:        メモリブロックのサイズ(バイト)。

  Returns:      メモリブロックを確保できた場合、そのメモリブロックの先頭アドレスを返します。
                確保できなかった場合、NULL を返します。
 *---------------------------------------------------------------------------*/
void*
MEMAllocFromAllocator(
    MEMAllocator*   pAllocator,
    u32             size
)
{
    //ASSERT(pAllocator);
    return (*pAllocator->pFunc->pfAlloc)(pAllocator, size);
}

/*---------------------------------------------------------------------------*
  Name:         MEMFreeToAllocator

  Description:  メモリブロックを解放しアロケータへ返却します。

                実際には、どのように返却されるかどうかはアロケータと
                それに関連付けられているメモリマネージャの実装によります。

  Arguments:    pAllocator:  Allocator構造体のアドレス。
                memBlock:    解放するメモリブロックへのポインタ。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMFreeToAllocator(
    MEMAllocator*   pAllocator,
    void*           memBlock
)
{
    //ASSERT(pAllocator && memBlock);
    (*pAllocator->pFunc->pfFree)(pAllocator, memBlock);
}

/*---------------------------------------------------------------------------*
  Name:         MEMInitAllocatorForExpHeap

  Description:  拡張ヒープからメモリの確保と解放を行うようにアロケータを初期化します。
                アロケータを通じて確保される全てのメモリブロックのアライメント値は
                引数alignmentで指定した値になります。

  Arguments:    pAllocator:  MEMAllocator構造体のアドレス。
                heap:        拡張ヒープのハンドル。
                alignment:   確保する各メモリブロックに適用するアライメント値

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMInitAllocatorForExpHeap(
    MEMAllocator*    pAllocator,
    MEMHeapHandle    heap,
    int              alignment
)
{
    static const MEMAllocatorFunc sAllocatorFunc =
    {
        AllocatorAllocForExpHeap_,
        AllocatorFreeForExpHeap_,
    };

    pAllocator->pFunc = &sAllocatorFunc;
    pAllocator->pHeap = heap;
    pAllocator->heapParam1 = (u32)alignment;
    pAllocator->heapParam2 = 0; // no use
}

/*---------------------------------------------------------------------------*
  Name:         MEMInitAllocatorForFrmHeap

  Description:  フレームヒープからメモリの確保と解放を行うようにアロケータを初期化します。
                アロケータを通じて確保される全てのメモリブロックのアライメント値は
                引数alignmentで指定した値になります。

  Arguments:    pAllocator:  MEMAllocator構造体のアドレス。
                heap:        フレームヒープのハンドル。
                alignment:   確保する各メモリブロックに適用するアライメント値

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMInitAllocatorForFrmHeap(
    MEMAllocator*    pAllocator,
    MEMHeapHandle    heap,
    int              alignment
)
{
    static const MEMAllocatorFunc sAllocatorFunc =
    {
        AllocatorAllocForFrmHeap_,
        AllocatorFreeForFrmHeap_,
    };

    pAllocator->pFunc = &sAllocatorFunc;
    pAllocator->pHeap = heap;
    pAllocator->heapParam1 = (u32)alignment;
    pAllocator->heapParam2 = 0; // no use
}

/*---------------------------------------------------------------------------*
  Name:         MEMInitAllocatorForUnitHeap

  Description:  ユニットヒープからメモリの確保と解放を行うようにアロケータを初期化します。
                ユニットヒープのメモリブロックサイズより大きなメモリブロックを確保することは出来ません。
                その場合、関数AllocFromAllocator() はNULLを返します。

  Arguments:    pAllocator:  MEMAllocator構造体のアドレス。
                heap:        ユニットヒープのハンドル。

  Returns:      なし。
 *---------------------------------------------------------------------------*/
void
MEMInitAllocatorForUnitHeap(
    MEMAllocator*    pAllocator,
    MEMHeapHandle    heap
)
{
    static const MEMAllocatorFunc sAllocatorFunc =
    {
        AllocatorAllocForUnitHeap_,
        AllocatorFreeForUnitHeap_,
    };

    pAllocator->pFunc = &sAllocatorFunc;
    pAllocator->pHeap = heap;
    pAllocator->heapParam1 = 0; // no use
    pAllocator->heapParam2 = 0; // no use
}

/*---------------------------------------------------------------------------*
  Name:         MEMInitAllocatorForDefaultHeap

  Description:

  Arguments:    pAllocator:  MEMAllocator構造体のアドレス。

  Returns:
 *---------------------------------------------------------------------------*/
void
MEMInitAllocatorForDefaultHeap( MEMAllocator* pAllocator )
{
    static const MEMAllocatorFunc sAllocatorFunc =
    {
        AllocatorAllocForDefaultHeap_,
        AllocatorFreeForDefaultHeap_,
    };

    pAllocator->pFunc = &sAllocatorFunc;
    pAllocator->pHeap = MEMGetBaseHeapHandle(MEM_ARENA_2);
    pAllocator->heapParam1 = 0; // no use
    pAllocator->heapParam2 = 0; // no use
}

} // namespace winext
} // namespace internal
} // namespace nw
