﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#ifndef NW_WINEXT_MEM_HEAPCOMMONI_H_
#define NW_WINEXT_MEM_HEAPCOMMONI_H_

#include <winext/cafe/mem/heapCommon.h>
#include <string.h>
#include <winext/cafe/os.h>

namespace nw {
namespace internal {
namespace winext {


/* =======================================================================
    型定義
   ======================================================================== */

typedef s32 IntPtr;     // void*ポインタと相互変換可能な符号付き整数型
typedef u32 UIntPtr;    // void*ポインタと相互変換可能な符号無し整数型


/* ========================================================================
    マクロ関数
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         RoundUp

  Description:  指定した境界値に合わせるため、上位方向に切り上げます。

  Arguments:    value:      対象となるデータ
                alignment:  境界値

  Returns:      指定した境界に切り上げられた値を返します。
 *---------------------------------------------------------------------------*/
#define RoundUp( value, alignment ) \
                (((value) + ((alignment)-1)) & ~((alignment)-1))

#define RoundUpPtr( ptr, alignment ) \
                ( (void*)RoundUp(GetUIntPtr(ptr), (alignment)) )


/*---------------------------------------------------------------------------*
  Name:         RoundDown

  Description:  指定した境界値に合わせるため、下位方向に切り下げます。

  Arguments:    value:      対象となるデータ
                alignment:  境界値

  Returns:      指定した境界に切り下げられた値を返します。
 *---------------------------------------------------------------------------*/
#define RoundDown( value, alignment ) \
                ( (value) & ~((alignment)-1) )

#define RoundDownPtr(ptr, alignment) \
                ( (void*)RoundDown(GetUIntPtr(ptr), (alignment)) )


/* =======================================================================
    関数プロトタイプ
   ======================================================================== */

void    MEMiInitHeapHead(
                MEMiHeapHead*     pHeapHd,
                u32               signature,
                void*             heapStart,
                void*             heapEnd,
                u16               optFlag );

void    MEMiFinalizeHeap( MEMiHeapHead* pHeapHd );

void    MEMiDumpHeapHead( MEMiHeapHead* pHeapHd );

void    MEMInitSLock  ( MEMSLock* slock );
void    MEMLockSLock  ( MEMSLock* slock );
void    MEMUnlockSLock( MEMSLock* slock );


/* ========================================================================
    inline関数
   ======================================================================== */

/* ------------------------------------------------------------------------
    ポインタ操作
   ------------------------------------------------------------------------ */

// ポインタからu32への変換
static inline UIntPtr
GetUIntPtr( const void* ptr )
{
    return (UIntPtr)(ptr);
}

// ２つのポインタのオフセット値を取得
static inline u32
GetOffsetFromPtr( const void* start, const void* end )
{
    return GetUIntPtr(end) - GetUIntPtr(start);
}

// ポインタにu32の値を加算
static inline void*
AddU32ToPtr( void* ptr, u32 val )
{
    return (void*)( GetUIntPtr(ptr) + val );
}

// constポインタにu32の値を加算
static inline const void*
AddU32ToCPtr( const void* ptr, u32 val )
{
    return (const void*)( GetUIntPtr(ptr) + val );
}

// ポインタにs32の値を加算
static inline void*
AddS32ToPtr( void* ptr, s32 val )
{
    return (void*)( GetUIntPtr(ptr) + val );
}

// constポインタにs32の値を加算
static inline const void*
AddS32ToCPtr( const void* ptr, s32 val )
{
    return (const void*)( GetUIntPtr(ptr) + val );
}

// ポインタからu32の値を減算
static inline void*
SubU32ToPtr( void* ptr, u32 val )
{
    return (void*)( GetUIntPtr(ptr) - val );
}

// constポインタからu32の値を減算
static inline const void*
SubU32ToCPtr( const void* ptr, u32 val )
{
    return (const void*)( GetUIntPtr(ptr) - val );
}

// ２つのポインタアドレスを比較
static inline int
ComparePtr( const void* a, const void* b )
{
    const u8* wa = (const u8*)a;
    const u8* wb = (const u8*)b;

    return wa - wb;
}


/*---------------------------------------------------------------------------*
  Name:         GetOptForHeap

  Description:  ヒープからオプションフラグを取得

  Arguments:    pHeapHd     ヒープハンドル

  Returns:      オプションフラグの値
 *---------------------------------------------------------------------------*/
static inline u16
GetOptForHeap( const MEMiHeapHead* pHeapHd )
{
    return (u16)pHeapHd->attribute.fields.optFlag;
}


/*---------------------------------------------------------------------------*
  Name:         SetOptForHeap

  Description:  ヒープへオプションフラグを設定

  Arguments:    pHeapHd     ヒープハンドル

  Returns:      なし。
 *---------------------------------------------------------------------------*/
static inline void
SetOptForHeap(
    MEMiHeapHead*     pHeapHd,
    u16               optFlag
)
{
    pHeapHd->attribute.fields.optFlag = (u8)optFlag;
}

/* ------------------------------------------------------------------------
    ヒープの排他制御
   ------------------------------------------------------------------------ */
/*---------------------------------------------------------------------------*
  Name:         LockHeap

  Description:  ヒープの排他制御をおこなうためのロック関数

  Arguments:    pHeapHd     ヒープハンドル

  Returns:      なし
 *---------------------------------------------------------------------------*/
/// #define LockHeap(hd)    _LockHeap(hd, __FUNCTION__)
/// _LockHeap( MEMiHeapHead*  pHeapHd, const char* fname )

static inline void
LockHeap( MEMiHeapHead*  pHeapHd )
{
    if ( GetOptForHeap( pHeapHd ) & MEM_HEAP_OPT_THREAD_SAFE )
    {
        // OSLockMutex( &pHeapHd->mutex );
        // OSReport("LOCK  : %s %p\n", fname, pHeapHd);
        MEMLockSLock(&pHeapHd->slock);
    }
}


/*---------------------------------------------------------------------------*
  Name:         LockHeap

  Description:  ヒープの排他制御をおこなうためのアンロック関数

  Arguments:    pHeapHd     ヒープハンドル

  Returns:      なし
 *---------------------------------------------------------------------------*/
static inline void
UnlockHeap( MEMiHeapHead* pHeapHd )
{
    if ( GetOptForHeap( pHeapHd ) & MEM_HEAP_OPT_THREAD_SAFE )
    {
        // OSUnlockMutex( &pHeapHd->mutex );
        MEMUnlockSLock(&pHeapHd->slock);
    }
}


/* ------------------------------------------------------------------------
    メモリ充填
   ------------------------------------------------------------------------ */

/*---------------------------------------------------------------------------*
  Name:         FillAllocMemory

  Description:  アロケート直後のメモリフィル

  Arguments:    pHeapHd     ヒープハンドル
                address     アドレス
                size        サイズ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static inline void
FillAllocMemory(
    MEMiHeapHead*   pHeapHd,
    void*           address,
    u32             size
)
{

    if ( GetOptForHeap(pHeapHd) & MEM_HEAP_OPT_0_CLEAR )
    {
        (void)memset( address, 0, size );
    }
    else
    {
        #if defined( _DEBUG )
            if ( GetOptForHeap(pHeapHd) & MEM_HEAP_OPT_DEBUG_FILL )
            {
                (void)memset( address, (int)MEMGetFillValForHeap(MEM_HEAP_FILL_ALLOC), size );
            }
        #endif
    }
}

#if ! defined(_DEBUG)

#define FillFreeMemory(  pHeapHd, address, size )  ((void) 0)
#define FillNoUseMemory( pHeapHd, address, size )  ((void) 0)

#else

/*---------------------------------------------------------------------------*
  Name:         FillFreeMemory

  Description:  フリー直後のメモリフィル

  Arguments:    pHeapHd     ヒープハンドル
                address     アドレス
                size        サイズ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static inline void
FillFreeMemory(
    MEMiHeapHead*     pHeapHd,
    void*             address,
    u32               size
)
{
    if ( GetOptForHeap(pHeapHd) & MEM_HEAP_OPT_DEBUG_FILL )
    {
        (void)memset( address, (int)MEMGetFillValForHeap(MEM_HEAP_FILL_FREE), size );
    }
}

/*---------------------------------------------------------------------------*
  Name:         FillNoUseMemory

  Description:  未使用メモリのメモリフィル

  Arguments:    pHeapHd     ヒープハンドル
                address     アドレス
                size        サイズ

  Returns:      なし
 *---------------------------------------------------------------------------*/
static inline void
FillNoUseMemory(
    MEMiHeapHead*     pHeapHd,
    void*             address,
    u32               size
)
{
    if ( GetOptForHeap(pHeapHd) & MEM_HEAP_OPT_DEBUG_FILL )
    {
        (void)memset( address, (int)MEMGetFillValForHeap(MEM_HEAP_FILL_NOUSE), size );
    }
}

#endif

/* ------------------------------------------------------------------------
     Getter and setter of the default heap handle.
   ------------------------------------------------------------------------ */

MEMHeapHandle _MEMGetDefaultHeapHandle( void );
void          _MEMSetDefaultHeapHandle( MEMHeapHandle newDefaultHeap );

} // namespace winext
} // namespace internal
} // namespace nw

/* NW_WINEXT_MEM_HEAPCOMMONI_H_ */
#endif
