﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string.h>
#include <math.h>
#include <winext/cafe/os.h>
// #include <private/OSInterruptPrivate.h>
#include <winext/cafe/ax.h>
#include <winext/cafe/axfx.h>
#include "AXFXPrivate.h"

namespace nw {
    namespace internal {
        namespace winext {

/*****************************************************************************
  High Quality Reverb (expanded version)
 *****************************************************************************/

/*---------------------------------------------------------------------------*
  Block Diagram :


    IN--+->[ Early Reclection ]----------->[ x gain ]---+
        |                                               |
        |                                               |
        +->[PreDelay]-->[ Fused Reverb ]-->[ x gain ]-[ + ]--[crosstalk]-->OUT
                                                                  |
                                                            (other channels)


    ( Early Reflection )

        (multi-tap delay : without feedback)

        IN--->[ delay line ]-->[ x gain ]-->[ + ]---->OUT
                |      |                     | |
                |      +------>[ x gain ]----+ |
                |                              |
                +------------->[ x gain ]------+


    ( Fused Reverb )

        IN->-+->[Comb]->[ + ]->[AP]->[AP]->[LPF]->[AP(last)]--->OUT
             |           | |
             +->[Comb]->-+ |
             |             |
             +->[Comb]->-\\+

            ※ length of last APs are different for decorrelation of each
               channels


    (Comb Filter)

        IN--->[+]--->[ delay line ] ---+---->OUT
               |                       |
               +------[x coef]<--------+


    (All Pass Filter)

                    +---->--[x -coef]---->-----+
                    |                          |
        IN--->--[+]-+-->[  delay line  ]->-+--[+]--->OUT
                 |                         |
                 +-------<--[x coef]--<----+



    (one pole LPF)

        IN--->[x (1 - coef) ]-->[+]------------------+------>OUT
                                 |                   |
                                 +--[x coef]<--[Z]<--+

                        0(high) <= coef < 1 (low)

 *---------------------------------------------------------------------------*/

/*
 * early reflection sizes (sizes must remain prime numbers)
 */
static u32 __EarlySizeTable[AXFX_REVERBHI_EXP_EARLY_MODE_MAX][3] =
{
    {  157,  479,  829 },  //  5msec
    {  317,  809, 1117 },  // 10msec
    {  479,  941, 1487 },  // 15msec
    {  641, 1259, 1949 },  // 20msec
    {  797, 1667, 2579 },  // 25msec
    {  967, 1901, 2903 },  // 30msec
    { 1123, 2179, 3413 },  // 35msec
    { 1279, 2477, 3889 }   // 40msec
};


/*
 * early reflection delay coefficients
 */
static f32 __EarlyCoefTable[AXFX_REVERBHI_EXP_EARLY_MODE_MAX][3] =
{
    {  0.4f,  -1.0f,  0.3f },    //  5msec
    {  0.5f,  -0.95f, 0.3f },    // 10msec
    {  0.6f,  -0.9f,  0.3f },    // 15msec
    {  0.75f, -0.85f, 0.3f },    // 20msec
    { -0.9f,   0.8f,  0.3f },    // 25msec
    { -1.0f,   0.7f,  0.3f },    // 30msec
    { -1.0f,   0.7f,  0.3f },    // 35msec
    { -1.0f,   0.7f,  0.3f }     // 40msec
};


/*
 * filter delay sizes (sizes must remain prime numbers)
 */
static u32 __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_MAX][8] =
{
    //  C0    C1    C3    A0   A1   A3l  A3r  A3s
    { 1789, 1999, 2333,  433, 149,  47,  73,  67 },  // OLD AXFX
    {  149,  293,  449,  251, 103,  47,  73,  67 },  // Metal Tank
    {  947, 1361, 1531,  433, 137,  47,  73,  67 },  // Small Room
    { 1279, 1531, 1973,  509, 149,  47,  73,  67 },  // Large Room
    { 1531, 1847, 2297,  563, 179,  47,  73,  67 },  // Hall
    { 1823, 2357, 2693,  571, 137,  47,  73,  67 },  // Cavernous
    { 1823, 2357, 2693,  571, 179,  47,  73,  67 }   // Longest (for memory allocation)
};

/*
 * private funcs
 */
static BOOL __AllocDelayLine(AXFX_REVERBHI_EXP *reverb);
static void __FreeDelayLine(AXFX_REVERBHI_EXP *reverb);
static void __BzeroDelayLines(AXFX_REVERBHI_EXP *reverb);
static BOOL __InitParams(AXFX_REVERBHI_EXP *reverb);


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpGetMemSize

  Description:  Returns how many memory bytes specified effect needs to alloc.

  Arguments:    reverb  expanded high quality reverb structure

  Returns:      memory size (byte)
 *---------------------------------------------------------------------------*/

s32 AXFXReverbHiExpGetMemSize(AXFX_REVERBHI_EXP *reverb)
{
    s32 i, ch, size = 0;
    u32 samplerate = AXGetInputSamplesPerSec();

    ASSERTMSG(reverb->preDelayTimeMax >= 0.f, AXFX_ERR_PARAMETER);

    size += __EarlySizeTable[AXFX_REVERBHI_EXP_EARLY_MODE_MAX - 1][2];
    size += (s32)(reverb->preDelayTimeMax * (f32)samplerate);
    for(i = 0; i < 3; i++)
    {
        size += __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][i];
    }
    for(i = 0; i < 2; i++)
    {
        size += __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][3 + i];
    }
    size *= 3;

    for (ch = 0; ch < 3; ch++)
    {
        size += __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][5 + ch];
    }

    return (s32)(size * sizeof(f32));
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpInit

  Description:  initialises expanded high quality reverb effect.

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbHiExpInit(AXFX_REVERBHI_EXP *reverb)
{
    u32 ch, i;
    BOOL result = TRUE;
    u32 samplerate = AXGetInputSamplesPerSec();
    BOOL mask = OSDisableInterrupts();
#if 0
    if (AXGetMode() == AX_MODE_DPL2)
    {
#ifdef _DEBUG
        COSWarn(COSREPORT_MODULE_SOUND,"AXFXReverbHiExpInit(): WARNING: Invalid AX output mode.\n");
#endif
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }
#endif
    reverb->active = AXFX_EFF_INACTIVE;

    //
    // checks parameter
    //
    ASSERTMSG(reverb->preDelayTimeMax >= 0.f, AXFX_ERR_PARAMETER);
    if(reverb->preDelayTimeMax < 0.f)
    {
        AXFXReverbHiExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    //
    // sets max delay size
    //
    reverb->earlyMaxLength = __EarlySizeTable[AXFX_REVERBHI_EXP_EARLY_MODE_MAX - 1][2];

    reverb->preDelayMaxLength = (u32)(reverb->preDelayTimeMax * (f32)samplerate);

    for(i = 0; i < 3; i++)
    {
        reverb->combMaxLength[i] = __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][i];
    }

    for(i = 0; i < 2; i++)
    {
        reverb->allpassMaxLength[i] = __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][3 + i];
    }

    for(ch = 0; ch < 3; ch++)
    {
        reverb->lastAllpassMaxLength[ch] = __FilterSizeTable[AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST][5 + ch];
    }

    //
    // allocates delay lines
    //
    result = __AllocDelayLine(reverb);
    if(result == FALSE)
    {
        AXFXReverbHiExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    //
    // bzeros delay lines
    //
    __BzeroDelayLines(reverb);

    //
    // initialises parameters
    //
    result = __InitParams(reverb);
    if(result == FALSE)
    {
        AXFXReverbHiExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpSettings

  Description:  updates expanded high quality reverb setting.
                (with free & alloc)

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbHiExpSettings(AXFX_REVERBHI_EXP *reverb)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    AXFXReverbHiExpShutdown(reverb);
    result = AXFXReverbHiExpInit(reverb);
    if(result == FALSE)
    {
        AXFXReverbHiExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active |= AXFX_EFF_UPDATE;
    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpSettingsUpdate

  Description:  updates expanded high quality reverb setting.
                (without free & alloc)

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXReverbHiExpSettingsUpdate(AXFX_REVERBHI_EXP *reverb)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    __BzeroDelayLines(reverb);

    result = __InitParams(reverb);
    if(result == FALSE)
    {
        AXFXReverbHiExpShutdown(reverb);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    reverb->active |= AXFX_EFF_UPDATE;
    reverb->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpShutdown

  Description:  shuts down the specified instance of the expanded high quality
                reverb effect and frees any associated memory.

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXReverbHiExpShutdown(AXFX_REVERBHI_EXP *reverb)
{
    BOOL mask = OSDisableInterrupts();

    reverb->active |= AXFX_EFF_INACTIVE;

    __FreeDelayLine(reverb);

    (void)OSRestoreInterrupts(mask);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXReverbHiExpCallback

  Description:  AUX update calback

  Arguments:    bufferUpdate    buffer to update
                reverb          expanded high quality reverb structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXReverbHiExpCallback(AXFX_BUFFERUPDATE *bufferUpdate, AXFX_REVERBHI_EXP *reverb)
{
    u32  ch, i;
    u32  samp;

    s32 *input[3];
    f32  data;
    f32  output[3];

    f32 *earlyLine;
    f32  earlyOut;

    f32 *preDelayLine;
    f32  preDelayOut;

    f32  filterOut;

    f32 *combLine;
    f32  combOutOne;

    f32 *allpass;
    f32  outTmp;
    f32  allpassIn;
    f32  allpassCoef;

    f32  lpfOut;

    f32  lpfCoef1;
    f32  lpfCoef2;

    f32  fusedOut[3];

    f32  fusedGain;
    f32  crosstalkGain;

    f32  crosstalkL;
    f32  crosstalkR;
    f32  crosstalkS;

    s32 *inBusData[3];
    s32 *outBusData[3];

    u32 aux_blocksize =  AXGetInputSamplesPerFrame();

    if(reverb->active != 0)
    {
        reverb->active &= ~AXFX_EFF_UPDATE;
        return;
    }

    input[0] = bufferUpdate->left;
    input[1] = bufferUpdate->right;
    input[2] = bufferUpdate->surround;

    if(reverb->busIn != NULL)
    {
        inBusData[0] = reverb->busIn->left;
        inBusData[1] = reverb->busIn->right;
        inBusData[2] = reverb->busIn->surround;
    }

    if(reverb->busOut != NULL)
    {
        outBusData[0] = reverb->busOut->left;
        outBusData[1] = reverb->busOut->right;
        outBusData[2] = reverb->busOut->surround;
    }

    lpfCoef1 = 1.f - reverb->lpfCoef;
    lpfCoef2 = reverb->lpfCoef;

    allpassCoef = reverb->allpassCoef;

    fusedGain = reverb->fusedGain * AXFX_REVERB_OUT_GAIN;
    crosstalkGain = reverb->crosstalk * AXFX_REVERB_CROSSTALK_GAIN;


    for(samp = 0; samp < aux_blocksize; samp++)
    {
        for(ch = 0; ch < 3; ch++)
        {
            // input
            if(reverb->busIn != NULL)
            {
                data = (f32)(*(input[ch]) + *(inBusData[ch]++));
            }
            else
            {
                data = (f32)(*input[ch]);
            }

            //
            // early reflection
            //
            earlyLine = reverb->earlyLine[ch];
            earlyOut = earlyLine[reverb->earlyPos[0]] * reverb->earlyCoef[0]
                     + earlyLine[reverb->earlyPos[1]] * reverb->earlyCoef[1]
                     + earlyLine[reverb->earlyPos[2]] * reverb->earlyCoef[2];

            earlyLine[reverb->earlyPos[2]] = data;


            //
            // pre-delay
            //
            if(reverb->preDelayLength != 0)
            {
                preDelayLine = reverb->preDelayLine[ch];
                preDelayOut = preDelayLine[reverb->preDelayPos];
                preDelayLine[reverb->preDelayPos] = data;
            }
            else
            {
                preDelayOut = data;
            }

            //
            // comb filter
            //
            filterOut = 0.f;
            for(i = 0; i < 3; i++)
            {
                combLine   = reverb->combLine[ch][i];
                combOutOne = combLine[reverb->combPos[i]];
                combLine[reverb->combPos[i]] = preDelayOut
                                + (combOutOne * reverb->combCoef[i]);
                filterOut += combOutOne;
            }


            //
            // all pass filter
            //

            for(i = 0; i < 2; i++)
            {
                allpass = reverb->allpassLine[ch][i];
                outTmp = allpass[reverb->allpassPos[i]];
                allpassIn =  filterOut + outTmp * allpassCoef;
                allpass[reverb->allpassPos[i]] = allpassIn;
                filterOut =  outTmp - allpassIn * allpassCoef;
            }


            //
            // LPF
            //
            lpfOut =  lpfCoef1 * filterOut
                    + lpfCoef2 * reverb->lastLpfOut[ch];
            reverb->lastLpfOut[ch] = lpfOut;


            //
            // last all pass filter (for decorrelation)
            //
            allpass = reverb->lastAllpassLine[ch];
            outTmp = allpass[reverb->lastAllpassPos[ch]];
            allpassIn = lpfOut + outTmp * allpassCoef;
            allpass[reverb->lastAllpassPos[ch]] = allpassIn;
            fusedOut[ch] = outTmp - allpassIn * allpassCoef;

            // updates delay position of last all pass filter
            if(++reverb->lastAllpassPos[ch] >= reverb->lastAllpassLength[ch])
            {
                reverb->lastAllpassPos[ch] = 0;
            }

            fusedOut[ch] *= fusedGain;
            fusedOut[ch] += earlyOut;
        }


        //
        // crosstalk
        //
        crosstalkL = fusedOut[1] + fusedOut[2];
        crosstalkR = fusedOut[0] + fusedOut[2];
        crosstalkS = fusedOut[0] + fusedOut[1];

        //
        // output
        //
        output[0] = fusedOut[0] + crosstalkL * crosstalkGain;
        output[1] = fusedOut[1] + crosstalkR * crosstalkGain;
        output[2] = fusedOut[2] + crosstalkS * crosstalkGain;

        *(input[0]++) = (s32)(output[0] * reverb->outGain);
        *(input[1]++) = (s32)(output[1] * reverb->outGain);
        *(input[2]++) = (s32)(output[2] * reverb->outGain);

        // bus out
        if(reverb->busOut != NULL)
        {
            *(outBusData[0]++) = (s32)(output[0] * reverb->sendGain);
            *(outBusData[1]++) = (s32)(output[1] * reverb->sendGain);
            *(outBusData[2]++) = (s32)(output[2] * reverb->sendGain);
        }


        //
        // updates delay positions
        //

        // early reflection
        for(i = 0; i < 3; i++)
        {
            if(++reverb->earlyPos[i] >= reverb->earlyLength)
            {
                reverb->earlyPos[i] = 0;
            }
        }

        // pre-delay
        if(reverb->preDelayLength != 0)
        {
            if(++reverb->preDelayPos >= reverb->preDelayLength)
            {
                reverb->preDelayPos = 0;
            }
        }

        // comb filter
        for(i = 0; i < 3; i++)
        {
            if(++reverb->combPos[i] >= reverb->combLength[i])
            {
                reverb->combPos[i] = 0;
            }
        }

        // all pass filter
        for(i = 0; i < 2; i++)
        {
            if(++reverb->allpassPos[i] >= reverb->allpassLength[i])
            {
                reverb->allpassPos[i] = 0;
            }
        }
    }
}



/*===========================================================================*
   Private Funcs
 *===========================================================================*/

/*---------------------------------------------------------------------------*
  Name:         __AllocDelayLine

  Description:  allocates delay lines

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __AllocDelayLine(AXFX_REVERBHI_EXP *reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // allocates early reflection delay
        reverb->earlyLine[ch] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->earlyMaxLength );
        ASSERTMSG(reverb->earlyLine[ch], AXFX_ERR_MEMORY);
        if(reverb->earlyLine[ch]  == NULL) return FALSE;

        // allocates pre delay
        if(reverb->preDelayMaxLength != 0)
        {
            reverb->preDelayLine[ch] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->preDelayMaxLength );
            ASSERTMSG(reverb->preDelayLine[ch], AXFX_ERR_MEMORY);
            if(reverb->preDelayLine[ch]  == NULL) return FALSE;
        }
        else
        {
            reverb->preDelayLine[ch] = NULL;
        }

        // allocates comb filter delay line
        for(i = 0; i < 3; i++)
        {
            reverb->combLine[ch][i] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->combMaxLength[i] );
            ASSERTMSG(reverb->combLine[ch][i], AXFX_ERR_MEMORY);
            if(reverb->combLine[ch][i]  == NULL) return FALSE;
        }

        // allocates all pass filter line
        for(i = 0; i < 2; i++)
        {
            reverb->allpassLine[ch][i] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->allpassMaxLength[i] );
            ASSERTMSG(reverb->allpassLine[ch][i], AXFX_ERR_MEMORY);
            if(reverb->allpassLine[ch][i]  == NULL) return FALSE;
        }

        // allocates last all pass filter line
        reverb->lastAllpassLine[ch] = (f32*)__AXFXAlloc( sizeof(f32) * reverb->lastAllpassMaxLength[ch] );
        ASSERTMSG(reverb->lastAllpassLine[ch], AXFX_ERR_MEMORY);
        if(reverb->lastAllpassLine[ch]  == NULL) return FALSE;
    }

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         __BzeroDelayLines

  Description:  bzeros all delay lines

  Arguments:    reverb        expanded high quality reverb structure

  Returns:      None
 *---------------------------------------------------------------------------*/
static void __BzeroDelayLines(AXFX_REVERBHI_EXP* reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // early reflection
        if(reverb->earlyLine[ch] != NULL)
        {
            (void)memset(reverb->earlyLine[ch], 0,
                    sizeof(f32) * reverb->earlyMaxLength);
        }

        // pre delay
        if(reverb->preDelayLine[ch] != NULL)
        {
            (void)memset(reverb->preDelayLine[ch], 0,
                    sizeof(f32) * reverb->preDelayMaxLength);
        }

        // comb filter
        for(i = 0; i < 3; i++)
        {
            if(reverb->combLine[ch][i] != NULL)
            {
                (void)memset(reverb->combLine[ch][i], 0,
                        sizeof(f32) * reverb->combMaxLength[i]);
            }
        }

        // all pass filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->allpassLine[ch][i] != NULL)
            {
                (void)memset(reverb->allpassLine[ch][i], 0,
                        sizeof(f32) * reverb->allpassMaxLength[i]);
            }
        }

        // last all pass filter
        if(reverb->lastAllpassLine[ch] != NULL)
        {
            (void)memset(reverb->lastAllpassLine[ch], 0,
                    sizeof(f32) * reverb->lastAllpassMaxLength[ch]);
        }
    }
}


/*---------------------------------------------------------------------------*
  Name:         __FreeDelayLine

  Description:  frees delay lines

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      none
 *---------------------------------------------------------------------------*/
static void __FreeDelayLine(AXFX_REVERBHI_EXP *reverb)
{
    u32 ch, i;

    for(ch = 0; ch < 3; ch++)
    {
        // early reflection
        if(reverb->earlyLine[ch] != NULL)
        {
            __AXFXFree(reverb->earlyLine[ch]);
            reverb->earlyLine[ch] = NULL;
        }

        // pre delay
        if(reverb->preDelayLine[ch] != NULL)
        {
            __AXFXFree(reverb->preDelayLine[ch]);
            reverb->preDelayLine[ch] = NULL;
        }

        // comb filter
        for(i = 0; i < 3; i++)
        {
            if(reverb->combLine[ch][i] != NULL)
            {
                __AXFXFree(reverb->combLine[ch][i]);
                reverb->combLine[ch][i] = NULL;
            }
        }

        // all pass filter
        for(i = 0; i < 2; i++)
        {
            if(reverb->allpassLine[ch][i] != NULL)
            {
                __AXFXFree(reverb->allpassLine[ch][i]);
                reverb->allpassLine[ch][i] = NULL;
            }
        }

        // last all pass filter
        if(reverb->lastAllpassLine[ch] != NULL)
        {
            __AXFXFree(reverb->lastAllpassLine[ch]);
            reverb->lastAllpassLine[ch] = NULL;
        }
    }
}



/*---------------------------------------------------------------------------*
  Name:         __InitParams

  Description:  initialises delay params

  Arguments:    reverb       expanded high quality reverb structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __InitParams(AXFX_REVERBHI_EXP *reverb)
{
    u32 ch, i;
    u32 samplerate = AXGetInputSamplesPerSec();

    //
    // check parameters
    //

    ASSERTMSG(reverb->earlyMode < AXFX_REVERBHI_EXP_EARLY_MODE_MAX, AXFX_ERR_PARAMETER);
    if(reverb->earlyMode >= AXFX_REVERBHI_EXP_EARLY_MODE_MAX) return FALSE;

    ASSERTMSG(reverb->preDelayTime >= 0.f && reverb->preDelayTime <= reverb->preDelayTimeMax, AXFX_ERR_PARAMETER);
    if(reverb->preDelayTime < 0.f || reverb->preDelayTime > reverb->preDelayTimeMax) return FALSE;

    ASSERTMSG(reverb->fusedMode < AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST, AXFX_ERR_PARAMETER);
    if(reverb->fusedMode >= AXFX_REVERBHI_EXP_FUSED_MODE_LONGEST) return FALSE;

    ASSERTMSG(reverb->fusedTime >= 0.f, AXFX_ERR_PARAMETER);
    if(reverb->fusedTime < 0.f) return FALSE;

    ASSERTMSG(reverb->coloration >= 0.f && reverb->coloration <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->coloration < 0.f || reverb->coloration > 1.f) return FALSE;

    ASSERTMSG(reverb->damping >= 0.f && reverb->damping <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->damping < 0.f || reverb->damping > 1.f) return FALSE;

    ASSERTMSG(reverb->crosstalk >= 0.f && reverb->crosstalk <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->crosstalk < 0.f || reverb->crosstalk > 1.f) return FALSE;

    ASSERTMSG(reverb->earlyGain >= 0.f && reverb->earlyGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->earlyGain < 0.f || reverb->earlyGain > 1.f) return FALSE;

    ASSERTMSG(reverb->fusedGain >= 0.f && reverb->fusedGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->fusedGain < 0.f || reverb->fusedGain > 1.f) return FALSE;



    ASSERTMSG(reverb->outGain >= 0.f && reverb->outGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->outGain < 0.f || reverb->outGain > 1.f) return FALSE;

    ASSERTMSG(reverb->sendGain >= 0.f && reverb->sendGain <= 1.f, AXFX_ERR_PARAMETER);
    if(reverb->sendGain < 0.f || reverb->sendGain > 1.f) return FALSE;


    //
    // initialises early reflection
    //
    reverb->earlyLength = __EarlySizeTable[reverb->earlyMode][2];
    for(i = 0; i < 3; i++)
    {
        reverb->earlyPos[i] =  reverb->earlyLength
                             - __EarlySizeTable[reverb->earlyMode][i];
        reverb->earlyCoef[i] = __EarlyCoefTable[reverb->earlyMode][i]
                             * reverb->earlyGain * AXFX_REVERB_OUT_GAIN;
    }

    //
    // initialises pre delay
    //
    reverb->preDelayPos = 0;
    reverb->preDelayLength = (u32)(reverb->preDelayTime * (f32)samplerate);

    // initialises comb filter
    for(i = 0; i < 3; i++)
    {
        reverb->combPos[i] = 0;
        reverb->combLength[i] = __FilterSizeTable[reverb->fusedMode][i];
        reverb->combCoef[i] =
                        powf(10.f,
                               (-3.f
                                 * (f32)(reverb->combLength[i])
                                 / (f32)(reverb->fusedTime * samplerate)
                               )
                            );
    }


    //
    // intialises allpass filter
    //
    for(i = 0; i < 2; i++)
    {
        reverb->allpassPos[i] = 0;
        reverb->allpassLength[i] = __FilterSizeTable[reverb->fusedMode][3 + i];
    }

    // last allpass filter
    for(ch = 0; ch < 3; ch++)
    {
        reverb->lastAllpassPos[ch] = 0;
        reverb->lastAllpassLength[ch] = __FilterSizeTable[reverb->fusedMode][5 + ch];
    }

    reverb->allpassCoef = reverb->coloration;


    //
    // initialises LPF
    //
    reverb->lpfCoef = 1.f - reverb->damping;
    if(reverb->lpfCoef > 0.95f) reverb->lpfCoef = 0.95f;

    for(ch = 0; ch < 3; ch++)
    {
        reverb->lastLpfOut[ch] = 0.f;
    }

    return TRUE;

}

        } // namespace winext
    } // namespace internal
} // namespace nw
