﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string.h>
#include <winext/cafe/os.h>
// #include <private/OSInterruptPrivate.h>
#include <winext/cafe/ax.h>
#include <winext/cafe/axfx.h>
#include "AXFXPrivate.h"

namespace nw {
    namespace internal {
        namespace winext {

/*****************************************************************************
  Delay (expanded version)
 *****************************************************************************/

/*---------------------------------------------------------------------------*
  Block Diagram :

    IN--->[LPF]-[+]->[***** delay line *****]-+---->OUT
                 |                            |
                 |                            |
                 +-<--[ x feedback gain ] <---+

 *---------------------------------------------------------------------------*/

static BOOL __AllocDelayLine(AXFX_DELAY_EXP *delay);
static void __FreeDelayLine(AXFX_DELAY_EXP *delay);
static BOOL __InitParams(AXFX_DELAY_EXP *delay);


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpGetMemSize

  Description:  Returns how many memory bytes specified effect needs to alloc.

  Arguments:    delay   expanded delay structure

  Returns:      memory size (byte)
 *---------------------------------------------------------------------------*/

s32 AXFXDelayExpGetMemSize(AXFX_DELAY_EXP *delay)
{
    ASSERTMSG(delay->maxDelay > 0.f, AXFX_ERR_PARAMETER);

    s32 samples_msec = AXGetInputSamplesPerSec() / 1000;

    return (s32)((s32)(delay->maxDelay * (f32)samples_msec) * sizeof(s32) * 3);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpInit

  Description:  initialises expanded delay effect.

  Arguments:    delay   expanded delay structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXDelayExpInit(AXFX_DELAY_EXP *delay)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();
    s32 samples_msec = AXGetInputSamplesPerSec() / 1000;

#if 0
    if (AXGetMode() == AX_MODE_DPL2)
    {
#ifdef _DEBUG
        COSWarn(COSREPORT_MODULE_SOUND,"AXFXDelayExpInit(): WARNING: Invalid AX output mode.\n");
#endif
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }
#endif
    delay->active = AXFX_EFF_INACTIVE;

    // check parameters
    ASSERTMSG(delay->maxDelay > 0.f, AXFX_ERR_PARAMETER);
    if(delay->maxDelay <= 0.f)
    {
        AXFXDelayExpShutdown(delay);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    // sets max delay length
    delay->maxLength = (u32)(delay->maxDelay * (f32)samples_msec);
    if(delay->maxLength <= 0) delay->maxLength = 1;

    // allocate delay lines
    result = __AllocDelayLine(delay);
    if(result == FALSE)
    {
        AXFXDelayExpShutdown(delay);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    // initialise parameters
    result = __InitParams(delay);
    if(result == FALSE)
    {
        AXFXDelayExpShutdown(delay);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    delay->active |= AXFX_EFF_UPDATE;
    delay->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpSettings

  Description:  updates expanded delay setting.
                (with free & alloc)

  Arguments:    delay   expanded delay structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXDelayExpSettings(AXFX_DELAY_EXP *delay)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    delay->active |= AXFX_EFF_INACTIVE;

    AXFXDelayExpShutdown(delay);

    result = AXFXDelayExpInit(delay);
    if(result == FALSE)
    {
        AXFXDelayExpShutdown(delay);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    delay->active |= AXFX_EFF_UPDATE;
    delay->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpSettingsUpdate

  Description:  updates expanded delay setting.
                (without free & alloc)

  Arguments:    delay   expanded delay structure

  Returns:      TRUE=success, FALSE=failure.
 *---------------------------------------------------------------------------*/
BOOL AXFXDelayExpSettingsUpdate(AXFX_DELAY_EXP *delay)
{
    BOOL result = TRUE;
    BOOL mask = OSDisableInterrupts();

    delay->active |= AXFX_EFF_INACTIVE;

    // initialise parameters
    result = __InitParams(delay);
    if(result == FALSE)
    {
        AXFXDelayExpShutdown(delay);
        (void)OSRestoreInterrupts(mask);
        return FALSE;
    }

    delay->active |= AXFX_EFF_UPDATE;
    delay->active &= ~AXFX_EFF_INACTIVE;
    (void)OSRestoreInterrupts(mask);

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpShutdown

  Description:  shuts down the specified instance of the expanded delay effect
                and frees any associated memory.

  Arguments:    delay   expanded delay structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXDelayExpShutdown(AXFX_DELAY_EXP *delay)
{
    BOOL mask = OSDisableInterrupts();

    delay->active |= AXFX_EFF_INACTIVE;

    __FreeDelayLine(delay);

    (void)OSRestoreInterrupts(mask);
}


/*---------------------------------------------------------------------------*
  Name:         AXFXDelayExpCallback

  Description:  AUX update calback

  Arguments:    bufferUpdate    buffer to update
                delay           expanded delay structure

  Returns:      None.
 *---------------------------------------------------------------------------*/
void AXFXDelayExpCallback(AXFX_BUFFERUPDATE *bufferUpdate, AXFX_DELAY_EXP *delay)
{
    u32 samp;
    s32 *indataL;
    s32 *indataR;
    s32 *indataS;

    s32 outL;
    s32 outR;
    s32 outS;

    s32 lpfOutL;
    s32 lpfOutR;
    s32 lpfOutS;

    s32 lpfCoef1, lpfCoef2;

    s32 *inBusData[3];
    s32 *outBusData[3];

    u32 aux_blocksize =  AXGetInputSamplesPerFrame();

    if(delay->active != 0)
    {
        delay->active &= ~AXFX_EFF_UPDATE;
        return;
    }

    lpfCoef1 = 0x80 - delay->lpfCoef;
    lpfCoef2 = delay->lpfCoef;

    indataL = bufferUpdate->left;
    indataR = bufferUpdate->right;
    indataS = bufferUpdate->surround;

    if(delay->busIn != NULL)
    {
        inBusData[0] = delay->busIn->left;
        inBusData[1] = delay->busIn->right;
        inBusData[2] = delay->busIn->surround;
    }

    if(delay->busOut != NULL)
    {
        outBusData[0] = delay->busOut->left;
        outBusData[1] = delay->busOut->right;
        outBusData[2] = delay->busOut->surround;
    }

    for(samp = 0; samp < aux_blocksize; samp++)
    {
        // delay line out
        outL = delay->lineL[delay->curPos];
        outR = delay->lineR[delay->curPos];
        outS = delay->lineS[delay->curPos];

        // LPF process
        if(delay->busIn != NULL)
        {
            lpfOutL = lpfCoef1 * (*(indataL) + *(inBusData[0]++)) + lpfCoef2 * delay->lastLpfOut[0];
            lpfOutR = lpfCoef1 * (*(indataR) + *(inBusData[1]++)) + lpfCoef2 * delay->lastLpfOut[1];
            lpfOutS = lpfCoef1 * (*(indataS) + *(inBusData[2]++)) + lpfCoef2 * delay->lastLpfOut[2];
        }
        else
        {
            lpfOutL = lpfCoef1 * *(indataL) + lpfCoef2 * delay->lastLpfOut[0];
            lpfOutR = lpfCoef1 * *(indataR) + lpfCoef2 * delay->lastLpfOut[1];
            lpfOutS = lpfCoef1 * *(indataS) + lpfCoef2 * delay->lastLpfOut[2];
        }

        lpfOutL >>= 7;
        delay->lastLpfOut[0] = lpfOutL;

        lpfOutR >>= 7;
        delay->lastLpfOut[1] = lpfOutR;

        lpfOutS >>= 7;
        delay->lastLpfOut[2] = lpfOutS;

        // delay line in
        delay->lineL[delay->curPos] = lpfOutL + ( (outL * delay->feedbackGain) >> 7);
        delay->lineR[delay->curPos] = lpfOutR + ( (outR * delay->feedbackGain) >> 7);
        delay->lineS[delay->curPos++] = lpfOutS + ( (outS * delay->feedbackGain) >> 7);

        if(delay->curPos >= delay->length)
        {
            delay->curPos = 0;
        }

        // output
        *(indataL++) = (outL * delay->outGainCalc) >> 7;
        *(indataR++) = (outR * delay->outGainCalc) >> 7;
        *(indataS++) = (outS * delay->outGainCalc) >> 7;

        if(delay->busOut != NULL)
        {
            *(outBusData[0]++) = (outL * delay->sendGainCalc) >> 7;
            *(outBusData[1]++) = (outR * delay->sendGainCalc) >> 7;
            *(outBusData[2]++) = (outS * delay->sendGainCalc) >> 7;
        }
    }
}



/*===========================================================================*
   Private Funcs
 *===========================================================================*/

/*---------------------------------------------------------------------------*
  Name:         __AllocDelayLine

  Description:  allocates delay lines

  Arguments:    delay       expanded delay structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __AllocDelayLine(AXFX_DELAY_EXP *delay)
{
    delay->lineL = (s32*)__AXFXAlloc( sizeof(s32) * delay->maxLength );
    ASSERTMSG(delay->lineL, AXFX_ERR_MEMORY);
    if(delay->lineL == NULL) return FALSE;

    delay->lineR = (s32*)__AXFXAlloc( sizeof(s32) * delay->maxLength );
    ASSERTMSG(delay->lineR, AXFX_ERR_MEMORY);
    if(delay->lineR == NULL) return FALSE;

    delay->lineS = (s32*)__AXFXAlloc( sizeof(s32) * delay->maxLength );
    ASSERTMSG(delay->lineS, AXFX_ERR_MEMORY);
    if(delay->lineS == NULL) return FALSE;

    return TRUE;
}


/*---------------------------------------------------------------------------*
  Name:         __FreeDelayLine

  Description:  frees delay lines

  Arguments:    delay       expanded delay structure

  Returns:      none
 *---------------------------------------------------------------------------*/
static void __FreeDelayLine(AXFX_DELAY_EXP *delay)
{
    if(delay->lineL)
    {
        __AXFXFree(delay->lineL);
        delay->lineL = NULL;
    }

    if(delay->lineR)
    {
        __AXFXFree(delay->lineR);
        delay->lineR = NULL;
    }

    if(delay->lineS)
    {
        __AXFXFree(delay->lineS);
        delay->lineS = NULL;
    }
}


/*---------------------------------------------------------------------------*
  Name:         __InitParams

  Description:  initialises delay params

  Arguments:    delay       expanded delay structure

  Returns:      TRUE=success, FALSE=failure
 *---------------------------------------------------------------------------*/
static BOOL __InitParams(AXFX_DELAY_EXP *delay)
{
    u32 ch;
    f32 lpf_coef;
    s32 samples_msec = AXGetInputSamplesPerSec() / 1000;

    //
    // parameter check
    //
    ASSERTMSG(delay->delay <= delay->maxDelay, AXFX_ERR_PARAMETER);
    if(delay->delay > delay->maxDelay) return FALSE;

    ASSERTMSG(delay->feedback >= 0.f && delay->feedback < 1.f, AXFX_ERR_PARAMETER);
    if(delay->feedback < 0.f || delay->feedback >=1.f) return FALSE;

    ASSERTMSG(delay->lpf >= 0.f && delay->lpf <= 1.f, AXFX_ERR_PARAMETER);
    if(delay->lpf < 0.f || delay->lpf > 1.f) return FALSE;


    ASSERTMSG(delay->outGain >= 0.f && delay->outGain <= 1.f, AXFX_ERR_PARAMETER);
    if(delay->outGain < 0.f || delay->outGain > 1.f) return FALSE;

    ASSERTMSG(delay->sendGain >= 0.f && delay->sendGain <= 1.f, AXFX_ERR_PARAMETER);
    if(delay->sendGain < 0.f || delay->sendGain > 1.f) return FALSE;



    //
    // initialise delay line
    //
    ASSERTMSG(delay->lineL, AXFX_ERR_NO_BUFFER);
    if(delay->lineL == NULL) return FALSE;

    ASSERTMSG(delay->lineR, AXFX_ERR_NO_BUFFER);
    if(delay->lineR == NULL) return FALSE;

    ASSERTMSG(delay->lineS, AXFX_ERR_NO_BUFFER);
    if(delay->lineS == NULL) return FALSE;

    (void)memset(delay->lineL, 0, sizeof(s32) * delay->maxLength);
    (void)memset(delay->lineR, 0, sizeof(s32) * delay->maxLength);
    (void)memset(delay->lineS, 0, sizeof(s32) * delay->maxLength);


    //
    // initialise parameters
    //
    delay->length = (u32)(delay->delay * (f32)samples_msec);
    if(delay->length <= 0) delay->length = 1;

    delay->curPos = 0;

    delay->feedbackGain = (s32)((f32)0x80 * delay->feedback);


    lpf_coef = 1.f - delay->lpf;
    if(lpf_coef > 0.95f) lpf_coef = 0.95f;

    delay->lpfCoef = (s32)((f32)0x80 * lpf_coef);


    for(ch = 0; ch < 3; ch++)
    {
        delay->lastLpfOut[ch] = 0;
    }


    delay->outGainCalc  = (s32)((f32)0x80 * delay->outGain);
    delay->sendGainCalc = (s32)((f32)0x80 * delay->sendGain);


    return TRUE;
}

        } // namespace winext
    } // namespace internal
} // namespace nw
