﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/cafe/os.h>
#include "AXPrivate.h"
#include "AXRenderCosts.h"
#include "AXHelper.h"

namespace nw {
namespace internal {
namespace winext {

/*---------------------------------------------------------------------------*
    local vars
 *---------------------------------------------------------------------------*/

ALIGNED_VAR(static u16, PPC_IO_BUFFER_ALIGN, __AXCommandList[2][AX_HOST_CL_WORDS]);

static u32              __AXCommandListPosition;
static u16              *__AXClWrite;
static s32              __AXClWrote;
static u32              __AXCommandListCycles;

#if (0) // v1.x
static u16              __AXMasterVolume;
static u16              __AXAuxAVolume;
static u16              __AXAuxBVolume;
static u16              __AXAuxCVolume;
static u16              __AXDRCVolume;
static u16              __AXDRCAuxVolume;
#endif

static u16              __AXTVMasterVolume[AX_MAX_NUM_TVS];
static u16              __AXTVAuxRetVolume[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM];
static u16              __AXTVActiveChannels[AX_MAX_NUM_TVS];

static u16              __AXDRCMasterVolume[AX_MAX_NUM_DRCS];
static u16              __AXDRCAuxRetVolume[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM];
static u16              __AXDRCActiveChannels[AX_MAX_NUM_DRCS];

static u16              __AXRmtMasterVolume[AX_MAX_NUM_RMTS];
static u16              __AXRmtAuxRetVolume[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM];

       u32              __AXDeviceMode[AX_MAX_NUM_DEVICES];

static f32              __AXCommandListDspCosts;
static f32              __AXCommandListAxCosts;

// from AXOut.c
extern u16              __debugTrace[32];
/*---------------------------------------------------------------------------*
    private functions
 *---------------------------------------------------------------------------*/

u32 __AXGetCommandListCycles(void)
{
    OSReport("%s is deprecated and non-functions\n", __FUNCTION__);
    return __AXCommandListCycles;
}

void __AXGetCommandListCosts(f32 *dsp_costs, f32 *ax_costs)
{
    *dsp_costs =  __AXCommandListDspCosts;
    *ax_costs  =  __AXCommandListAxCosts;
}

/*---------------------------------------------------------------------------*
    This is called by __AXIstRun() to get the address of the command
    list to send to DSP
 *---------------------------------------------------------------------------*/
u32 __AXGetCommandListAddress(AXPB * root_dsp, AXPB * root_ppc)
{
    u32 address;

    // save the address to the command list to return
    address = (u32)&__AXCommandList[__AXCommandListPosition][0];

    // update its voice list head pointers
    {
        ((u16*)address)[4] = root_dsp ? root_dsp->currHi : 0;
        ((u16*)address)[5] = root_dsp ? root_dsp->currLo : 0;

        ((u16*)address)[7] = (u16)((((u32)root_ppc)>>16)&0x0000ffff);
        ((u16*)address)[8] = (u16)( ((u32)root_ppc     )&0x0000ffff);

//        DCFlushRange((void*)address, 32);
    }

    // increment the position.
    __AXCommandListPosition++;
    __AXCommandListPosition &= 1;

    // set the write position to the new commandlist
    __AXClWrite = &__AXCommandList[__AXCommandListPosition][0];

    return address;
}


/*---------------------------------------------------------------------------*
    This is called by the __AXClNextFrame() to write to the command list.
 *---------------------------------------------------------------------------*/
void __AXWriteToCommandList(u16 data)
{
    *__AXClWrite = data;
    __AXClWrite++;
    __AXClWrote++;
}

/*---------------------------------------------------------------------------*
    This is called by the AI interrupt service routine to generate a command
    list for the next audio frame.
 *---------------------------------------------------------------------------*/
void __AXNextFrame(
        void *buffer,
        AXRmtBuffs *rbuffer,
        AXPB * /* root_dsp */,
        AXPB * /* root_ppc */,
        void *drcbuffer_0,
        void *drcbuffer_1)
{
    u32 data;
    u16 *pCommandList;
    u16 aux_id;

    // Save the command list pointer so we can flush the cache when we are
    // done writing it
    pCommandList = __AXClWrite;
    // Initialize the CL size watcher
    __AXClWrote = 0;

    __AXCommandListAxCosts = 0.0;
    __AXCommandListDspCosts = 0.0;

    // Print the studio parameter block
    data = __AXGetStudio();
    data = (u32)reinterpret_cast<u32>((u32*)data);

    // Start by initializing the "studio"
    __AXWriteToCommandList(AX_CL_INIT_STUDIO);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));

    // Process all the PBs, cycle counts for the voices will be accounted for
    // during PB sync
    // at this time do not know the list
    // the ptrs will be updated in __AXGetCommandListAddress()
    // in the next frame, when this list will be used
    __AXWriteToCommandList(AX_CL_PROCESS_DSP_VOICE);
    __AXWriteToCommandList(0);
    __AXWriteToCommandList(0);

    __AXWriteToCommandList(AX_CL_PROCESS_PPC_VOICE);
    __AXWriteToCommandList(0);
    __AXWriteToCommandList(0);

    // process TV Aux effects
    for (aux_id = 0; aux_id < AX_AUX_ID_MAX_NUM; aux_id++)
    {
        __AXGetAuxInput(AX_DEVICE_TV, AX_TV_ID0, aux_id, &data);

        if (data)
        {
            //
            // [u16 opcode][u16 bus_id][u16 numchans][u16 target_vol][U32 src_addr][U32 dest_addr]
            __AXWriteToCommandList(AX_CL_PROCESS_AUX);
            __AXWriteToCommandList(aux_id+1);
            __AXWriteToCommandList(__AXTVActiveChannels[AX_TV_ID0]);
            __AXWriteToCommandList(__AXTVAuxRetVolume[AX_TV_ID0][aux_id]);

            data = (u32)reinterpret_cast<u32>((u32*)data);
            __AXWriteToCommandList((u16)(data >> 16));
            __AXWriteToCommandList((u16)(data & 0xFFFF));

            __AXGetAuxOutput(AX_DEVICE_TV, AX_TV_ID0, aux_id, &data);
            data = (u32)reinterpret_cast<u32>((u32*)data);
            __AXWriteToCommandList((u16)(data >> 16));
            __AXWriteToCommandList((u16)(data & 0xFFFF));

            __AXCommandListAxCosts += static_cast<f32>(AX_COST_AUX_STAGE_IDLE);
            __AXCommandListAxCosts += AX_COST_AUX_STAGE_RVB;
            __AXCommandListDspCosts += DSP_COST_AUX_STAGE_IDLE;
        }
    }
    // process DRC Aux effects
    for (aux_id = 0; aux_id < AX_AUX_ID_MAX_NUM; aux_id++)
    {
        __AXGetAuxInput(AX_DEVICE_DRC, AX_DRC_ID0, aux_id, &data);

        if (data)
        {
            // [u16 opcode][u16 bus_id][u16 numchans][u16 target_vol][U32 src_addr][U32 dest_addr]
            __AXWriteToCommandList(AX_CL_PROCESS_AUX);
            __AXWriteToCommandList(AX_MAX_NUM_BUSES+aux_id+1);
            __AXWriteToCommandList(__AXDRCActiveChannels[AX_DRC_ID0]);
            __AXWriteToCommandList(__AXDRCAuxRetVolume[AX_DRC_ID0][aux_id]);

            data = (u32)reinterpret_cast<u32>((u32*)data);
            __AXWriteToCommandList((u16)(data >> 16));
            __AXWriteToCommandList((u16)(data & 0xFFFF));

            __AXGetAuxOutput(AX_DEVICE_DRC, AX_DRC_ID0, aux_id, &data);
            data = (u32)reinterpret_cast<u32>((u32*)data);
            __AXWriteToCommandList((u16)(data >> 16));
            __AXWriteToCommandList((u16)(data & 0xFFFF));

            __AXCommandListAxCosts += static_cast<f32>(AX_COST_AUX_STAGE_IDLE);
            __AXCommandListAxCosts += AX_COST_AUX_STAGE_RVB;
            __AXCommandListDspCosts += DSP_COST_AUX_STAGE_IDLE;
        }
    }

    // Dump the remote output to RAM
    __AXWriteToCommandList(AX_CL_OUTPUT_REMOTE);
    data = (u32)reinterpret_cast<u32>((u32*)rbuffer->rmt0);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));
    data = (u32)reinterpret_cast<u32>((u32*)rbuffer->rmt1);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));
    data = (u32)reinterpret_cast<u32>((u32*)rbuffer->rmt2);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));
    data = (u32)reinterpret_cast<u32>((u32*)rbuffer->rmt3);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));
    __AXCommandListCycles += AX_CL_CYCLES_OUTPUT_REMOTE;

    // Dump the TV bus output to RAM for encoding for AI
    __AXWriteToCommandList(AX_CL_FINALIZE);             // [u16 opcode]
    __AXWriteToCommandList(AX_TV_MAIN_BUS_IDX);                          // [u16 bus_id]
    __AXWriteToCommandList(__AXTVActiveChannels[AX_TV_ID0]);    // [u16 numchans]
    __AXWriteToCommandList(__AXTVMasterVolume[AX_TV_ID0]);      // [u16 target_vol]
    data = (u32)reinterpret_cast<u32>((u32*)buffer);
    __AXWriteToCommandList((u16)(data >> 16));          // [U32 dest_addr]
    __AXWriteToCommandList((u16)(data & 0xFFFF));

    __AXCommandListDspCosts  += DSP_COST_FINAL_STAGE;
    __AXCommandListAxCosts  += AX_COST_FINAL_STAGE;

    // Dump the AX_DRC_ID0 bus output to RAM for encoding for AI2

    __AXWriteToCommandList(AX_CL_FINALIZE);        // [u16 opcode]
    __AXWriteToCommandList(AX_DRC0_MAIN_BUS_IDX);              // [u16 bus_id]
    __AXWriteToCommandList(__AXDRCActiveChannels[AX_DRC_ID0]);   // [u16 numchans]
    __AXWriteToCommandList(__AXDRCMasterVolume[AX_DRC_ID0]);     // [u16 target_vol]
    data = (u32)reinterpret_cast<u32>((u32*)drcbuffer_0);
    __AXWriteToCommandList((u16)(data >> 16));          // [U32 dest_addr]
    __AXWriteToCommandList((u16)(data & 0xFFFF));

    __AXCommandListDspCosts  += DSP_COST_FINAL_STAGE;
    __AXCommandListAxCosts  += AX_COST_FINAL_STAGE;

    if (__AXDRCActiveChannels[AX_DRC_ID1])
    {
        __AXWriteToCommandList(AX_CL_FINALIZE);        // [u16 opcode]
        __AXWriteToCommandList(AX_DRC1_MAIN_BUS_IDX);              // [u16 bus_id]
        __AXWriteToCommandList(__AXDRCActiveChannels[AX_DRC_ID1]);   // [u16 numchans]
        __AXWriteToCommandList(__AXDRCMasterVolume[AX_DRC_ID1]);     // [u16 target_vol]
        data = (u32)reinterpret_cast<u32>((u32*)drcbuffer_1);
        __AXWriteToCommandList((u16)(data >> 16));          // [U32 dest_addr]
        __AXWriteToCommandList((u16)(data & 0xFFFF));
    }

    // Terminate the command list
    data = (u32)((u32*)__debugTrace);
    __AXWriteToCommandList(AX_CL_PROCESS_GET_DEBUG_TRACE);
    __AXWriteToCommandList((u16)(data >> 16));
    __AXWriteToCommandList((u16)(data & 0xFFFF));

    __AXWriteToCommandList(AX_CL_END);

    // lets flush the cache to the RAM
//    DCFlushRange(pCommandList, AX_HOST_CL_BYTES);
//    OSReport("CL Size: %d words\n", __AXClWrote);
}

/*---------------------------------------------------------------------------*
    Initialize code module
 *---------------------------------------------------------------------------*/
void __AXClInit(void)
{
#if 0
#ifdef _DEBUG
    extern struct { u32 value; char name[60]; } ax_debug_info[];
    OSReport("Initializing AXCL code module\n");
    ax_debug_info[2].value = (u32)__AXCommandList;
#endif
#endif
    // OSReport("CL0: %8.8x ", (u32) &__AXCommandList[0][0]);
    // OSReport("CL1: %8.8x \n", (u32) &__AXCommandList[1][0]);
    ASSERT(((u32)&__AXCommandList[0][0] & (PPC_IO_BUFFER_ALIGN - 1)) == 0);
    ASSERT(((u32)&__AXCommandList[1][0] & (PPC_IO_BUFFER_ALIGN-1)) == 0);

    __AXDeviceMode[AX_DEVICE_TV]                  = AX_MODE_STEREO;
    __AXCommandListPosition     = 0;
    __AXClWrite                 = &__AXCommandList[0][0];
    // v2.0
    __AXTVActiveChannels[AX_TV_ID0]        = AX_MAX_NUM_TV_CHS; // always in full 6 ch mode
    __AXTVMasterVolume[AX_TV_ID0]          = 0x8000;
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_A] = 0x8000;
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_B] = 0x8000;
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_C] = 0x8000;

    __AXDRCActiveChannels[AX_DRC_ID0]        = AX_MAX_NUM_DRC_CHS; // Stereo Mode! ;
    __AXDRCMasterVolume[AX_DRC_ID0]          = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID0][AX_AUX_ID_A] = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID0][AX_AUX_ID_B] = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID0][AX_AUX_ID_C] = 0x8000;

    __AXDRCActiveChannels[AX_DRC_ID1]        = 0;
    __AXDRCMasterVolume[AX_DRC_ID1]          = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID1][AX_AUX_ID_A] = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID1][AX_AUX_ID_B] = 0x8000;
    __AXDRCAuxRetVolume[AX_DRC_ID1][AX_AUX_ID_C] = 0x8000;

}


/*---------------------------------------------------------------------------*
    Shut down the code module gracefully
 *---------------------------------------------------------------------------*/
void __AXClQuit(void)
{
#ifdef _DEBUG
    OSReport("Shutting down AXCL code module\n");
#endif
}


/*---------------------------------------------------------------------------*
    exposed functions
 *---------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
    This function is called by the user to set mixing mode for the S buffer,
    AX_MODE_STEREO will ignore the S buffer, AX_MODE_SURROUND and AX_MODE_DPL2
    will surround encode the S buffer, AX_MODE_6CHAN will simply route to it.
 *---------------------------------------------------------------------------*/
// temp declare this until ai.h fixed
ALIGNED_VAR(static s16, PPC_IO_BUFFER_ALIGN, __pcm5_1_zero_buffer[144*6*2] = { 0 });
void __AXInitOutBalanced(void);


// this function is deprecated for external usage
void AXSetMode(u32 /* mode */)
{
    OSReport("%s is deprecated\n", __FUNCTION__);
}

// this is for internal use only
void __AXSetMode(u32 mode)
{

    mode &= 7;
    ASSERT( (mode == AX_MODE_STEREO)       ||
            (mode == AX_MODE_SURROUND)     ||
            (mode == AX_MODE_6CHAN)        ||
            (mode == AX_MODE_DPL2)
            );

    __AXDeviceMode[AX_DEVICE_TV] = mode;
    __AXTVActiveChannels[AX_TV_ID0] = AX_MAX_NUM_TV_CHS;
}


/*---------------------------------------------------------------------------*
    This function is called by the user to get current mixing mode for the S
    buffer,
 *---------------------------------------------------------------------------*/
u32 AXGetMode(void)
{
    return __AXDeviceMode[AX_DEVICE_TV];
}

/*---------------------------------------------------------------------------*
    This function returns Master volume.
 *---------------------------------------------------------------------------*/
u16 AXGetMasterVolume(void)
{
    return __AXTVMasterVolume[AX_TV_ID0];
}

/*---------------------------------------------------------------------------*
    Get and Set mov master volume.
 *---------------------------------------------------------------------------*/
u16 AXGetMasterDRCVolume(void)
{
    return __AXDRCMasterVolume[AX_DRC_ID0];
}

void AXSetMasterDRCVolume(u16 volume)
{
    __AXDRCMasterVolume[AX_DRC_ID0] = volume;
}

/*---------------------------------------------------------------------------*
    This function returns AuxA return volume.
 *---------------------------------------------------------------------------*/
u16 AXGetAuxAReturnVolume(void)
{
    return __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_A];
}

/*---------------------------------------------------------------------------*
    This function returns AuxB return volume.
 *---------------------------------------------------------------------------*/
u16 AXGetAuxBReturnVolume(void)
{
    return __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_B];
}

/*---------------------------------------------------------------------------*
    This function returns AuxC return volume.
 *---------------------------------------------------------------------------*/
u16 AXGetAuxCReturnVolume(void)
{
    return __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_C];
}

/*---------------------------------------------------------------------------*
    This function returns DRC Aux return volume.
 *---------------------------------------------------------------------------*/
u16 AXGetDRCAuxReturnVolume(void)
{
    return __AXDRCAuxRetVolume[AX_DRC_ID0][AX_AUX_ID_A];
}

/*---------------------------------------------------------------------------*
    This function sets Master volume.
 *---------------------------------------------------------------------------*/
void AXSetMasterVolume(u16 volume)
{
    ASSERT(volume <= 0x8000);

    if (volume > 0x8000)
    {
        volume = 0x8000;
    }

    __AXTVMasterVolume[AX_TV_ID0] = volume;
}

/*---------------------------------------------------------------------------*
    This function sets AuxA return volume.
 *---------------------------------------------------------------------------*/
void AXSetAuxAReturnVolume(u16 volume)
{
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_A] = volume;
}

/*---------------------------------------------------------------------------*
    This function sets AuxB return volume.
 *---------------------------------------------------------------------------*/
void AXSetAuxBReturnVolume(u16 volume)
{
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_B] = volume;
}

/*---------------------------------------------------------------------------*
    This function sets AuxC return volume.
 *---------------------------------------------------------------------------*/
void AXSetAuxCReturnVolume(u16 volume)
{
    __AXTVAuxRetVolume[AX_TV_ID0][AX_AUX_ID_C] = volume;
}

/*---------------------------------------------------------------------------*
    This function sets DRC Aux return volume.
 *---------------------------------------------------------------------------*/
void AXSetDRCAuxReturnVolume(u16 volume)
{
    __AXDRCAuxRetVolume[AX_DRC_ID0][AX_AUX_ID_A] = volume;
}


/*******************************************************************************
 * Function:    AXSetDeviceVolume
 * Arguments:
 *              device  :    (in) device type
 *              Id      :    (in) Id of the device type
 *              volume  :    (in) volume to be set
 *  Sets the master volume for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXSetDeviceVolume(AXPBDeviceType device, u32 Id, u16 volume)
{
    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    u16 *mVolPtr;
    // Steps:
    // - Check for validity of the inputs
    // - in the switch code, set up max channels, pointers appropriately
    retErrCode = __AXCheckDeviceArgs(device, Id);
    if(AXPB_ERROR_NONE == retErrCode)
    {
        switch(device)
        {
            case AX_DEVICE_TV:
                // case of TV

                mVolPtr = &__AXTVMasterVolume[Id];
                break;  // end of case TV
            case AX_DEVICE_DRC:
                // now do the mix updates as required
                mVolPtr = &__AXDRCMasterVolume[Id];
                break;  // end of case DRC
            case AX_DEVICE_RMT:
                mVolPtr = &__AXRmtMasterVolume[Id];
                break;  // end of case RMT
        } // end of switch(device)


        // set the volume for that particular device
        *mVolPtr = volume;
    } // end of error checking

    return retErrCode;
} // end of AXSetDeviceVolume
/*******************************************************************************
 * Function:    AXGetDeviceVolume
 * Arguments:
 *              device  :    (in) device type
 *              Id      :    (in) Id of the device type
 *              *volume :    (in) pointer to variable to return volume
 *  Gets the master volume for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXGetDeviceVolume(AXPBDeviceType device, u32 Id, u16 *volume)
{
    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    u16 *mVolPtr;
    // Steps:
    // - Check for validity of the inputs
    // - in the switch code, set up max channels, pointers appropriately
    retErrCode = __AXCheckDeviceArgs(device, Id);
    if(AXPB_ERROR_NONE == retErrCode)
    {
        switch(device)
        {
            case AX_DEVICE_TV:
                // case of TV

                mVolPtr = &__AXTVMasterVolume[Id];
                break;  // end of case TV
            case AX_DEVICE_DRC:
                // now do the mix updates as required
                mVolPtr = &__AXDRCMasterVolume[Id];
                break;  // end of case DRC
            case AX_DEVICE_RMT:
                mVolPtr = &__AXRmtMasterVolume[Id];
                break;  // end of case RMT
        } // end of switch(device)


        // set the volume for that particular device
        *volume= *mVolPtr ;
    } // end of error checking

    return retErrCode;
} // end of AXGetDeviceVolume

/*******************************************************************************
 * Function:    AXSetAuxReturnVolume
 * Arguments:
 *              device  :    (in) device type
 *              Id      :    (in) Id of the device type
 *              auxId   :    (in) Aux for which the volume needs to be set
 *              volume  :    (in) volume to be set
 *  Sets the master volume for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXSetAuxReturnVolume(AXPBDeviceType device, u32 Id, u32 auxId, u16 volume)
{
    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    u16 *mVolPtr;
    // Steps:
    // - Check for validity of the inputs
    // - in the switch code, set up max channels, pointers appropriately
    retErrCode = __AXCheckDeviceArgs(device, Id);
    if(AXPB_ERROR_NONE == retErrCode)
    {
        // check the auxId as well since __AXCheckDeviceArgs doesnt check for it

        //ASSERT(auxId < AX_AUX_ID_MAX_NUM);
        if(auxId >= AX_AUX_ID_MAX_NUM)
        {
            retErrCode = AXPB_ERROR_AUX_ID;
            return retErrCode;
        }
        switch(device)
        {
            case AX_DEVICE_TV:
                // case of TV
                mVolPtr = &__AXTVAuxRetVolume[Id][auxId];
                break;  // end of case TV
            case AX_DEVICE_DRC:
                // DRC
                mVolPtr = &__AXDRCAuxRetVolume[Id][auxId];
                break;  // end of case DRC
            case AX_DEVICE_RMT:
                // RMT
                mVolPtr = &__AXRmtAuxRetVolume[Id][auxId];
                break;  // end of case RMT
        } // end of switch(device)


        // set the volume for that particular device
        *mVolPtr = volume;
    } // end of error checking

    return retErrCode;
} // end of AXSetAuxReturnVolume
/*******************************************************************************
 * Function:    AXGetAuxReturnVolume
 * Arguments:
 *              device  :    (in) device type
 *              Id      :    (in) Id of the device type
 *              auxId   :    (in) Aux for which the volume needs to be set
 *              *volume :    (in) pointer to variable to return volume
 *  Gets the master volume for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXGetAuxReturnVolume(AXPBDeviceType device, u32 Id, u32 auxId, u16 *volume)
{
    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    u16 *mVolPtr;
    // Steps:
    // - Check for validity of the inputs
    // - in the switch code, set up max channels, pointers appropriately
    retErrCode = __AXCheckDeviceArgs(device, Id);
    if(AXPB_ERROR_NONE == retErrCode)
    {
        // check the auxId as well since __AXCheckDeviceArgs doesnt check for it

        //ASSERT(auxId < AX_AUX_ID_MAX_NUM);
        if(auxId >= AX_AUX_ID_MAX_NUM)
        {
            retErrCode = AXPB_ERROR_AUX_ID;
            return retErrCode;
        }
        switch(device)
        {
            case AX_DEVICE_TV:
                // case of TV
                mVolPtr = &__AXTVAuxRetVolume[Id][auxId];
                break;  // end of case TV
            case AX_DEVICE_DRC:
                // DRC
                mVolPtr = &__AXDRCAuxRetVolume[Id][auxId];
                break;  // end of case DRC
            case AX_DEVICE_RMT:
                // RMT
                mVolPtr = &__AXRmtAuxRetVolume[Id][auxId];
                break;  // end of case RMT
        } // end of switch(device)


        // set the volume for that particular device
        *volume = *mVolPtr;
    } // end of error checking

    return retErrCode;
} // end of AXSetAuxReturnVolume
/*******************************************************************************
 * Function:    AXSetDeviceMode
 * Arguments:
 *              device  :    (in) device type
 *              mode    :    (in) mode to set
 *  Sets the mode of operation for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXSetDeviceMode(AXPBDeviceType device, u32 mode)
{

    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;

    // currently not checking if mode passed is correct, later add it
    switch(device)
    {
        case AX_DEVICE_TV:

            AXSetMode(mode);    // set up hardware too
            break;
        case AX_DEVICE_DRC:
            __AXDeviceMode[AX_DEVICE_DRC] = mode;
            break;
        case AX_DEVICE_RMT:
            __AXDeviceMode[AX_DEVICE_RMT] = mode;
            break;
        default:
            //ASSERT(0);
            retErrCode = AXPB_ERROR_DEVICE_TYPE;
            break;
    } // end of switch

    return retErrCode;
} // end of AXSetDeviceMode
/*******************************************************************************
 * Function:    AXGetDeviceMode
 * Arguments:
 *              device  :    (in) device type
 *              *mode   :    (in) pointer to variable to get mode
 *  Sets the mode of operation for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXGetDeviceMode(AXPBDeviceType device, u32 *mode)
{

    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;

    if(device < AX_MAX_NUM_DEVICES)
    {
        *mode = __AXDeviceMode[device];
    }
    else
    {
        //ASSERT(0);
        retErrCode = AXPB_ERROR_DEVICE_TYPE;
    }


    return retErrCode;
} // end of AXGetDeviceMode
/*******************************************************************************
 * Function:    AXGetDeviceChannelCount
 * Arguments:
 *              device  :    (in) device type
 *              Id      :    (in) Id of the device type
 *              *count  :    (in) pointer to variable to return count
 *  Gets the number of active channels for the device.
*******************************************************************************/
AXPB_ERROR_CODE AXGetDeviceChannelCount(AXPBDeviceType device, u32 Id, u32 *count)
{

    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;

    if(device < AX_MAX_NUM_DEVICES)
    {
        switch (device)
        {
        case AX_DEVICE_TV:
            if (Id < AX_MAX_NUM_TVS)
            {
                *count = __AXTVActiveChannels[Id];
            }
            else
            {
                //ASSERT(0);
                retErrCode = AXPB_ERROR_DEVICE_ID;
            }
            break;

        case AX_DEVICE_DRC:
            if (Id < AX_MAX_NUM_DRCS)
            {
                *count = __AXDRCActiveChannels[Id];
            }
            else
            {
                //ASSERT(0);
                retErrCode = AXPB_ERROR_DEVICE_ID;
            }
            break;
        }
    }
    else
    {
        //ASSERT(0);
        retErrCode = AXPB_ERROR_DEVICE_TYPE;
    }


    return retErrCode;
} // end of AXGetDeviceMode

} // namespace winext
} // namespace internal
} // namespace nw
