﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string.h>
#include <winext/cafe/os.h>
#include <winext/cafe/os/OSInterrupt.h>
#include "AXPrivate.h"
#include "AXHelper.h"
#if defined(_WIN32)
#include <Windows.h>
#endif


namespace nw {
namespace internal {
namespace winext {

static u32              __AXAuxDspWritePosition;
static u32              __AXAuxDspReadPosition;
static u32              __AXAuxCpuReadWritePosition;

// v2.0
ALIGNED_VAR(static s32, PPC_IO_BUFFER_ALIGN, __AXBufferTVAux[AX_MAX_NUM_TVS][3][AX_AUX_ID_MAX_NUM][AX_MAX_NUM_TV_CHS][AX_IN_SAMPLES_PER_FRAME]);
ALIGNED_VAR(static s32, PPC_IO_BUFFER_ALIGN, __AXBufferDRCAux[AX_MAX_NUM_DRCS][3][AX_AUX_ID_MAX_NUM][AX_MAX_NUM_DRC_CHS][AX_IN_SAMPLES_PER_FRAME]);

static AXUserAuxCallback    __AXTVAuxCallback[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM];
static void             *__AXTVAuxContext[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM];
static u8               __AXTVClearAux[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM][3];
static AXUserAuxCallback    __AXDRCAuxCallback[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM];
static void             *__AXDRCAuxContext[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM];
static u8               __AXDRCClearAux[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM][3];
static AXUserAuxCallback    __AXRmtAuxCallback[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM];
static void             *__AXRmtAuxContext[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM];
static u8               __AXRmtClearAux[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM][3];

static AXAuxCallback    __AXOldTVAuxCallback[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM];
static void             *__AXOldTVAuxContext[AX_MAX_NUM_TVS][AX_AUX_ID_MAX_NUM];

static AXAuxCallback    __AXOldDRCAuxCallback[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM];
static void             *__AXOldDRCAuxContext[AX_MAX_NUM_DRCS][AX_AUX_ID_MAX_NUM];

static AXAuxCallback    __AXOldRmtAuxCallback[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM];
static void             *__AXOldRmtAuxContext[AX_MAX_NUM_RMTS][AX_AUX_ID_MAX_NUM];


void __AXClearTVAuxBuffers(void )
{
    memset(__AXBufferTVAux,    0, sizeof(__AXBufferTVAux));
 }

void __AXClearDRCAuxBuffers(void )
{
   memset(__AXBufferDRCAux,   0, sizeof(__AXBufferDRCAux));
}
/*---------------------------------------------------------------------------*
    Initialize this code module.
 *---------------------------------------------------------------------------*/
void __AXAuxInit(void)
{
#ifdef _DEBUG
    OSReport("Initializing AXAux code module\n");
#endif

    __AXAuxDspWritePosition     = 0;
    __AXAuxDspReadPosition      = 1;
    __AXAuxCpuReadWritePosition = 2;


    __AXClearTVAuxBuffers();
    memset(__AXTVAuxCallback,  0, sizeof(__AXTVAuxCallback));
    memset(__AXTVAuxContext,   0, sizeof(__AXTVAuxContext));
    memset(__AXOldTVAuxCallback,  0, sizeof(__AXOldTVAuxCallback));
    memset(__AXOldTVAuxContext,   0, sizeof(__AXOldTVAuxContext));
    memset(__AXTVClearAux,     0, sizeof(__AXTVClearAux));

    __AXClearDRCAuxBuffers();
    memset(__AXDRCClearAux,    0, sizeof(__AXDRCClearAux));
    memset(__AXDRCAuxCallback, 0, sizeof(__AXDRCAuxCallback));
    memset(__AXDRCAuxContext,  0, sizeof(__AXDRCAuxContext));
    memset(__AXOldDRCAuxCallback, 0, sizeof(__AXOldDRCAuxCallback));
    memset(__AXOldDRCAuxContext,  0, sizeof(__AXOldDRCAuxContext));
}

/*---------------------------------------------------------------------------*
    Shutdown this code module.
 *---------------------------------------------------------------------------*/
void __AXAuxQuit(void)
{
#ifdef _DEBUG
    OSReport("Shutting down AXAux code module\n");
#endif

    u16 deviceCnt, auxCnt;

    for(deviceCnt = 0; deviceCnt < AX_MAX_NUM_TVS ; deviceCnt++)
    {
        for(auxCnt = 0; auxCnt < AX_AUX_ID_MAX_NUM ; auxCnt++)
        {
            __AXTVAuxCallback[deviceCnt][auxCnt] = NULL;
        } // auxCnt
    } // end of device


    for(deviceCnt = 0; deviceCnt < AX_MAX_NUM_DRCS ; deviceCnt++)
    {
        for(auxCnt = 0; auxCnt < AX_AUX_ID_MAX_NUM ; auxCnt++)
        {
            __AXDRCAuxCallback[deviceCnt][auxCnt] = NULL;
        } // auxCnt
    } // end of device


    for(deviceCnt = 0; deviceCnt < AX_MAX_NUM_RMTS ; deviceCnt++)
    {
        for(auxCnt = 0; auxCnt < AX_AUX_ID_MAX_NUM ; auxCnt++)
        {
            __AXRmtAuxCallback[deviceCnt][auxCnt] = NULL;
        } // auxCnt
    } // end of device



} // __AXAuxQuit


/*---------------------------------------------------------------------------*
    Pass caller pointer to an AUX send. If there's no registered
    callback pass back 0 so the caller knows there's no AUX effect.
 *---------------------------------------------------------------------------*/
void __AXGetAuxInput(u32 device, u32 id, u32 aux_id, u32 *p)
{
    u32 data = 0;

    switch (device)
    {
    case AX_DEVICE_TV:
        if (__AXOldTVAuxCallback[id][aux_id] || __AXTVAuxCallback[id][aux_id])
        {
            data = (u32)&__AXBufferTVAux[id][__AXAuxDspWritePosition][aux_id][0][0];
        }
        break;

    case AX_DEVICE_DRC:
        if (__AXOldDRCAuxCallback[id][aux_id] || __AXDRCAuxCallback[id][aux_id])
        {
            data = (u32)&__AXBufferDRCAux[id][__AXAuxDspWritePosition][aux_id][0][0];
        }
        break;
    }

    *p = data;
}
/*---------------------------------------------------------------------------*
    Pass caller pointer to an AUX return.
 *---------------------------------------------------------------------------*/
void __AXGetAuxOutput(u32 device, u32 id, u32 aux_id, u32 *p)
{
    u32 data = 0;

    switch (device)
    {
    case AX_DEVICE_TV:
        if (__AXOldTVAuxCallback[id][aux_id] || __AXTVAuxCallback[id][aux_id])
        {
            data = (u32)&__AXBufferTVAux[id][__AXAuxDspReadPosition][aux_id][0][0];
        }
        break;

    case AX_DEVICE_DRC:
        if (__AXOldDRCAuxCallback[id][aux_id] || __AXDRCAuxCallback[id][aux_id])
        {
            data = (u32)&__AXBufferDRCAux[id][__AXAuxDspReadPosition][aux_id][0][0];
        }
        break;
    }

    *p = data;
}

// -----------------------------------------------------------------------------
s32* __AXGetAuxBus(AXPBDeviceType device, u32 deviceId, u32 auxId, u32 ch)
{
    switch (device)
    {
        case AX_DEVICE_TV:
            return __AXBufferTVAux[0][__AXAuxCpuReadWritePosition][auxId][ch];
        case AX_DEVICE_DRC:
            return __AXBufferDRCAux[deviceId][__AXAuxCpuReadWritePosition][auxId][ch];
        default:
            ASSERT(FALSE);
            return NULL;
            break;
    }
}

// -----------------------------------------------------------------------------
void __AXClearAuxBus()
{
    memset(__AXBufferTVAux, 0, sizeof(__AXBufferTVAux));
    memset(__AXBufferDRCAux, 0, sizeof(__AXBufferDRCAux));
}

/*---------------------------------------------------------------------------*
    Process AUX bus, call user callbacks, take care of cache coherency,
    push and mod send and return positions.
 *---------------------------------------------------------------------------*/
void __AXProcessAux(void)
{
    s32*            ptr;
    u32             len;
    u32             drc_id;
    u32             aux_id;
    u32             num_chans;
    s32 *auxData[AX_MAX_NUM_TV_CHS];
    AXAUXCBSTRUCT auxCbStruct;

    // Process TV Aux Busses
    // logic:
    //      - get the number of channels per device
    //      - if a callback with the new API is set, service it,
    //      - else, if a callback with old API is set, service it.
    //      - else, clear out the buffer and get out
    AXGetDeviceChannelCount(AX_DEVICE_TV, AX_TV_ID0, &num_chans);
    for (aux_id = AX_AUX_ID_A; aux_id < AX_AUX_ID_MAX_NUM; aux_id++)
    {
        // call the callback if any
        if (__AXTVAuxCallback[AX_TV_ID0][aux_id] || __AXOldTVAuxCallback[AX_TV_ID0][aux_id] )
        {
            for(u32 chCnt=0;chCnt< AX_MAX_NUM_TV_CHS ; chCnt++)
            {
                auxData[chCnt] = &__AXBufferTVAux[AX_TV_ID0][__AXAuxCpuReadWritePosition][aux_id][chCnt][0];
            }

            auxCbStruct.numChs = num_chans;
            auxCbStruct.numSamples = AX_IN_SAMPLES_PER_FRAME;

            // the following 4 lines are for zeroing out the third channel expected by axfx
            if(2 == num_chans)
            {
                memset(auxData[2],0, AX_IN_SAMPLES_PER_FRAME*4);
            }
#if 0
            if (!PPC_ONLY)
            {
                DCInvalidateRange(auxData[AX_CH_LEFT], AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * num_chans);

                if(__AXTVAuxCallback[AX_TV_ID0][aux_id])
                {
                   (*__AXTVAuxCallback[AX_TV_ID0][aux_id])(auxData, __AXTVAuxContext[AX_TV_ID0][aux_id], &auxCbStruct);
                }
                else
                {
                    (*__AXOldTVAuxCallback[AX_TV_ID0][aux_id])((void *)auxData, (void*) __AXOldTVAuxContext[AX_TV_ID0][aux_id]);
                }

                DCFlushRangeNoSync(auxData[AX_CH_LEFT], AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * num_chans);
            }
            else
#endif
            {
                if(__AXTVAuxCallback[AX_TV_ID0][aux_id])
                {
                   (*__AXTVAuxCallback[AX_TV_ID0][aux_id])(auxData, __AXTVAuxContext[AX_TV_ID0][aux_id], &auxCbStruct);
                }
                else
                {
                    (*__AXOldTVAuxCallback[AX_TV_ID0][aux_id])((void *)auxData, (void*) __AXOldTVAuxContext[AX_TV_ID0][aux_id]);
                }

            }
        } // if cbs set
        else
        {
            // if cb is not set, clear out the buffer. this is required because, when the user starts
            // aux callback in the middle of application playing sound, we want the first buffer sent to
            // the aux callback to be clear and not from previous sound that was playing
            if(__AXTVClearAux[AX_TV_ID0][aux_id][__AXAuxCpuReadWritePosition])
            {
                ptr = &__AXBufferTVAux[AX_TV_ID0][__AXAuxCpuReadWritePosition][aux_id][AX_CH_LEFT][0];
                len = AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * AX_MAX_NUM_TV_CHS;
                memset(ptr, 0, len);
//                DCFlushRange(ptr, len);
                __AXTVClearAux[AX_TV_ID0][aux_id][__AXAuxCpuReadWritePosition] = 0;

            }
        }
    }
    // Process DRC Aux Busses
    for (drc_id = AX_DRC_ID0; drc_id < AX_MAX_NUM_DRCS; drc_id++)
    {
        AXGetDeviceChannelCount(AX_DEVICE_DRC, AX_DRC_ID0, &num_chans);
        for (aux_id = AX_AUX_ID_A; aux_id < AX_AUX_ID_MAX_NUM; aux_id++)
        {
            // call the callback if any
            if (__AXDRCAuxCallback[drc_id][aux_id] || __AXOldDRCAuxCallback[drc_id][aux_id])
            {

                for(u32 chCnt=0;chCnt< AX_MAX_NUM_DRC_CHS ; chCnt++)
                {
                    auxData[chCnt] = &__AXBufferDRCAux[drc_id][__AXAuxCpuReadWritePosition][aux_id][chCnt][0];
                }

                auxCbStruct.numChs = num_chans;
                auxCbStruct.numSamples = AX_IN_SAMPLES_PER_FRAME;

                // the following 4 lines are for zeroing out the third channel expected by axfx
                if(2 == num_chans)
                {
                    memset(auxData[2],0, AX_IN_SAMPLES_PER_FRAME*4);
                }

#if 0
                if (!PPC_ONLY)
                {
                    // potential issue here when we move to 32k/48k, samples might not be sequential
                    DCInvalidateRange(auxData[AX_CH_LEFT], AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * num_chans);

                    if(__AXDRCAuxCallback[drc_id][aux_id])
                    {
                        (*__AXDRCAuxCallback[drc_id][aux_id])(auxData, __AXDRCAuxContext[drc_id][aux_id], &auxCbStruct);
                    }
                    else
                    {
                        (*__AXOldDRCAuxCallback[drc_id][aux_id])((void *)auxData, (void*) __AXOldDRCAuxContext[drc_id][aux_id]);
                    }


                    DCFlushRangeNoSync(auxData[AX_CH_LEFT], AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * num_chans);
                }
                else
#endif
                {
                    if(__AXDRCAuxCallback[drc_id][aux_id])
                    {
                        (*__AXDRCAuxCallback[drc_id][aux_id])(auxData, __AXDRCAuxContext[drc_id][aux_id], &auxCbStruct);
                    }
                    else
                    {
                        (*__AXOldDRCAuxCallback[drc_id][aux_id])((void *)auxData, (void*) __AXOldDRCAuxContext[drc_id][aux_id]);
                    }

                }
            }
            else
            {
                // if cb is not set, clear out the buffer. this is required because, when the user starts
                // aux callback in the middle of application playing sound, we want the first buffer sent to
                // the aux callback to be clear and not from previous sound that was playing
                if(__AXDRCClearAux[drc_id][aux_id][__AXAuxCpuReadWritePosition])
                {
                    ptr = &__AXBufferDRCAux[drc_id][__AXAuxCpuReadWritePosition][aux_id][AX_CH_LEFT][0];
                    len = AX_IN_SAMPLES_PER_FRAME * sizeof(s32) * AX_MAX_NUM_DRC_CHS;
                    memset(ptr, 0, len);
//                    DCFlushRange(ptr, len);
                    __AXDRCClearAux[drc_id][aux_id][__AXAuxCpuReadWritePosition] = 0;

                }
            } // end of if (cbset)
        } // end of for all DRCS
    }
    // push the buffer positions
    __AXAuxDspWritePosition     ++;
    __AXAuxDspWritePosition     %= 3;
    __AXAuxDspReadPosition      ++;
    __AXAuxDspReadPosition      %= 3;
    __AXAuxCpuReadWritePosition ++;
    __AXAuxCpuReadWritePosition %= 3;
}

/*---------------------------------------------------------------------------*
    exposed API functions
 *---------------------------------------------------------------------------*/


/*---------------------------------------------------------------------------*
    User API to register AUX A callback.
 *---------------------------------------------------------------------------*/
void AXRegisterAuxACallback(AXAuxCallback callback, void *context)
{
    BOOL old = OSDisableInterrupts();

    __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_A]    = callback;
    __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_A]     = context;

    if (!callback)
    {
        memset(__AXTVClearAux[AX_TV_ID0][AX_AUX_ID_A], 1, 3);
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to register AUX B callback.
 *---------------------------------------------------------------------------*/
void AXRegisterAuxBCallback(AXAuxCallback callback, void *context)
{
    BOOL old = OSDisableInterrupts();

    __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_B]    = callback;
    __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_B]     = context;

    if (!callback)
    {
        memset(__AXTVClearAux[AX_TV_ID0][AX_AUX_ID_B], 1, 3);
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to register AUX C callback.
 *---------------------------------------------------------------------------*/
void AXRegisterAuxCCallback(AXAuxCallback callback, void *context)
{
    BOOL old = OSDisableInterrupts();

    __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_C]    = callback;
    __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_C]     = context;

    if (!callback)
    {
        memset(__AXTVClearAux[AX_TV_ID0][AX_AUX_ID_C], 1, 3);
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to register DRC AUX callback.
 *---------------------------------------------------------------------------*/
void AXRegisterAuxDRCCallback(AXAuxCallback callback, void *context)
{
    BOOL old = OSDisableInterrupts();

    __AXOldDRCAuxCallback[AX_DRC_ID0][AX_AUX_ID_A]    = callback;
    __AXOldDRCAuxContext[AX_DRC_ID0][AX_AUX_ID_A]     = context;

    if (!callback)
    {
        memset(__AXDRCClearAux[AX_DRC_ID0][AX_AUX_ID_A], 1, 3);
    }

    OSRestoreInterrupts(old);
}

/*---------------------------------------------------------------------------*
    User API to get AUX A callback.
 *---------------------------------------------------------------------------*/
void AXGetAuxACallback(AXAuxCallback *callback, void **context)
{
    *callback = __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_A];
    *context  = __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_A];
}

/*---------------------------------------------------------------------------*
    User API to get AUX B callback.
 *---------------------------------------------------------------------------*/
void AXGetAuxBCallback(AXAuxCallback *callback, void **context)
{
    *callback = __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_B];
    *context  = __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_B];
}

/*---------------------------------------------------------------------------*
    User API to get AUX C callback.
 *---------------------------------------------------------------------------*/
void AXGetAuxCCallback(AXAuxCallback *callback, void **context)
{
    *callback = __AXOldTVAuxCallback[AX_TV_ID0][AX_AUX_ID_C];
    *context  = __AXOldTVAuxContext[AX_TV_ID0][AX_AUX_ID_C];
}
/*---------------------------------------------------------------------------*
    User API to get DRC AUX callback.
 *---------------------------------------------------------------------------*/
void AXGetAuxDRCCallback(AXAuxCallback *callback, void **context)
{
    *callback = __AXOldDRCAuxCallback[AX_DRC_ID0][AX_AUX_ID_A];
    *context  = __AXOldDRCAuxContext[AX_DRC_ID0][AX_AUX_ID_A];
}
/*******************************************************************************
 * Function AXRegisterAuxCallback
 * Arguments
 *          device      :    (in) device to which callback has to be registered
 *          Id          :    (in) Id of the device type to which the callback has to be registered
 *          auxId       :    (in) Id of the aux to which the callback has to be registered
 *          fcb         :    (in) callback function to register
 *          *context    :    (in) context for this callback
 * Registers the callback for aux specified for the particular device
*******************************************************************************/
AXPB_ERROR_CODE AXRegisterAuxCallback(AXPBDeviceType device, u32 Id, u32 auxId, AXUserAuxCallback fcb, void *context)
{

    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    retErrCode = __AXCheckDeviceArgs(device, Id);

    if(AXPB_ERROR_NONE == retErrCode)
    {
        //ASSERT(auxId < AX_AUX_ID_MAX_NUM );
        if(auxId >= AX_AUX_ID_MAX_NUM)
        {
            return AXPB_ERROR_AUX_ID;
        }
        switch(device)
        {
            case AX_DEVICE_TV:
                __AXTVAuxCallback[Id][auxId] = fcb;
                __AXTVAuxContext[Id][auxId] = context;
                if (!fcb)
                {
                    memset(__AXTVClearAux[Id][auxId], 1, 3);
                }
                break;
            case AX_DEVICE_DRC:
                __AXDRCAuxCallback[Id][auxId] = fcb;
                __AXDRCAuxContext[Id][auxId] = context;
                if (!fcb)
                {
                    memset(__AXDRCClearAux[Id][auxId], 1, 3);
                }
                break;
            case AX_DEVICE_RMT:
                __AXRmtAuxCallback[Id][auxId] = fcb;
                __AXRmtAuxContext[Id][auxId] = context;
                if (!fcb)
                {
                    memset(__AXRmtClearAux[Id][auxId], 1, 3);
                }
                break;
        } // end of switch
    }

    return retErrCode;
} // end of AXRegisterAuxCallback
/*******************************************************************************
 * Function AXGetAuxCallback
 * Arguments
 *          device      :    (in) device to which callback has to be registered
 *          Id          :    (in) Id of the device type to which the callback has to be registered
 *          auxId       :    (in) Id of the aux to which the callback has to be registered
 *          *fcb        :    (in) pointer to variable to return the callback function
 *          **context   :    (in) pointer to variable to return the context
 * gets the callback for aux specified for the particular device
*******************************************************************************/
AXPB_ERROR_CODE AXGetAuxCallback(AXPBDeviceType device, u32 Id, u32 auxId, AXUserAuxCallback *fcb, void **context)
{

    AXPB_ERROR_CODE retErrCode = AXPB_ERROR_NONE;
    retErrCode = __AXCheckDeviceArgs(device, Id);

    if(AXPB_ERROR_NONE == retErrCode)
    {
        //ASSERT(auxId < AX_AUX_ID_MAX_NUM );
        if(auxId >= AX_AUX_ID_MAX_NUM)
        {
            return AXPB_ERROR_AUX_ID;
        }
        switch(device)
        {
            case AX_DEVICE_TV:
                *fcb = __AXTVAuxCallback[Id][auxId];
                *context = __AXTVAuxContext[Id][auxId];
                break;
            case AX_DEVICE_DRC:
                *fcb = __AXDRCAuxCallback[Id][auxId];
                *context = __AXDRCAuxContext[Id][auxId];
                break;
            case AX_DEVICE_RMT:
                *fcb = __AXRmtAuxCallback[Id][auxId];
                *context = __AXRmtAuxContext[Id][auxId];
                break;
        } // end of switch
    }

    return retErrCode;
} // end of AXGetAuxCallback

} // namespace winext
} // namespace internal
} // namespace nw
