﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <winext/cafe/os.h>
#include <winext/cafe/os/OSInterrupt.h>
#include <winext/cafe/ax.h>
#include "AXPrivate.h"
#if defined(__APPLE__)
#include <TargetConditionals.h>
#endif

#if defined(ANDROID)
#include "../ai/SoundEngineSLES.h"
#elif TARGET_OS_IPHONE
#include "../ai/SoundEngineCoreAudio.h"
#else
#include "../ai/SoundEngineMME.h"
#endif

// #define DEBUG_DUMP
#ifdef DEBUG_DUMP
#include <cstdio>

namespace
{
    std::FILE* s_pFile;
    const char* DUMP_FILE = "_ax.dump";
    short s_PreX = 0;
}
#endif

namespace nw {
namespace internal {
namespace winext {

#if defined(ANDROID)
    CSoundEngineSLES* g_pSoundEngine = CSoundEngineSLES::Instance();
#elif TARGET_OS_IPHONE
    CSoundEngineCoreAudio* g_pSoundEngine = CSoundEngineCoreAudio::Instance();
#else
    CSoundEngineMME* g_pSoundEngine = CSoundEngineMME::Instance();
#endif

extern void __AXWaveOutCallbackFunc(
    int channels,
    signed short* buffer,
    unsigned long samples,
    int sampleRate
);

extern void (*__AINextCallback)(
    int channels,
    signed short* buffer,
    unsigned long samples,
    int sampleRate);

#define DMA_BUFFER_SAMPLES         8192
#define DMA_BUFFER_BYTE_PER_SAMPLE (sizeof(s16))
#define DMA_BUFFER_CHANNEL_COUNT   2
#define DMA_BUFFER_LENGTH          (DMA_BUFFER_SAMPLES * DMA_BUFFER_BYTE_PER_SAMPLE * DMA_BUFFER_CHANNEL_COUNT)
#define DMA_BUFFER_COUNT           2    // ダブルバッファ

static BOOL  __init = FALSE;
static s32 sCompressorState = 0;

static u8 sDMABuffer[DMA_BUFFER_COUNT][DMA_BUFFER_LENGTH];
static s32 sDMABufferCurrentIndex = 0;

// TODO: void compressor(s32* comp_state, s32* lch, s32* rch);

static unsigned long GetDMABufferLength()
{
    return g_pSoundEngine->GetBufferLength();
}

static void AiDmaCallbackFunc()
{
    sDMABufferCurrentIndex++;

    if(sDMABufferCurrentIndex >= DMA_BUFFER_COUNT)
    {
        sDMABufferCurrentIndex = 0;
    }

    AIInitDMA(reinterpret_cast<u32>(sDMABuffer[sDMABufferCurrentIndex]), GetDMABufferLength());
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXInit(void)
{
    AXInitEx(AX_OUTPUT_BUFFER_DOUBLE);
#ifdef DEBUG_DUMP
    s_pFile = std::fopen( DUMP_FILE, "wb" );
#endif
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXInitEx(u32 outputBufferMode)
{
#ifdef _DEBUG
    OSReport("Initializing AX\n");
#endif

    if (__init)
    {
        return;
    }

//    OSRegisterVersion(__AXVersion);

    //ASSERT(GetDMABufferLength() < DMA_BUFFER_LENGTH);

    // get the interface driver started
    AIInit(NULL);
    AISetDSPSampleRate(AI_SAMPLERATE_32KHZ);

    __AXAllocInit();
    __AXVPBInit();
    __AXSPBInit();
    __AXAuxInit();
    __AXClInit();

    __AXOutInit(outputBufferMode);
    AIInitDMA(reinterpret_cast<u32>(sDMABuffer[0]), GetDMABufferLength());
    AIRegisterDMACallback(AiDmaCallbackFunc);


   __AINextCallback = g_pSoundEngine->SetWaveOutCallback(__AXWaveOutCallbackFunc);

    __init = TRUE;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXInitSpecifyMem(u32 num, void* mem)
{
    AXInitExSpecifyMem(num, mem, AX_OUTPUT_BUFFER_DOUBLE);
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXInitExSpecifyMem(u32 num, void* mem, u32 outputBufferMode)
{
#ifdef _DEBUG
    OSReport("Initializing AX\n");
#endif

    if (__init)
    {
        return;
    }

//    OSRegisterVersion(__AXVersion);

    // get the interface driver started
    AIInit(NULL);
    AISetDSPSampleRate(AI_SAMPLERATE_32KHZ);

    __AXAllocInit();
    __AXVPBInitSpecifyMem(num, mem);
    __AXSPBInit();
    __AXAuxInit();
    __AXClInit();

    __AXOutInit(outputBufferMode);
    AIInitDMA(reinterpret_cast<u32>(sDMABuffer[0]), GetDMABufferLength());
    AIRegisterDMACallback(AiDmaCallbackFunc);

    __init = TRUE;
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
void AXQuit(void)
{
#ifdef _DEBUG
    OSReport("Shutting down AX\n");
#endif

    if (!__init)
    {
        return;
    }

    g_pSoundEngine->SetWaveOutCallback(
        NULL
    );

    AIReset();

//    __AXOutQuit();

    __AXAllocQuit();
    __AXVPBQuit();
    __AXSPBQuit();
    __AXAuxQuit();
    __AXClQuit();

    g_pSoundEngine->CloseStream();
    g_pSoundEngine->DestroyInstance();
    g_pSoundEngine = nullptr;

    __init = FALSE;

#ifdef DEBUG_DUMP
    std::fclose( s_pFile );
#endif
}

/*---------------------------------------------------------------------------*
 *---------------------------------------------------------------------------*/
BOOL AXIsInit(void)
{
    return __init;
}


BOOL AXUserIsProtected()
{
    // HACK : 空実装です。
    return false;
}
BOOL __AXVoiceIsProtected(AXVPB* /* p */)
{
    return false;
}
void __AXVoiceProtect(AXVPB * /* p */)
{
}

void __AXMixInit(u32 /* mix_mode */)
{
}

void __AXSynthesize(
    int channels,
    signed short* buffer,
    unsigned long samples,
    int sampleRate
)
{
    __AXWaveOutCallbackFunc(channels,buffer,samples,sampleRate);
}

void AXSetVoiceDRCMix( AXVPB*, AXPBDRCMIX* )
{
}

#define SAMPLES_PER_FRAME               14 // samples
#define BYTES_PER_FRAME                  8 // byte

s32 AXDecodeAdpcmData(const u8* input, DSPADPCM* info, s32 samples, s16* output)
{

    u32  start_addr;
    u32  end_addr;
    u32  curr_addr;
    u8*  read_ptr;
    u16  pred;
    u16  scale;
    s16  yn1;
    s16  yn2;
    s16  a1;
    s16  a2;
    s16  gain;
    u8   byte;
    s16  nibble;
    s32  val;
    s32  yields;
    s16* outptr;
    s32  frames;
    s32  lefts;
    s32  nibbles;

    //
    // 引数のチェック
    //

    //ASSERT(input && output && info);
    //ASSERT(samples >= 0);

    if (!input || !output || !info || samples <= 0)
    {
        return 0;
    }

    //
    // DSPADPCM 情報のチェック
    //

    //ASSERT(info->num_samples);
    //ASSERT(info->loop_flag <= AXPBADDR_LOOP_ON);
    //ASSERT(info->format == AX_PB_FORMAT_ADPCM);
    //ASSERT(info->sa <= info->ea);
    //ASSERT(info->ca <= info->ea);

    frames  = (s32)info->num_samples / SAMPLES_PER_FRAME;
    lefts   = (s32)info->num_samples % SAMPLES_PER_FRAME;
    nibbles = frames * BYTES_PER_FRAME * 2; // bytes -> nibbles
    if (lefts)
    {
        nibbles += lefts;
        nibbles += 2; // ADPCM frame header
    }

    //ASSERT(info->num_adpcm_nibbles == nibbles);

    if (!info->num_samples ||
        info->num_adpcm_nibbles != nibbles ||
        info->loop_flag > AXPBADDR_LOOP_ON ||
        info->format != AX_PB_FORMAT_ADPCM ||
        info->sa > info->ea ||
        info->ca > info->ea)
    {
        return 0;
    }

    start_addr = info->sa;
    end_addr   = info->ea;
    curr_addr  = info->ca;

    pred  = (u16)(info->ps >> 4);
    scale = (u16)(info->ps & 0xf);
    yn1   = (s16)(info->yn1);
    yn2   = (s16)(info->yn2);

    //
    // ADPCM デコード
    //

    a1   = (s16)info->coef[2 * pred + 0];
    a2   = (s16)info->coef[2 * pred + 1];
    gain = (s16)(1 << scale);

    read_ptr = ((u8*)input) + (curr_addr >> 1);

    if (curr_addr & 0x1)
    {
        byte = *read_ptr++;
    }

    yields = 0;
    outptr = output;

    for (;;)
    {
        // pred & scale 取得
        if (!(curr_addr & 0xf))
        {
            byte = *read_ptr++;

            pred  = (u16)(byte >> 4);
            scale = (u16)(byte & 0xf);

            a1   = (s16)info->coef[2 * pred + 0];
            a2   = (s16)info->coef[2 * pred + 1];
            gain = (s16)(1 << scale);

            curr_addr += 2;
        }

        // ニブルデータ取得
        if (!(curr_addr & 0x1))
        {
            byte = *read_ptr++;

            nibble = (s16)(byte >> 4);
        }
        else
        {
            nibble = (s16)(byte & 0xf);
        }

        curr_addr++;

        // デコード
        nibble <<= 12;
        nibble >>= 1;

        val   = a1 * yn1;
        val  += a2 * yn2;
        val  += gain * nibble;
        val >>= 10;
        val  += 1;
        val >>= 1;

        if (val > 32767)
        {
            val = 32767;
        }
        else if (val < -32768)
        {
            val = -32768;
        }

        // 出力
        *outptr++ = (s16)val;

        yields++;
        samples--;

        yn2 = yn1;
        yn1 = (s16)val;

        // 終了判定
        //  - 要求数に達したら。
        //  - ループ無しの場合に、エンドアドレスに達したら。
        if (samples <= 0 || (!info->loop_flag && curr_addr > end_addr))
        {
            break;
        }

        // アドレス更新
        if (curr_addr > end_addr)
        {
            curr_addr = start_addr;

            pred  = (u16)(info->lps >> 4);
            scale = (u16)(info->lps & 0xf);
            yn1   = (s16)(info->lyn1);
            yn2   = (s16)(info->lyn2);

            a1   = (s16)info->coef[2 * pred + 0];
            a2   = (s16)info->coef[2 * pred + 1];
            gain = (s16)(1 << scale);

            read_ptr = ((u8*)input) + (curr_addr >> 1);

            if (curr_addr & 0x1)
            {
                byte = *read_ptr++;
            }
        }
    }

//    DCFlushRange(output, (u32)(yields * sizeof(s16)));

    return yields;
}

// -----------------------------------------------------------------------------
u32 AXGetInputSamplesPerFrame()
{
    // 32KHz レンダラのみサポート
    return AX_IN_SAMPLES_PER_FRAME; // 96
}

// -----------------------------------------------------------------------------
u32 AXGetInputSamplesPerSec()
{
    // 32KHz レンダラのみサポート
    return AX_IN_SAMPLES_PER_SEC;   // 32000
}

} // namespace winext
} // namespace internal
} // namespace nw
