﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winext/types.h>
#include "ai_dma.h"

// Android headers
#include <SLES/OpenSLES.h>
#include <SLES/OpenSLES_Android.h>
#include <android/log.h>

// Linux headers
#include <assert.h>
#include <stdlib.h>
#include <pthread.h>

// C++11 headers
#include <atomic>

#include "SoundEngineSLES.h"
using namespace nw::internal::winext;

// Defines
#define NUM_BQ 8
#define OSReport(...) do{}while(0);

// Local Variables
static SLObjectItf sEngineObject = NULL;
static SLEngineItf sEngineEngine;
static SLObjectItf sOutputMixObject = NULL;
static SLObjectItf sBqPlayerObject = NULL;
static SLPlayItf sBqPlayerPlay;
static SLVolumeItf sBqPlayerVolume;
static SLAndroidSimpleBufferQueueItf sBqPlayerBufferQueue;

static pthread_mutex_t sMutex = PTHREAD_MUTEX_INITIALIZER;
static s32 sNumEnqueueingBuffer;
static s32 sNumEnqueuedBuffer;


static void bqPlayerCallback(SLAndroidSimpleBufferQueueItf bq, void *context)
{
#if 0
    // バッファキューの状態をログ出力
    {
        SLAndroidSimpleBufferQueueState state;
        SLresult result = (*sBqPlayerBufferQueue)->GetState(sBqPlayerBufferQueue, &state);
        assert(SL_RESULT_SUCCESS == result);
        (void)result;
        {
            __android_log_print(ANDROID_LOG_INFO, "AIDMA", "[%s:%05d] count(%d), index(%d)", __func__, __LINE__, state.count, state.index);
        }
    }
#endif

    // 波形生成を要求する
    CSoundEngineSLES::Instance()->RequestRender();

}

void AIDMA_Init(u32 sampleRate)
{
    assert(sampleRate == 48000);

    // setup SLES Resources
    SLresult result;

    // -- engine --
    // create engine
    result = slCreateEngine(&sEngineObject, 0, NULL, 0, NULL, NULL);
    assert(SL_RESULT_SUCCESS == result);

    // realize the engine
    result = (*sEngineObject)->Realize(sEngineObject, SL_BOOLEAN_FALSE);
    assert(SL_RESULT_SUCCESS == result);

    // get the engine interface, which is needed in order to create other objects
    result = (*sEngineObject)->GetInterface(sEngineObject, SL_IID_ENGINE, &sEngineEngine);
    assert(SL_RESULT_SUCCESS == result);

    // create output mix, with environmental reverb specified as a non-required interface
    result = (*sEngineEngine)->CreateOutputMix(sEngineEngine, &sOutputMixObject, 0, NULL, NULL);
    assert(SL_RESULT_SUCCESS == result);

    // realize the output mix
    result = (*sOutputMixObject)->Realize(sOutputMixObject, SL_BOOLEAN_FALSE);
    assert(SL_RESULT_SUCCESS == result);

    // -- player --
    // configure audio source
    SLDataLocator_AndroidSimpleBufferQueue loc_bufq = { SL_DATALOCATOR_ANDROIDSIMPLEBUFFERQUEUE, NUM_BQ };

    SLDataFormat_PCM format_pcm_48k = { SL_DATAFORMAT_PCM, 2, SL_SAMPLINGRATE_48,
        SL_PCMSAMPLEFORMAT_FIXED_16, SL_PCMSAMPLEFORMAT_FIXED_16,
        (SL_SPEAKER_FRONT_LEFT | SL_SPEAKER_FRONT_RIGHT), SL_BYTEORDER_LITTLEENDIAN };

    SLDataFormat_PCM format_pcm_32k = { SL_DATAFORMAT_PCM, 2, SL_SAMPLINGRATE_32,
        SL_PCMSAMPLEFORMAT_FIXED_16, SL_PCMSAMPLEFORMAT_FIXED_16,
        (SL_SPEAKER_FRONT_LEFT | SL_SPEAKER_FRONT_RIGHT), SL_BYTEORDER_LITTLEENDIAN };


    SLDataSource audioSrc = { &loc_bufq, &format_pcm_48k };
    // configure audio sink
    SLDataLocator_OutputMix loc_outmix = { SL_DATALOCATOR_OUTPUTMIX, sOutputMixObject };
    SLDataSink audioSnk = { &loc_outmix, NULL };

    // create audio player
    const SLInterfaceID playerIds[2] = { SL_IID_BUFFERQUEUE, SL_IID_VOLUME };
    const SLboolean playerReq[2] = { SL_BOOLEAN_TRUE, SL_BOOLEAN_TRUE };
    result = (*sEngineEngine)->CreateAudioPlayer(sEngineEngine, &sBqPlayerObject, &audioSrc, &audioSnk,
        2, playerIds, playerReq);
    assert(SL_RESULT_SUCCESS == result);
    (void)result;

    // realize the player
    result = (*sBqPlayerObject)->Realize(sBqPlayerObject, SL_BOOLEAN_FALSE);
    assert(SL_RESULT_SUCCESS == result);

    // get the play interface
    result = (*sBqPlayerObject)->GetInterface(sBqPlayerObject, SL_IID_PLAY, &sBqPlayerPlay);
    assert(SL_RESULT_SUCCESS == result);

    // get the buffer queue interface
    result = (*sBqPlayerObject)->GetInterface(sBqPlayerObject, SL_IID_BUFFERQUEUE,
        &sBqPlayerBufferQueue);
    assert(SL_RESULT_SUCCESS == result);

    // register callback on the buffer queue
    result = (*sBqPlayerBufferQueue)->RegisterCallback(sBqPlayerBufferQueue, bqPlayerCallback, NULL);
    assert(SL_RESULT_SUCCESS == result);

    // get the volume interface
    result = (*sBqPlayerObject)->GetInterface(sBqPlayerObject, SL_IID_VOLUME, &sBqPlayerVolume);
    assert(SL_RESULT_SUCCESS == result);

    // TODO: この場所で大丈夫かは後で確認したほうがいいかも (プレイフラグ戻るかも問題)
    result = (*sBqPlayerPlay)->SetPlayState(sBqPlayerPlay, SL_PLAYSTATE_PLAYING);
    assert(SL_RESULT_SUCCESS == result);
}

s32  AIDMA_GetBufferQueueCapacity()
{
    return NUM_BQ;
}

void AIDMA_ExecuteDMA(u8* addr, u32 size)
{
    SLresult result;

    result = (*sBqPlayerBufferQueue)->Enqueue(sBqPlayerBufferQueue, addr, size);
    assert(SL_RESULT_SUCCESS == result);
    (void)result;
}

void AIDMA_StopDMA()
{
    SLresult result;

    // Stop player
    result = (*sBqPlayerPlay)->SetPlayState(sBqPlayerPlay, SL_PLAYSTATE_STOPPED);
    assert(SL_RESULT_SUCCESS == result);
    (void)result;
}

void AIDMA_Quit()
{
    AIDMA_StopDMA();

    (*sBqPlayerObject)->Destroy(sBqPlayerObject);
    (*sOutputMixObject)->Destroy(sOutputMixObject);
    (*sEngineObject)->Destroy(sEngineObject);

    sBqPlayerObject     = NULL;
    sOutputMixObject    = NULL;
    sEngineObject       = NULL;
}

