﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/ut/ut_List.h>

#define OBJ_TO_LINK(list,obj)   ((Link*)(((u32)(obj))+(list)->offset))


namespace nw {
namespace ut {

//---------------------------------------------------------------------------
void
List_Init(List* list, u16 offset)
{
    NW_NULL_ASSERT(list);

    list->headObject = NULL;
    list->tailObject = NULL;
    list->numObjects = 0;
    list->offset     = offset;
}

//---------------------------------------------------------------------------
//! @brief        最初のオブジェクトをリストに加えます。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//! @param[out]   object  リストに繋げたいオブジェクトへのポインタ。
//---------------------------------------------------------------------------
static void
SetFirstObject(List* list, void* object)
{
    Link* link;

    NW_NULL_ASSERT(list  );
    NW_NULL_ASSERT(object);

    link = OBJ_TO_LINK(list, object);

    link->nextObject = NULL;
    link->prevObject = NULL;
    list->headObject = object;
    list->tailObject = object;
    list->numObjects++;
}

//---------------------------------------------------------------------------
void
List_Append(List* list, void* object)
{
    NW_NULL_ASSERT(list  );
    NW_NULL_ASSERT(object);

    if (list->headObject == NULL)
    {
        // リストが空の時。
        SetFirstObject(list, object);
    }
    else
    {
        Link* link = OBJ_TO_LINK(list, object);

        link->prevObject = list->tailObject;
        link->nextObject = NULL;

        OBJ_TO_LINK(list, list->tailObject)->nextObject = object;
        list->tailObject = object;
        list->numObjects++;
        NW_WARNING( list->numObjects != 0, "Overflow numObjects" );
    }
}


//---------------------------------------------------------------------------
void
List_Prepend(List* list, void* object)
{
    NW_NULL_ASSERT(list  );
    NW_NULL_ASSERT(object);

    if (list->headObject == NULL)
    {
        // リストが空の時。
        SetFirstObject(list, object);
    }
    else
    {
        Link* link = OBJ_TO_LINK(list, object);

        link->prevObject = NULL;
        link->nextObject = list->headObject;

        OBJ_TO_LINK(list, list->headObject)->prevObject = object;
        list->headObject = object;
        list->numObjects++;
        NW_WARNING( list->numObjects != 0, "Overflow numObjects" );
    }
}


//---------------------------------------------------------------------------
void
List_Insert(List* list, void* target, void* object)
{
    NW_NULL_ASSERT(list  );
    NW_NULL_ASSERT(object);

    if (target == NULL)
    {
        // targetが指定されていない場合は、List_Append()と同じ。
        List_Append(list, object);
    }
    else if (target == list->headObject)
    {
        // targetがリストの先頭である場合はList_Prepend()と同じ。
        List_Prepend(list, object);
    }
    else
    {
        Link* link    = OBJ_TO_LINK(list, object);
        void* prevObj = OBJ_TO_LINK(list, target)->prevObject;
        Link* prevLnk = OBJ_TO_LINK(list, prevObj);

        link->prevObject    = prevObj;
        link->nextObject    = target;
        prevLnk->nextObject = object;
        OBJ_TO_LINK(list, target)->prevObject = object;
        list->numObjects++;
        NW_WARNING( list->numObjects != 0, "Overflow numObjects" );
    }
}


//---------------------------------------------------------------------------
void
List_Remove(List* list, void* object)
{
    Link* link;

    NW_NULL_ASSERT(list  );
    NW_NULL_ASSERT(object);

    link = OBJ_TO_LINK(list, object);

    if (link->prevObject == NULL)
    {
        list->headObject = link->nextObject;
    }
    else
    {
        OBJ_TO_LINK(list, link->prevObject)->nextObject = link->nextObject;
    }
    if (link->nextObject == NULL)
    {
        list->tailObject = link->prevObject;
    }
    else
    {
        OBJ_TO_LINK(list, link->nextObject)->prevObject = link->prevObject;
    }
    link->prevObject = NULL;
    link->nextObject = NULL;
    list->numObjects--;
}


//---------------------------------------------------------------------------
void*
List_GetNext( const List* list, const void* object)
{
    NW_NULL_ASSERT(list);

    if (object == NULL)
    {
        return list->headObject;
    }
    return OBJ_TO_LINK(list, object)->nextObject;
}

//---------------------------------------------------------------------------
void*
List_GetPrev(const List* list, const void* object)
{
    NW_NULL_ASSERT(list);

    if (object == NULL)
    {
        return list->tailObject;
    }
    return OBJ_TO_LINK(list, object)->prevObject;
}

//---------------------------------------------------------------------------
void*
List_GetNth( const List* list, u16 index)
{
    int      count  = 0;
    Link* object = NULL;

    NW_NULL_ASSERT(list);

    while ((object = static_cast<Link*>(List_GetNext(list, object))) != NULL)
    {
        if (index == count)
        {
            return object;
        }
        count++;
    }
    return NULL;
}

#if 0
//---------------------------------------------------------------------------
//! @brief        リストに繋がれているオブジェクトの数をカウントします。先頭か
//!               ら順番にリストをカウントする為、リストが長いと時間がかかります。
//!
//! @param[out]   list    リスト構造体へのポインタ。
//!
//! @return       リストに繋がれているオブジェクトの数。
//---------------------------------------------------------------------------
int
CountListNode(List* list)
{
    int   count  = 0;
    Node* object = NULL;

    NW_ASSERT_NOT_NULL(list);

    while ((object = List_GetNext(list, object)) != NULL)
    {
        count++;
    }

    return count;
}
#endif

} // namespace ut
} // namespace nw
