﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/assert.h>
#include <nw/ut/ut_BinaryFileFormat.h>
#include <nw/ut/ut_Inlines.h>


namespace nw {
namespace ut {


//---------------------------------------------------------------------------
bool
IsValidBinaryFile(
    const BinaryFileHeader* pHeader,
    u32                     signature,
    u32                     version,
    u16                     minBlocks  /* = 1 */
)
{
    NW_ASSERT_VALID_POINTER( pHeader );

    if ( pHeader->signature != signature )
    {
        NW_WARNING(false, "Signature check failed ('%c%c%c%c' must be '%c%c%c%c').",
            static_cast<char>(BitExtract(pHeader->signature, 24, 8)),
            static_cast<char>(BitExtract(pHeader->signature, 16, 8)),
            static_cast<char>(BitExtract(pHeader->signature,  8, 8)),
            static_cast<char>(BitExtract(pHeader->signature,  0, 8)),
            static_cast<char>(BitExtract(signature, 24, 8)),
            static_cast<char>(BitExtract(signature, 16, 8)),
            static_cast<char>(BitExtract(signature,  8, 8)),
            static_cast<char>(BitExtract(signature,  0, 8))
        );
        return false;
    }

    if ( pHeader->byteOrder != BYTE_ORDER_MARK )
    {
        NW_WARNING(false, "Unsupported byte order.");
        return false;
    }

    if ( NW_UT_VERSION_MAJOR(version) != NW_UT_VERSION_MAJOR(pHeader->version) ||
         NW_UT_VERSION_MINOR(version) < NW_UT_VERSION_MINOR(pHeader->version)  ||
         NW_UT_VERSION_BINARYBUGFIX(version) > NW_UT_VERSION_BINARYBUGFIX(pHeader->version) )
    {
        NW_WARNING(false, "Version check failed (bin:'%d.%d.%d.%d', lib:'%d.%d.%d.%d').",
            NW_UT_VERSION_MAJOR(pHeader->version),
            NW_UT_VERSION_MINOR(pHeader->version),
            NW_UT_VERSION_MICRO(pHeader->version),
            NW_UT_VERSION_BINARYBUGFIX(pHeader->version),
            NW_UT_VERSION_MAJOR(version),
            NW_UT_VERSION_MINOR(version),
            NW_UT_VERSION_MICRO(version),
            NW_UT_VERSION_BINARYBUGFIX(version)
        );
        return false;
    }

    if( pHeader->fileSize < sizeof(BinaryFileHeader) + sizeof(BinaryBlockHeader) * minBlocks )
    {
        NW_WARNING(false, "Too small file size(=%d).", pHeader->fileSize);
        return false;
    }

    if( pHeader->dataBlocks < minBlocks )
    {
        NW_WARNING(false, "Too small number of data blocks(=%d).", pHeader->dataBlocks);
        return false;
    }

    return true;
}

//---------------------------------------------------------------------------
bool
IsValidBinaryFile(
                  const RawEndianBinaryFileHeader* pHeader,
                  u32                     signature,
                  u32                     version,
                  u16                     minBlocks  /* = 1 */
                  )
{
    NW_ASSERT_VALID_POINTER( pHeader );

    if ( pHeader->signature != signature )
    {
        NW_WARNING(false, "Signature check failed ('%c%c%c%c' must be '%c%c%c%c').",
            static_cast<char>(BitExtract(pHeader->signature, 24, 8)),
            static_cast<char>(BitExtract(pHeader->signature, 16, 8)),
            static_cast<char>(BitExtract(pHeader->signature,  8, 8)),
            static_cast<char>(BitExtract(pHeader->signature,  0, 8)),
            static_cast<char>(BitExtract(signature, 24, 8)),
            static_cast<char>(BitExtract(signature, 16, 8)),
            static_cast<char>(BitExtract(signature,  8, 8)),
            static_cast<char>(BitExtract(signature,  0, 8))
            );
        return false;
    }

    if ( pHeader->byteOrder != BYTE_ORDER_MARK )
    {
        NW_WARNING(false, "Unsupported byte order.");
        return false;
    }

    if ( NW_UT_VERSION_MAJOR(version) != NW_UT_VERSION_MAJOR(pHeader->version) ||
        NW_UT_VERSION_MINOR(version) < NW_UT_VERSION_MINOR(pHeader->version)  ||
        NW_UT_VERSION_BINARYBUGFIX(version) > NW_UT_VERSION_BINARYBUGFIX(pHeader->version) )
    {
        NW_WARNING(false, "Version check failed (bin:'%d.%d.%d.%d', lib:'%d.%d.%d.%d').",
            NW_UT_VERSION_MAJOR(pHeader->version),
            NW_UT_VERSION_MINOR(pHeader->version),
            NW_UT_VERSION_MICRO(pHeader->version),
            NW_UT_VERSION_BINARYBUGFIX(pHeader->version),
            NW_UT_VERSION_MAJOR(version),
            NW_UT_VERSION_MINOR(version),
            NW_UT_VERSION_MICRO(version),
            NW_UT_VERSION_BINARYBUGFIX(version)
            );
        return false;
    }

    if( pHeader->fileSize < sizeof(RawEndianBinaryFileHeader) + sizeof(RawEndianBinaryFileHeader) * minBlocks )
    {
        NW_WARNING(false, "Too small file size(=%d).", pHeader->fileSize);
        return false;
    }

    if( pHeader->dataBlocks < minBlocks )
    {
        NW_WARNING(false, "Too small number of data blocks(=%d).", pHeader->dataBlocks);
        return false;
    }

    return true;
}


//---------------------------------------------------------------------------
bool
IsReverseEndianBinaryFile( const BinaryFileHeader* pFileHeader )
{
    NW_ASSERT_VALID_POINTER( pFileHeader );
    return ( pFileHeader->byteOrder != BYTE_ORDER_MARK );
}


//---------------------------------------------------------------------------
BinaryBlockHeader*
GetNextBinaryBlockHeader(
    BinaryFileHeader*       pFileHeader,
    BinaryBlockHeader*      pBlockHeader
)
{
    NW_ASSERT_VALID_POINTER( pFileHeader );

    void* ptr;
    if ( ! IsReverseEndianBinaryFile( pFileHeader ) )
    {
        if ( pBlockHeader == NULL )
        {
            if ( pFileHeader->dataBlocks == 0 ) return NULL;
            ptr = AddOffsetToPtr( pFileHeader, pFileHeader->headerSize );
        }
        else
        {
            ptr = AddOffsetToPtr( pBlockHeader, pBlockHeader->size );
        }

        if ( ptr >= AddOffsetToPtr( pFileHeader, pFileHeader->fileSize ) )
        {
            return NULL;
        }
    }
    else
    {
        if ( pBlockHeader == NULL )
        {
            if ( pFileHeader->dataBlocks == 0 ) return NULL;
            ptr = AddOffsetToPtr( pFileHeader, ReverseEndian( pFileHeader->headerSize ) );
        }
        else
        {
            ptr = AddOffsetToPtr( pBlockHeader, ReverseEndian( pBlockHeader->size ) );
        }

        if ( ptr >= AddOffsetToPtr( pFileHeader, ReverseEndian( pFileHeader->fileSize ) ) )
        {
            return NULL;
        }
    }

    return reinterpret_cast<BinaryBlockHeader*>( ptr );
}

} // namespace ut
} // namespace nw
