﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>

#if defined(NW_MCS_ENABLE)

#include <nw/mcs/mcs_SimpleRingBuffer.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/ut/ut_ScopedLock.h>

#include <cstring>
#include <new>

#if defined( NW_PLATFORM_CAFE )
#pragma ghs nowarning 177 // function was declared but never referenced.
#endif

/* ========================================================================
    static
   ======================================================================== */

namespace
{

using namespace nw::mcs::internal;

inline u32
RoundUp(
    u32    value,
    u32    alignment
)
{
    return (value + alignment -1) & ~(alignment -1);
}

inline void*
RoundUp(
    void*  value,
    u32    alignment
)
{
    return reinterpret_cast<void*>(RoundUp(reinterpret_cast<u32>(value), alignment));
}

inline u32
RoundDown(
    u32    value,
    u32    alignment
)
{
    return value & ~(alignment -1);
}

u32
GetReadableBytes(
    const SimpleDataRange&  range,
    u32                     bufSize
)
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return range.end + bufSize - range.start;
    }
}

u32
GetSequenceReadableBytes(
    const SimpleDataRange&  range,
    u32                     bufSize
)
{
    if (range.start <= range.end)
    {
        return range.end - range.start;
    }
    else
    {
        return bufSize - range.start;
    }
}

u32
GetWritableBytes(
    const SimpleDataRange&  range,
    u32                     bufSize
)
{
    u32 start = (range.start + bufSize - 4) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return start + bufSize - range.end;
    }
}

u32
GetSequenceWritableBytes(
    const SimpleDataRange&  range,
    u32                     bufSize
)
{
    u32 start = (range.start + bufSize - 4) % bufSize;

    if (start >= range.end)
    {
        return start - range.end;
    }
    else
    {
        return bufSize - range.end;
    }
}

}   // namespace


/* ========================================================================
    外部
   ======================================================================== */

namespace nw
{
namespace mcs
{
namespace internal
{

//---------------------------------------------------------------------------
void
SimpleRingBuffer::Init(
    void*   buf,
    u32     size
)
{
    u32 bufStart = reinterpret_cast<u32>(buf);
    u32 bufEnd = bufStart + size;

    bufStart = RoundUp(bufStart, 4);   // 4バイトアライン
    bufEnd = RoundDown(bufEnd, 4);     // 4バイトアライン

    NW_ASSERT(bufStart < bufEnd);
    NW_ASSERT(bufEnd - bufStart > sizeof(SimpleRingBufferHeader) + 4); // ヘッダサイズ超過であること

    m_pHeader = new( reinterpret_cast<void*>(bufStart) ) SimpleRingBufferHeader();
    m_pHeader->bufSize = bufEnd - bufStart - sizeof(SimpleRingBufferHeader);

    m_pHeader->dataRange.start = 0;
    m_pHeader->dataRange.end   = 0;
    m_pHeader->mutex.Initialize();
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::Discard()
{
    ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

    m_pHeader->dataRange.start = 0;
    m_pHeader->dataRange.end   = 0;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::GetReadableBytes()
{
    SimpleDataRange dataRange;

    {
        ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

        dataRange = m_pHeader->dataRange;
    }

    return ::GetReadableBytes(dataRange, m_pHeader->bufSize);
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::GetWritableBytes()
{
    SimpleDataRange dataRange;

    {
        ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

        dataRange = m_pHeader->dataRange;
    }

    return ::GetWritableBytes(dataRange, m_pHeader->bufSize);
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Read(
    void*       buf,
    u32         size
)
{
    SimpleDataRange dataRange;

    BeginRead(&dataRange);

    u32 ret = ContinueRead(&dataRange, buf, size);

    EndRead(dataRange);

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginRead(SimpleDataRange* pDataRange)
{
    // データ開始・終了位置を取得
    ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

    *pDataRange = m_pHeader->dataRange;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::ContinueRead(
    SimpleDataRange*    pDataRange,
    void*               buf,
    u32                 size
)
{
    u32 readableBytes = ::GetReadableBytes(*pDataRange, m_pHeader->bufSize);
    if (readableBytes == 0)
    {
        return 0;
    }

    u8* pDst = static_cast<u8*>(buf);
    const u8 *const pSrc = reinterpret_cast<u8*>(m_pHeader) + sizeof(SimpleRingBufferHeader);

    u32 restSize = GetMin(readableBytes, size);

    u32 ret = restSize;

    while (restSize > 0)
    {
        const u32 readByte = GetMin(GetSequenceReadableBytes(*pDataRange, m_pHeader->bufSize), restSize);
        (void)std::memcpy(pDst, pSrc + pDataRange->start, readByte);
        pDataRange->start = (pDataRange->start + readByte) % m_pHeader->bufSize;
        pDst += readByte;
        restSize -= readByte;
    }

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::EndRead(const SimpleDataRange& dataRange)
{
    // 読み込み開始位置を更新
    ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

    m_pHeader->dataRange.start = dataRange.start;
}


//---------------------------------------------------------------------------
u32
SimpleRingBuffer::Write(
    const void*     buf,
    u32             size
)
{
    SimpleDataRange dataRange;

    BeginWrite(&dataRange);

    u32 ret = ContinueWrite(&dataRange, buf, size);

    EndWrite(dataRange);

    return ret;
}


//---------------------------------------------------------------------------
void
SimpleRingBuffer::BeginWrite(SimpleDataRange* pDataRange)
{
    // データ開始・終了位置を取得
    ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

    *pDataRange = m_pHeader->dataRange;
}

/*
    end ポイントが、buf + bufSize でないようにすること。このときは、bufを指すようにする。

    書き込むときは、end はstartと同じ値にならないようにすること。常にend は常に startに対して 4バイト以上空きを取ることになる。
*/
u32
SimpleRingBuffer::ContinueWrite(
    SimpleDataRange*    pDataRange,
    const void*         buf,
    u32                 size
)
{
    u32 writableBytes = ::GetWritableBytes(*pDataRange, m_pHeader->bufSize);
    if (writableBytes == 0)
    {
        return 0;
    }

    u8 *const pDst = reinterpret_cast<u8*>(m_pHeader) + sizeof(SimpleRingBufferHeader);
    const u8* pSrc = static_cast<const u8*>(buf);

    u32 restSize = GetMin(writableBytes, size);

    u32 ret = restSize;

    while (restSize > 0)
    {
        const u32 writeByte = GetMin(GetSequenceWritableBytes(*pDataRange, m_pHeader->bufSize), restSize);
        (void)std::memcpy(pDst + pDataRange->end, pSrc, writeByte);
        pDataRange->end = (pDataRange->end + writeByte) % m_pHeader->bufSize;
        pSrc += writeByte;
        restSize -= writeByte;
    }

    return ret;
}

void
SimpleRingBuffer::EndWrite(const SimpleDataRange& dataRange)
{
    // 書き込み開始位置を更新
    ut::ScopedLock<ut::Mutex> lock(m_pHeader->mutex);

    m_pHeader->dataRange.end = dataRange.end;
}

}   // namespace internal
}   // namespace mcs
}   // namespace nw

#endif  // #if defined(NW_MCS_ENABLE)
