﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <winsock.h>
#pragma comment(lib, "wsock32.lib")

#include <nw/mcs/mcs_Base.h>

#if defined(NW_MCS_ENABLE)

#include <nw/mcs/mcs_FileIO.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/ut/ut_ScopedLock.h>

namespace
{

using namespace nw::mcs;
using namespace nw::mcs::internal;

u32
WriteStream(
    FileInfo*   /* pFileInfo */,
    const void* buf,
    u32         size
);

}

namespace nw
{
namespace mcs
{

using namespace internal;


//  CE切り替え
static bool ce_enable = true;
void FileIO_CEEnable(bool enable)
{
    ce_enable = enable;
}


u32
WriteStream(
    FileInfo*   pFileInfo,
    const u8* buf,
    u32         size
)
{
    return ::WriteStream(pFileInfo, static_cast<const void*>( buf ), size);
}

//---------------------------------------------------------------------------
//! @brief       データの受信を行います。
//!              元の Mcs_Read を呼び出した後、エンディアン変更を行います。
//!
//! @param[in]   channel     データ受信の識別用の値。ユーザー任意に決められます。
//! @param[in]   data        読み込むデータを格納するバッファへのポインタ。
//! @param[in]   size        読み込むデータを格納するバッファのサイズ。
//! @param[in]   num        ４バイトのエンディアン変更を行う数。
//!
//! @return      関数が成功した場合0を返します。
//!              失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
u32
static
Mcs_Read_CE(
    ChannelType channel,
    void*       data,
    u32         size,
    u32         num = 0
)
{
    if( num == 0 )
        num = size / sizeof( u32 );
    u32 errorCode = Mcs_Read(channel, data, size );
    u32* buf = static_cast<u32*>(data);
    for( u32 i = 0; i < num; i++ )
    {
        buf[i] = htonl( buf[i] );
    }
    return errorCode;
}

//---------------------------------------------------------------------------
//! @brief       データの受信を行います。
//!              元の Mcs_Peek を呼び出した後、エンディアン変更を行います。
//!
//! @param[in]   channel     データ受信の識別用の値。ユーザー任意に決められます。
//! @param[in]   data        読み込むデータを格納するバッファへのポインタ。
//! @param[in]   size        読み込むデータを格納するバッファのサイズ。
//! @param[in]   num        ４バイトのエンディアン変更を行う数。
//!
//! @return       関数が成功した場合0を返します。
//!               失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
static u32
Mcs_Peek_CE(
    ChannelType channel,
    void*       data,
    u32         size,
    u32         num = 0
)
{
    if( num == 0 )
        num = size / sizeof( u32 );
    u32 errorCode = Mcs_Peek(channel, data, size );
    u32* buf = static_cast<u32*>(data);
    for( u32 i = 0; i < num; i++ )
    {
        buf[i] = htonl( buf[i] );
    }
    return errorCode;
}

//---------------------------------------------------------------------------
//! @brief       データを送信します。
//!              エンディアン変更を行った後、元のWriteStreamを呼び出します。
//!
//! @param[in]   pFindInfo   現時点では使用しません。
//! @param[in]   buf         送信するデータを格納するバッファへのポインタ。
//! @param[in]   size        送信するデータのサイズ。
//! @param[in]   num         ４バイトのエンディアン変更を行う数。
//!
//! @return      関数が成功した場合0を返します。
//!               失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
static u32
WriteStream_CE(
    FileInfo*   pFileInfo,
    void* data,
    u32         size,
    u32         num = 0
)
{
    if( num == 0 )
        num = size / sizeof( u32 );
    u32* buf = static_cast<u32*>(data);
    for( u32 i = 0; i < num; i++ )
    {
        buf[i] = ntohl( buf[i] );
    }
    return ::WriteStream( pFileInfo, static_cast<const void*>( data ), size );
}

//  汎用。全ての変数のエンディアン変更を行う
#define Mcs_Read_CEFORALLU32(struct_type)\
u32 Mcs_Read(\
    ChannelType channel,\
    struct_type* p,\
    u32         size\
)\
{\
    (void)channel;                              \
    return Mcs_Read_CE(FILEIO_CHANNEL, p, size);\
}
//  汎用。全ての変数のエンディアン変更を行う
#define Mcs_Peek_CEFORALLU32(struct_type)\
u32 Mcs_Peek(\
    ChannelType channel,\
    struct_type* p,\
    u32         size\
)\
{\
    (void)channel;                              \
    return Mcs_Peek_CE(FILEIO_CHANNEL, p, size);\
}
//  汎用。全ての変数のエンディアン変更を行う
#define WriteStream_CEFORALLU32(struct_type)\
u32 WriteStream(\
    FileInfo*   pFileInfo,\
    struct_type* p,\
    u32         size\
)\
{\
    return WriteStream_CE(pFileInfo, static_cast<void*>( p ), size);\
}
/*
template< typename T >
u32
WriteStream(
    FileInfo*   pFileInfo,
    T* p,
    u32         size
)
{
    return WriteStream_CE(pFileInfo, static_cast<void*>( p ), size);
}
*/
WriteStream_CEFORALLU32(FileIOChunkHeader)
WriteStream_CEFORALLU32(FileIOCommand)
WriteStream_CEFORALLU32(FileIOReadCommand)

//  FileIOOpenCommand用、最初の５つの変数だけエンディアン変更を行う
u32
WriteStream(
    FileInfo*   pFileInfo,
    FileIOOpenCommand* cmd,
    u32         size
)
{
    return WriteStream_CE(pFileInfo, cmd, size, 5);
}
WriteStream_CEFORALLU32(FileIOWriteCommand)
WriteStream_CEFORALLU32(FileIOSeekCommand)

//  FileIOFindFirstCommand用、最初の５つの変数だけエンディアン変更を行う
u32
WriteStream(
    FileInfo*   pFileInfo,
    FileIOFindFirstCommand* cmd,
    u32         size
)
{
    return WriteStream_CE(pFileInfo, cmd, size, 5);
}

WriteStream_CEFORALLU32(FileIOFindNextCommand)

//  FileIOShowFileDlgCommand用、最初の５つの変数だけエンディアン変更を行う
u32
WriteStream(
    FileInfo*   pFileInfo,
    FileIOShowFileDlgCommand* cmd,
    u32         size
)
{
    return WriteStream_CE(pFileInfo, cmd, size, 5);
}

//  FileIOExpandPathCommand用、最初の２つの変数だけエンディアン変更を行う
u32
WriteStream(
    FileInfo*   pFileInfo,
    FileIOExpandPathCommand* cmd,
    u32         size
)
{
    return WriteStream_CE(pFileInfo, cmd, size, 2);
}

//  全ての変数のエンディアン変更を行うものは、マクロで行う

Mcs_Read_CEFORALLU32( FileIOChunkHeader )
Mcs_Read_CEFORALLU32( FileIOResponse )
Mcs_Read_CEFORALLU32( FileIOOpenResponse )
Mcs_Read_CEFORALLU32( FileIOReadResponse )
Mcs_Read_CEFORALLU32( FileIOSeekResponse )

//  FileIOFindResponse用、最初の９つの変数だけエンディアン変更を行う
u32
Mcs_Read(
    ChannelType channel,
    FileIOFindResponse* res,
    u32         size
)
{
    (void)channel;
    return Mcs_Read_CE(FILEIO_CHANNEL, res, size, 9);
}

Mcs_Read_CEFORALLU32( FileIOShowFileDlgResponse )
Mcs_Read_CEFORALLU32( FileIOExpandPathResponse )
Mcs_Peek_CEFORALLU32( FileIOChunkHeader )

//  元のFileIO_Readを宣言します。
u32
_FileIO_Read(
    FileInfo*   pFileInfo,
    void*       buffer,
    u32         length,
    u32*        pReadBytes
);
//---------------------------------------------------------------------------
//! @brief       元のFileIO_Readを呼んだ後、エンディアン変更を行います。
//!
//! @param[out]  pFileInfo    ファイル情報構造体へのポインタ。
//! @param[in]   buffer       読み込むためのバッファ。
//! @param[in]   length       読み込むバイト数。
//! @param[out]  pReadBytes   実際に読み込んだバイト数を返す変数へのポインタ。
//!
//! @return      関数が成功した場合0を返します。
//!               失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
u32
FileIO_Read(
    FileInfo*   pFileInfo,
    void*       buffer,
    u32         length,
    u32*        pReadBytes
)
{
    //  元のFileIO_Readを呼びます。
    u32 result = _FileIO_Read( pFileInfo, buffer, length, pReadBytes );
    return result;

}

}   // namespace mcs
}   // namespace nw

//  元のFileIO_Readを、_FileIO_Readに定義しなおします。
//  （こちらのFileIO_Readが呼ばれるように）
#define FileIO_Read     _FileIO_Read

// #if defined(NW_MCS_ENABLE)
#endif

#include "./mcs_FileIO.cpp"
