﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>
#include <nw/mcs/mcs_Base.h>

#if defined(NW_MCS_ENABLE)

#include <nw/mcs/mcs_FileIO.h>
#include <nw/mcs/mcs_Common.h>
#include <nw/ut/os/ut_Mutex.h>
#include <nw/ut/ut_ScopedLock.h>
#include <nw/ut/ut_String.h>

#include <nw/dev/dev_Fs.h>
#include <cstring>

/* ========================================================================
    定数
   ======================================================================== */

namespace
{

const int SEND_RETRY_MAX        = 10;
const int LOOP_RETRY_MAX        = 1000;
const int LOOP_RETRY_NOTIMEOUT  = -1;
const int FILE_MOUNT_PATH_MAX   = 16;

using namespace nw::mcs;
using namespace nw::mcs::internal;

int            s_InitializedCount   = 0;        // 初期化回数の参照カウント
nw::ut::Mutex  s_Mutex;
FileIOWork*    s_pFileIOWork;
void*          s_RegisteredBuffer = NULL;
FSClient*      s_FsClient;
FSCmdBlock*    s_FsCmdBlock;
char           s_FsMountPoint[ FILE_MOUNT_PATH_MAX ] = "";

#if defined(NW_PLATFORM_CAFE)
// FSの非同期関数の完了を待つ間、定期的にMcs_Polling()を呼び出します。
class FsAsyncWaiting
{
    static OSEvent s_FsEvent;
    static const OSTimeNanoseconds s_PollingInterval = 1000 * 1000 * 1000; // 1sec
    FSAsyncParams m_AsyncParams;
    FSStatus m_Status;

public:
    static void StaticInitialize()
    {
        OSInitEvent(&s_FsEvent, false, OS_EVENT_AUTO);
    }

    FsAsyncWaiting()
        : m_Status(FS_STATUS_OK)
    {
        m_AsyncParams.userCallback = AsyncCallback;
        m_AsyncParams.userContext  = this;
        m_AsyncParams.ioMsgQueue   = NULL;
    }

    operator FSAsyncParams*()
    {
        return &m_AsyncParams;
    }

    FSStatus GetStatus() const
    {
        return m_Status;
    }

    FSStatus WaitDone()
    {
        while (!OSWaitEventWithTimeout(&s_FsEvent, s_PollingInterval))
        {
            Mcs_Polling();
        }

        return this->GetStatus();
    }

private:
    static void AsyncCallback(
        FSClient*    /* client */,
        FSCmdBlock*  /* command */,
        FSStatus     status,
        void*        pContext)
    {
        FsAsyncWaiting* p = static_cast<FsAsyncWaiting*>(pContext);

        p->m_Status = status;

        OSSignalEvent(&s_FsEvent);
    }
};

OSEvent FsAsyncWaiting::s_FsEvent;
#endif

#if defined( NW_PLATFORM_WIN32 ) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
  using namespace nw::internal::winext;
#endif

/* ========================================================================
    staticな関数
   ======================================================================== */

//---------------------------------------------------------------------------
//! @brief       mcsサーバとの接続状態を確認します。
//!
//! @return      mcsサーバと接続されていたらtrue、
//!               接続されていなかったらfalseを返します。
//---------------------------------------------------------------------------
inline bool
CheckConnect()
{
    if (Mcs_IsServerConnect())
    {
        return true;
    }
    else
    {
        NW_WARNING(false, "Mcs file I/O: not server connect\n");
        return false;
    }
}

//---------------------------------------------------------------------------
//! @brief       パス文字列をコピーします。
//!
//! @param[out]  dst   文字列のコピー先バッファ。
//! @param[in]   src   コピー元文字列。
//---------------------------------------------------------------------------
void
CopyPathString(
    char*       dst,
    const char* src
)
{
#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
    (void)std::strncpy(dst, src, FILEIO_PATH_MAX - 1);
    dst[FILEIO_PATH_MAX - 1] = '\0';
#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif
}

char*
StrNCpy(
    char*       dst,
    const char* src,
    size_t      n
)
{
    if (src)
    {
    #ifdef NW_COMPILER_MSVC
    #pragma warning(push)
    #pragma warning(disable : 4996)
    #endif
        (void)std::strncpy(dst, src, n);
    #ifdef NW_COMPILER_MSVC
    #pragma warning(pop)
    #endif
    }
    else
    {
        (void)std::memset(dst, '\0', n);
    }

    return dst;
}

//---------------------------------------------------------------------------
//! @brief       データを送信します。
//!
//!              送信が成功するまで数回リトライしています。
//!
//! @param[in,out] pFindInfo 現時点では使用しません。
//! @param[in]   buf         送信するデータを格納するバッファへのポインタ。
//! @param[in]   size        送信するデータのサイズ。
//!
//! @return      関数が成功した場合0を返します。
//!               失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
u32
WriteStream(
    FileInfo*   /* pFileInfo */,
    const void* buf,
    u32         size
)
{
    u32 writableBytes = 0;
    for (int retryCount = 0; retryCount < SEND_RETRY_MAX; ++retryCount)
    {
        u32 errorCode = Mcs_GetWritableBytes(FILEIO_CHANNEL, &writableBytes);
        if (errorCode != MCS_ERROR_SUCCESS)
        {
            return errorCode;
        }

        if (size <= writableBytes)
        {
            return Mcs_Write(FILEIO_CHANNEL, buf, size);
        }
        else
        {
            SleepThread(16);
        }
    }

    NW_WARNING(false, "NW Mcs File I/O: send time out writableBytes=[%u] sendSize=[%u]\n", writableBytes, size);
    return FILEIO_ERROR_COMERROR;
}

//---------------------------------------------------------------------------
//! @brief       受信待ちをします。
//!              コマンドのレスポンスが受信されるまで処理をブロックします。
//!
//! @param[in,out] pFindInfo  現時点では使用しません。
//!
//! @return      関数が成功した場合0を返します。
//!               失敗した場合エラーコード(0以外の値)を返します。
//---------------------------------------------------------------------------
u32
ReceiveResponse(
    FileInfo*   /* pFindInfo */,
    int         retryCnt            = LOOP_RETRY_MAX
)
{
    FileIOChunkHeader resChunkHead;

    //----------------------------------------------------------------------
    // レスポンスが帰ってくる(共有メモリに書き込まれる)までポーリングを呼びます。
    // レスポンスヘッダが読み込めるようになるまで待機しています。
    int loopCnt = 0;
    for (; LOOP_RETRY_NOTIMEOUT == retryCnt || loopCnt < retryCnt; ++loopCnt)
    {
        if (u32 errorCode = Mcs_Polling())
        {
            return errorCode;
        }

        if (Mcs_GetReadableBytes(FILEIO_CHANNEL) >= sizeof(resChunkHead))
        {
            break;
        }

        SleepThread(8);
    }
    // 一定時間レスポンスが無かった場合には、エラーとします。
    if (LOOP_RETRY_NOTIMEOUT != retryCnt && loopCnt >= retryCnt)
    {
        NW_WARNING(false, "NW Mcs File I/O: polling time out( wait for chunk header.)\n");
        return FILEIO_ERROR_COMERROR;
    }

    //----------------------------------------------------------------------
    // 受信バッファから破棄しないで、レスポンスヘッダの内容を読み取ります。
    (void)Mcs_Peek(FILEIO_CHANNEL, &resChunkHead, sizeof(resChunkHead));


    //----------------------------------------------------------------------
    // ヘッダの情報から、
    // チャンクサイズ分受信が完了するまで、待機します。
    loopCnt = 0;
    for (; LOOP_RETRY_NOTIMEOUT == retryCnt || loopCnt < retryCnt; ++loopCnt)
    {
        if (u32 errorCode = Mcs_Polling())
        {
            return errorCode;
        }

        if (Mcs_GetReadableBytes(FILEIO_CHANNEL) + resChunkHead.chunkSize >= sizeof(FileIOChunkHeader))
        {
            break;
        }

        SleepThread(8);
    }
    // 一定時間レスポンスが無かった場合には、エラーとします。
    if (LOOP_RETRY_NOTIMEOUT != retryCnt && loopCnt >= retryCnt)
    {
        NW_WARNING(false, "NW Mcs File I/O: polling time out( wait for data. chunkSize=[%u])\n", resChunkHead.chunkSize );
        return FILEIO_ERROR_COMERROR;
    }

    return MCS_ERROR_SUCCESS;
}

}   // namespace

/* ========================================================================
    外部関数
   ======================================================================== */
//
//　FileIO_XXX系関数の処理のながれ
//
//   FileIO_XXXはいづれも、大まかに下記のような流れで処理を行います。
//       ・接続状態の確認
//       ・コマンドの生成送信
//       ・コマンドレスポンスの受信待機(ReceiveResponse())
//       ・レスポンスデータを解析して目的のデータの取り出し
//
// コマンドレスポンスの受信待機を行っているReceiveResponse()内では、
// 以下の操作をブロック状態で行います。
// 指定回数(LOOP_RETRY_NOTIMEOUT)以上リトライしても、レスポンスが得られない場合は、
// タイムアウトエラーとして処理します。
//
//      ・MCSモジュールをポーリングしてHIO共有メモリの状態を更新。(Mcs_Polling())
//      ・読み込み可能バイトを監視して、レスポンスヘッダの受信を待つ。(Mcs_GetReadableBytes())
//      ・受信したレスポンスヘッダから、データサイズを読み取り、レスポンスデータの受信試行をする。
//
namespace nw
{
namespace mcs
{

using namespace internal;

//---------------------------------------------------------------------------
void
FileIO_Initialize()
{
    if (s_InitializedCount)
    {
        ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);
        ++s_InitializedCount;
        return;
    }

    ++s_InitializedCount;
    s_Mutex.Initialize();

#if defined(NW_PLATFORM_CAFE)
    FsAsyncWaiting::StaticInitialize();
#endif
}

//---------------------------------------------------------------------------
void
FileIO_Finalize()
{
    {
        ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

        --s_InitializedCount;

        if (s_InitializedCount > 0)
        {
            return;
        }
    }

    s_Mutex.Finalize();
}

//---------------------------------------------------------------------------
bool
FileIO_IsInitialized()
{
#if ! defined( NW_MCS_FILEIO_USE_FS )
    return (s_InitializedCount > 0) && s_pFileIOWork;
#else
    return (s_InitializedCount > 0) && s_pFileIOWork && (s_FsClient != NULL) && (s_FsCmdBlock != NULL);
#endif
}

//---------------------------------------------------------------------------
void
FileIO_RegisterBuffer(void* tempBuf)
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT(s_pFileIOWork == 0);

    s_pFileIOWork = static_cast<FileIOWork*>(tempBuf);
    Mcs_RegisterBuffer(FILEIO_CHANNEL, s_pFileIOWork->commBuf, sizeof(s_pFileIOWork->commBuf));

    s_RegisteredBuffer = tempBuf;
}

//---------------------------------------------------------------------------
void
FileIO_UnregisterBuffer()
{
    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT_NOT_NULL(s_pFileIOWork);

    Mcs_UnregisterBuffer(FILEIO_CHANNEL);

    s_pFileIOWork = 0;
    s_RegisteredBuffer = NULL;
}

//---------------------------------------------------------------------------
void*
FileIO_GetRegisteredBuffer()
{
    return s_RegisteredBuffer;
}

//---------------------------------------------------------------------------
void
FileIO_RegisterClientHandle(FSClient* handle, FSCmdBlock* cmdBlock, const char* mountPoint)
{
    s_FsClient = handle;
    s_FsCmdBlock = cmdBlock;
    ut::strcpy( s_FsMountPoint, FILE_MOUNT_PATH_MAX, mountPoint );
}


//---------------------------------------------------------------------------
static u32 ErrorCodeFromFsToFileIO_(FSStatus code)
{
    switch ( code )
    {
    case FS_STATUS_OK:
        return FILEIO_ERROR_SUCCESS;
    case FS_STATUS_UNSUPPORTED_CMD:
    case FS_STATUS_MEDIA_NOT_READY:
    case FS_STATUS_MAX:
        return FILEIO_ERROR_IOERROR;
    case FS_STATUS_ACCESS_ERROR:
    case FS_STATUS_PERMISSION_ERROR:
        return FILEIO_ERROR_SECURITYERROR;
    }

    return FILEIO_ERROR_UNKNOWN;
}


//---------------------------------------------------------------------------
u32
FileIO_Open(
    FileInfo*   pFileInfo,
    const char* fileName,
    u32         openFlag
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);
    NW_ASSERT_NOT_NULL(fileName);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

#if ! defined( NW_MCS_FILEIO_USE_FS )
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    {
        FileIOOpenCommand& cmd = *reinterpret_cast<FileIOOpenCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FILEOPEN;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.flag        = openFlag;
        cmd.mode        = 0;
        CopyPathString(reinterpret_cast<char*>(cmd.fileName), fileName);

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    // レスポンスの受信を待ち合わせます。
    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOOpenResponse res;

    if (sizeof(res) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    pFileInfo->handle       = res.handle;
    pFileInfo->fileSize     = res.fileSize;
    pFileInfo->errorCode    = res.errorCode;
#else

    const int POSIX_PATH_MAX = FILEIO_PATH_MAX + FILE_MOUNT_PATH_MAX;
    char      filenameBuffer[POSIX_PATH_MAX ];

    // 環境変数の展開。
    if ( openFlag & FILEIO_FLAG_INCENVVAR && ut::strnchr(fileName, POSIX_PATH_MAX, '%'))
    {
        int len = std::strlen(s_FsMountPoint);

        if (len >= POSIX_PATH_MAX)
        {
            pFileInfo->errorCode = FILEIO_ERROR_FILENOTFOUND;
            return pFileInfo->errorCode;
        }

        ut::strcpy(filenameBuffer, POSIX_PATH_MAX, s_FsMountPoint);

        u32 errorCode = FileIO_ExpandPath(fileName, filenameBuffer + len, POSIX_PATH_MAX - len);

        if ( errorCode != FILEIO_ERROR_SUCCESS )
        {
            pFileInfo->errorCode = FILEIO_ERROR_FILENOTFOUND;
            return pFileInfo->errorCode;
        }
    }
    else
    {
        ut::strcpy(filenameBuffer, POSIX_PATH_MAX, s_FsMountPoint);
        ut::strncat(filenameBuffer, POSIX_PATH_MAX, fileName,  POSIX_PATH_MAX);
    }

    // Windows 形式から Posix への変換
    char* pDriveLetter = ut::strnchr(filenameBuffer, FILEIO_PATH_MAX + FILE_MOUNT_PATH_MAX, ':');

    if (pDriveLetter)
    {
        NW_ASSERT( pDriveLetter != filenameBuffer );

        char drive = *(pDriveLetter - 1);
        *pDriveLetter       = drive;
        *(pDriveLetter - 1) = '/';
    }

    std::replace(filenameBuffer, filenameBuffer + std::strlen(filenameBuffer), '\\', '/');

    const char* mode;

    switch (openFlag & FILEIO_FLAG_READWRITE)
    {
    case FILEIO_FLAG_READ:
        mode = "r";
        break;
    case FILEIO_FLAG_WRITE:
        mode = "w";
        break;
    case FILEIO_FLAG_READWRITE:
        mode = "r+";
        break;
    default:
        mode = "";
    }

    FSStatus result = FSOpenFile(s_FsClient, s_FsCmdBlock, filenameBuffer, mode, &pFileInfo->handle, FS_RET_ALL_ERROR);

    if (result != FS_STATUS_OK &&
        (openFlag & FILEIO_FLAG_READWRITE) == FILEIO_FLAG_READWRITE)
    {
        if (!(openFlag & FILEIO_FLAG_NOCREATE))
        {
            result = FSOpenFile(s_FsClient, s_FsCmdBlock, filenameBuffer, "w+", &pFileInfo->handle, FS_RET_ALL_ERROR);
        }
    }

    if ( result != FS_STATUS_OK )
    {
        NW_MCS_LOG("FSOpenFile() failed. (%d)\n", result);
        pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );
        return pFileInfo->errorCode;
    }

    FSStat fileState;
    result = FSGetStatFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, &fileState, FS_RET_ALL_ERROR );

    pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );

    if ( result == FS_STATUS_OK )
    {
        pFileInfo->fileSize = fileState.size;
    }

#endif

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_Close(FileInfo* pFileInfo)
{
    NW_ASSERT_NOT_NULL(pFileInfo);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

#if ! defined( NW_MCS_FILEIO_USE_FS )
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    {
        FileIOCommand& cmd = *reinterpret_cast<FileIOCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FILECLOSE;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.handle      = pFileInfo->handle;

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOResponse result;

    if (sizeof(result) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &result, sizeof(result));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

#else

    FSStatus result = FSCloseFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, FS_RET_ALL_ERROR );
    pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );

#endif

    return pFileInfo->errorCode;
}


//---------------------------------------------------------------------------
u32
FileIO_Read(
    FileInfo*   pFileInfo,
    void*       buffer,
    u32         length,
    u32*        pReadBytes
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

#if ! defined( NW_MCS_FILEIO_USE_FS )
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    //----------------------------------------------------------------------------------------
    // ”File Readを行う”というコマンドを生成して、送信します。
    {
        FileIOReadCommand& cmd = *reinterpret_cast<FileIOReadCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FILEREAD;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.handle      = pFileInfo->handle;
        cmd.pBuffer     = buffer;
        cmd.size        = length;

        // コマンド送信
        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }


    //----------------------------------------------------------------------------------------
    // 送信したコマンドの結果(レスポンス)受信を待機します。(FileIOReadResponse)
    while (true)
    {
        pFileInfo->errorCode = ReceiveResponse(pFileInfo);
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        FileIOChunkHeader chunkHead;

        pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        // FileIOReadResponse
        FileIOReadResponse res;

        if (sizeof(res) > chunkHead.chunkSize)
        {
            return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
        }

        pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        if (res.errorCode != FILEIO_ERROR_CONTINUE && res.errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode = res.errorCode;
        }

        if (sizeof(res) + res.size != chunkHead.chunkSize)
        {
            return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
        }

        // データを実際に読み込んでいます。
        pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, res.pBuffer, res.size);
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        if (res.errorCode == FILEIO_ERROR_SUCCESS)
        {
            pFileInfo->errorCode = res.errorCode;
            *pReadBytes = res.totalSize;   // 読み込みバイト数
            break;
        }
    }
#else
#if defined(NW_PLATFORM_CAFE)
    FsAsyncWaiting asyncWaiting;
    FSStatus result = FSReadFileAsync(
        s_FsClient,
        s_FsCmdBlock,
        buffer,
        1,
        length,
        pFileInfo->handle,
        0,
        FS_RET_ALL_ERROR,
        asyncWaiting);

    if (result == FS_STATUS_OK)
    {
        result = asyncWaiting.WaitDone();
    }
#else
    FSStatus result = FSReadFile(
        s_FsClient,
        s_FsCmdBlock,
        buffer,
        1,
        length,
        pFileInfo->handle,
        0,
        FS_RET_ALL_ERROR);
#endif

    if (result > 0)
    {
        pFileInfo->errorCode = FILEIO_ERROR_SUCCESS;
        *pReadBytes = result;
    }
    else
    {
        pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );
    }

#endif

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_Write(
    FileInfo*   pFileInfo,
    const void* buffer,
    u32         length
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

#if ! defined( NW_MCS_FILEIO_USE_FS )
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    const u8* fromBuf = static_cast<const u8*>(buffer);
    // maxSize = 読み込み時にデータ転送に利用するデータチャンク(ペイロード部)の最大サイズです。
    const u32 maxSize =  FILEIO_CHUNK_SIZE_MAX - sizeof(FileIOWriteCommand);

    while (length > 0)
    {
        const u32 sendSize = GetMin(length, maxSize);

        //-------------------------------------------
        // FileIO_Write コマンドを送信します。
        {
            FileIOWriteCommand& cmd = *reinterpret_cast<FileIOWriteCommand*>(s_pFileIOWork->tempBuf);

            cmd.chunkID     = FILEIO_COMMAND_FILEWRITE;
            cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader) + sendSize;
            cmd.pFileInfo   = pFileInfo;
            cmd.handle      = pFileInfo->handle;
            cmd.size        = sendSize;

            // コマンド送信
            pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
            if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
            {
                NW_WARNING(false, "NW Mcs : FileIO_Write( send command ) : polling time out\n" );
                return pFileInfo->errorCode;
            }
        }

        //-------------------------------------------
        // 実際に書き込むデータを送信します。
        pFileInfo->errorCode = WriteStream(pFileInfo, fromBuf, sendSize);
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            NW_WARNING(false, "NW Mcs : FileIO_Write( send data ) : polling time out\n" );
            return pFileInfo->errorCode;
        }

        //-------------------------------------------
        // 書き込み送信に対するレスポンスを待ちます。
        pFileInfo->errorCode = ReceiveResponse(pFileInfo);
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        FileIOChunkHeader chunkHead;

        pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        FileIOResponse res;

        if (sizeof(res) != chunkHead.chunkSize)
        {
            return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
        }

        pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }

        pFileInfo->errorCode = res.errorCode;

        // エラーが発生したら抜ける
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            break;
        }

        length -= sendSize;
        fromBuf += sendSize;
    }

#else

    {
#if defined(NW_PLATFORM_CAFE)
        FsAsyncWaiting asyncWaiting;
        FSStatus result = FSWriteFileAsync(
            s_FsClient,
            s_FsCmdBlock,
            const_cast<void*>(buffer),
            1,
            length,
            pFileInfo->handle,
            0,
            FS_RET_ALL_ERROR,
            asyncWaiting);

        if (result == FS_STATUS_OK)
        {
            result = asyncWaiting.WaitDone();
        }
#else
        FSStatus result = FSWriteFile(
            s_FsClient,
            s_FsCmdBlock,
            const_cast<void*>(buffer),
            1,
            length,
            pFileInfo->handle,
            0,
            FS_RET_ALL_ERROR );
#endif

        if (result > 0)
        {
            pFileInfo->errorCode = FILEIO_ERROR_SUCCESS;
        }
        else
        {
            pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );
        }
    }
#endif

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_Seek(
    FileInfo*   pFileInfo,
    s32         distanceToMove,
    u32         moveMethod,
    u32*        pNewFilePointer
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

#if ! defined( NW_MCS_FILEIO_USE_FS )
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    {
        FileIOSeekCommand& cmd = *reinterpret_cast<FileIOSeekCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID         = FILEIO_COMMAND_FILESEEK;
        cmd.chunkSize       = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo       = pFileInfo;
        cmd.handle          = pFileInfo->handle;
        cmd.distanceToMove  = distanceToMove;
        cmd.moveMethod      = moveMethod;

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOSeekResponse res;

    if (sizeof(res) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    pFileInfo->errorCode = res.errorCode;

    if (pFileInfo->errorCode == FILEIO_ERROR_SUCCESS && pNewFilePointer)
    {
        *pNewFilePointer = res.filePointer;
    }

#else
    (void)pNewFilePointer;
    FSStatus result = 0;

    switch ( moveMethod )
    {
    case FILEIO_SEEK_BEGIN:
        {
            u32 newPosition = static_cast<u32>(distanceToMove);
            result = FSSetPosFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, newPosition, FS_RET_ALL_ERROR );
        }
        break;

    case FILEIO_SEEK_CURRENT:
        {
            FSFilePosition position;
            result = FSGetPosFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, &position, FS_RET_ALL_ERROR );

            if ( result != FS_STATUS_OK )
            {
                pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );
                return pFileInfo->errorCode;
            }

            u32 newPosition = position + static_cast<u32>(distanceToMove);
            result = FSSetPosFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, newPosition, FS_RET_ALL_ERROR );
        }
        break;

    case FILEIO_SEEK_END:
        {
            u32 newPosition = pFileInfo->fileSize - static_cast<u32>(distanceToMove);
            result = FSSetPosFile( s_FsClient, s_FsCmdBlock, pFileInfo->handle, newPosition, FS_RET_ALL_ERROR );
        }
        break;
    }

    pFileInfo->errorCode = ErrorCodeFromFsToFileIO_( result );

#endif

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_FindFirst(
    FileInfo*       pFileInfo,
    FindData*       pFindData,
    const char*     fileName
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);
    NW_ASSERT_NOT_NULL(pFindData);
    NW_ASSERT_NOT_NULL(fileName);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

    //----------------------------
    // 接続状態の確認
    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    //----------------------------
    // コマンドの生成送信
    {
        FileIOFindFirstCommand& cmd = *reinterpret_cast<FileIOFindFirstCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FINDFIRST;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.pFindData   = pFindData;
        cmd.mode        = 0;
        CopyPathString(reinterpret_cast<char*>(cmd.fileName), fileName);

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    //----------------------------
    // コマンドレスポンスの受信待機
    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    //----------------------------
    // レスポンスデータを解析して目的のデータの取り出し
    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOFindResponse res;

    if (sizeof(res) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    pFileInfo->handle       = res.handle;
    pFileInfo->fileSize     = 0;
    pFileInfo->errorCode    = res.errorCode;

    if (pFileInfo->errorCode == FILEIO_ERROR_SUCCESS)
    {
        // 成功したら FindData のメンバをセット
        pFindData->size         = res.fileSize;
        pFindData->attribute    = res.fileAttribute;
        pFindData->date         = res.fileDate;
        pFindData->reserved1    = 0;
        pFindData->reserved2    = 0;
        CopyPathString(pFindData->name, reinterpret_cast<char*>(res.fileName));
    }

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_FindNext(
    FileInfo*   pFileInfo,
    FindData*   pFindData
)
{
    NW_ASSERT_NOT_NULL(pFileInfo);
    NW_ASSERT_NOT_NULL(pFindData);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    {
        FileIOFindNextCommand& cmd = *reinterpret_cast<FileIOFindNextCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FINDNEXT;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.handle      = pFileInfo->handle;
        cmd.pFindData   = pFindData;

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOFindResponse res;

    if (sizeof(res) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    pFileInfo->errorCode = res.errorCode;

    if (pFileInfo->errorCode == FILEIO_ERROR_SUCCESS)
    {
        // 成功したら FindData のメンバをセット
        pFindData->size         = res.fileSize;
        pFindData->attribute    = res.fileAttribute;
        pFindData->date         = res.fileDate;
        pFindData->reserved1    = 0;
        pFindData->reserved2    = 0;
        CopyPathString(pFindData->name, reinterpret_cast<char*>(res.fileName));
    }

    return pFileInfo->errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_FindClose(FileInfo* pFileInfo)
{
    NW_ASSERT_NOT_NULL(pFileInfo);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

    if (! CheckConnect())
    {
        return pFileInfo->errorCode = FILEIO_ERROR_NOTCONNECT;
    }

    {
        FileIOCommand& cmd = *reinterpret_cast<FileIOCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_FINDCLOSE;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.pFileInfo   = pFileInfo;
        cmd.handle      = pFileInfo->handle;

        pFileInfo->errorCode = WriteStream(pFileInfo, &cmd, sizeof(cmd));
        if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
        {
            return pFileInfo->errorCode;
        }
    }

    pFileInfo->errorCode = ReceiveResponse(pFileInfo);
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOChunkHeader chunkHead;

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    FileIOResponse res;

    if (sizeof(res) != chunkHead.chunkSize)
    {
        return pFileInfo->errorCode = FILEIO_ERROR_PROTOCOLERROR;
    }

    pFileInfo->errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (pFileInfo->errorCode != FILEIO_ERROR_SUCCESS)
    {
        return pFileInfo->errorCode;
    }

    return pFileInfo->errorCode = res.errorCode;
}

//---------------------------------------------------------------------------
u32
FileIO_ExpandPath(
    const char* fileName,
    char*       resultBuf,
    u32         resultBufSize
)
{
    NW_ASSERT_NOT_NULL(fileName);
    NW_ASSERT_NOT_NULL(resultBuf);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

    if (! CheckConnect())
    {
        return FILEIO_ERROR_NOTCONNECT;
    }

    u32 errorCode;
    {
        FileIOExpandPathCommand& cmd = *reinterpret_cast<FileIOExpandPathCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_EXPANDPATH;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        CopyPathString(reinterpret_cast<char*>(cmd.fileName), fileName);

        errorCode = WriteStream(NULL, &cmd, sizeof(cmd));
        if (errorCode != FILEIO_ERROR_SUCCESS)
        {
            return errorCode;
        }
    }

    errorCode = ReceiveResponse(NULL, LOOP_RETRY_NOTIMEOUT);

    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    FileIOChunkHeader chunkHead;

    errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    FileIOExpandPathResponse res;

    if (sizeof(res) > chunkHead.chunkSize)
    {
        return FILEIO_ERROR_PROTOCOLERROR;
    }

    errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    if (res.errorCode != FILEIO_ERROR_SUCCESS)
    {
        return res.errorCode;
    }

    if (sizeof(res) + res.size != chunkHead.chunkSize)
    {
        return FILEIO_ERROR_PROTOCOLERROR;
    }

    // ファイル文字列を読み取る
    errorCode = Mcs_Read(FILEIO_CHANNEL, resultBuf, ut::Min(resultBufSize, res.size));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    if (resultBufSize < res.size)
    {
        resultBuf[resultBufSize - 1] = '\0';

        // 残っている内容を読み飛ばす
        errorCode = Mcs_Skip(FILEIO_CHANNEL, res.size - resultBufSize);
        if (errorCode != FILEIO_ERROR_SUCCESS)
        {
            return errorCode;
        }
    }
    else
    {
        (void)std::memset(resultBuf + res.size, '\0', resultBufSize - res.size);
    }

    return FILEIO_ERROR_SUCCESS;
}

//---------------------------------------------------------------------------
u32
FileIO_ShowFileDialog(
    char*       fileNameBuf,
    u32         fileNameBufSize,
    u32         flag,
    const char* initialPath,
    const char* filter,
    const char* defaultExt,
    const char* title
)
{
    NW_ASSERT_NOT_NULL(fileNameBuf);
    NW_ASSERT(fileNameBufSize >= 1);

    ut::ScopedLock<ut::Mutex> lockObj(s_Mutex);

    NW_ASSERT( FileIO_IsInitialized() );

    if (! CheckConnect())
    {
        return FILEIO_ERROR_NOTCONNECT;
    }

    u32 errorCode;
    {
        FileIOShowFileDlgCommand& cmd = *reinterpret_cast<FileIOShowFileDlgCommand*>(s_pFileIOWork->tempBuf);

        cmd.chunkID     = FILEIO_COMMAND_SHOWFILEDLG;
        cmd.chunkSize   = sizeof(cmd) - sizeof(FileIOChunkHeader);
        cmd.flag        = flag;
        cmd.mode        = 0;
        cmd.reserved    = 0;

        if (initialPath)
        {
            CopyPathString(reinterpret_cast<char*>(cmd.initialPath), initialPath);
        }
        else    // NULLの場合は、空文字列にする。
        {
            (void)std::memset(cmd.initialPath, '\0', sizeof(cmd.initialPath));
        }
        (void)StrNCpy(reinterpret_cast<char*>(cmd.filter),        filter,        sizeof(cmd.filter));
        (void)StrNCpy(reinterpret_cast<char*>(cmd.defaultExt),    defaultExt,    sizeof(cmd.defaultExt));
        (void)StrNCpy(reinterpret_cast<char*>(cmd.title),         title,         sizeof(cmd.title));

        errorCode = WriteStream(NULL, &cmd, sizeof(cmd));
        if (errorCode != FILEIO_ERROR_SUCCESS)
        {
            return errorCode;
        }
    }

    errorCode = ReceiveResponse(NULL, LOOP_RETRY_NOTIMEOUT);    // ダイアログを表示するため、タイムアウト無しにしている。
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    FileIOChunkHeader chunkHead;

    errorCode = Mcs_Read(FILEIO_CHANNEL, &chunkHead, sizeof(chunkHead));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    FileIOShowFileDlgResponse res;

    if (sizeof(res) > chunkHead.chunkSize)
    {
        return FILEIO_ERROR_PROTOCOLERROR;
    }

    errorCode = Mcs_Read(FILEIO_CHANNEL, &res, sizeof(res));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    if (res.errorCode != FILEIO_ERROR_SUCCESS)
    {
        return res.errorCode;
    }

    if (sizeof(res) + res.size != chunkHead.chunkSize)
    {
        return FILEIO_ERROR_PROTOCOLERROR;
    }

    // ファイル文字列を読み取る
    errorCode = Mcs_Read(FILEIO_CHANNEL, fileNameBuf, ut::Min(fileNameBufSize, res.size));
    if (errorCode != FILEIO_ERROR_SUCCESS)
    {
        return errorCode;
    }

    if (fileNameBufSize < res.size)
    {
        fileNameBuf[fileNameBufSize - 1] = '\0';

        // 残っている内容を読み飛ばす
        errorCode = Mcs_Skip(FILEIO_CHANNEL, res.size - fileNameBufSize);
        if (errorCode != FILEIO_ERROR_SUCCESS)
        {
            return errorCode;
        }
    }
    else
    {
        (void)std::memset(fileNameBuf + res.size, '\0', fileNameBufSize - res.size);
    }

    return FILEIO_ERROR_SUCCESS;
}

}   // namespace mcs
}   // namespace nw

// #if defined(NW_MCS_ENABLE)
#endif
