﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/config.h>
#if defined(NW_PLATFORM_WIN32)

#include <windows.h>

#pragma warning ( push )
    #pragma  warning ( disable : 4189 )
    #pragma  warning ( disable : 4100 )
    #pragma  warning ( disable : 4063 )
    #pragma  warning ( disable : 4005 )
    #include <texUtils.h>
#pragma warning ( pop )

#include <nw/font/font_TextureTilingConverter.h>

namespace nw {
namespace font {
namespace internal {

namespace
{
typedef BOOL (*InitializeFunction)(TC2Config* pConfig);
typedef BOOL (*ConvertTilingFunction)(GX2Surface *pInSurface, GX2TileMode dstTileMode, u32 initialSwizzle, GX2Surface* pOutSurface);
typedef BOOL (*DestroyGX2SurfaceFunction)(GX2Surface* pGX2Surface);
typedef BOOL (*DestroyFunction)(void);
}

TextureTilingConverterImpl* TextureTilingConverter::m_pImpl = nullptr;

class TextureTilingConverterImpl
{
public:

    //-----------------------------------------------------------------------------
    TextureTilingConverterImpl()
        : m_hDll(NULL)
        , Initialize(NULL)
        , ConvertTiling(NULL)
        , DestroyGX2Surface(NULL)
        , Destroy(NULL)
    {
    }

    //-----------------------------------------------------------------------------
    BOOL InitializeFunctions()
    {
        if(Initialize == nullptr || ConvertTiling == nullptr || DestroyGX2Surface == nullptr || Destroy == nullptr)
        {
            m_hDll = LoadLibraryEx(L"texUtils", NULL, LOAD_WITH_ALTERED_SEARCH_PATH);
            if (m_hDll == nullptr)
            {
                return FALSE;
            }

            // 関数アドレスを取得します。
            *reinterpret_cast<void**>(&Initialize) = GetProcAddress(m_hDll, "TC2Initialize");
            *reinterpret_cast<void**>(&ConvertTiling) = GetProcAddress(m_hDll, "TC2ConvertTiling");
            *reinterpret_cast<void**>(&DestroyGX2Surface) = GetProcAddress(m_hDll, "TC2DestroyGX2Surface");
            *reinterpret_cast<void**>(&Destroy) = GetProcAddress(m_hDll, "TC2Destroy");

            if (Initialize == nullptr || ConvertTiling == nullptr || DestroyGX2Surface == nullptr || Destroy == nullptr)
            {
                return FALSE;
            }
        }

        return TRUE;
    }

    //-----------------------------------------------------------------------------
    BOOL FinalizeFunctions()
    {
        FreeLibrary(m_hDll);
        m_hDll              = nullptr;

        Initialize          = nullptr;
        ConvertTiling       = nullptr;
        DestroyGX2Surface   = nullptr;
        Destroy             = nullptr;
    }

    //-----------------------------------------------------------------------------
    // DLL から取得する関数のポインタ
    InitializeFunction              Initialize;
    ConvertTilingFunction           ConvertTiling;
    DestroyGX2SurfaceFunction       DestroyGX2Surface;
    DestroyFunction                 Destroy;

private:
    HMODULE          m_hDll; // DLL のハンドラ
};

//-----------------------------------------------------------------------------

BOOL TextureTilingConverter::Initialize(TC2Config& config)
{
    if(m_pImpl == nullptr)
    {
        TextureTilingConverterImpl* newImpl = new TextureTilingConverterImpl();
        if(newImpl->InitializeFunctions() == FALSE)
        {
            delete newImpl;
            return FALSE;
        }

        m_pImpl = newImpl;
    }

    return m_pImpl->Initialize(&config);
}

//-----------------------------------------------------------------------------

BOOL TextureTilingConverter::ConvertTiling(GX2Surface& src, GX2Surface& dst)
{
    return m_pImpl != nullptr && m_pImpl->ConvertTiling(&src, GX2_TILE_MODE_LINEAR_SPECIAL, 0, &dst);
}

//-----------------------------------------------------------------------------

BOOL TextureTilingConverter::DestorySurface(GX2Surface& surface)
{
    return m_pImpl != nullptr && m_pImpl->DestroyGX2Surface(&surface);
}

//-----------------------------------------------------------------------------

BOOL TextureTilingConverter::Finalize()
{
    return m_pImpl != nullptr && m_pImpl->Destroy();
}

}
}
}

#endif // NW_PLATFORM_WIN32
