﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/types.h>

#include <nw/config.h>
#include <nw/font/font_CharWriter.h>
#include <nw/font/font_DispStringBuffer.h>

#include <nw/assert.h>  // NW_GL_ASSERT用

namespace nw {
namespace font {

/* =======================================================================
        public
   ======================================================================== */

/* ------------------------------------------------------------------------
        コンストラクタ/デストラクタ
   ------------------------------------------------------------------------ */

CharWriter::CharWriter()
:   m_FixedWidth(0),
    m_pFont(NULL),
    m_pDispStringBuffer(NULL),
    m_IsWidthFixed(false),
    m_ItalicRatio(0.0f),
    m_GradationMode(GRADMODE_NONE)
{
    SetTextColor(ut::Color4u8(ut::Color4u8::WHITE));  // m_TextColor, mVertexColor
    SetShadowAlpha(255);                              // m_ShadowAlpha
    SetScale(1, 1);                                   // m_Scale
    SetCursor(0, 0, 0);                               // m_CursorPos
}

CharWriter::~CharWriter()
{
}

/* ------------------------------------------------------------------------
        文字サイズ
   ------------------------------------------------------------------------ */

void
CharWriter::SetFontSize(
    f32 width,
    f32 height
)
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    NW_FONT_MIN_ASSERT(m_pFont->GetWidth(),  1);
    NW_FONT_MIN_ASSERT(m_pFont->GetHeight(), 1);
    SetScale(
        width  / m_pFont->GetWidth(),
        height / m_pFont->GetHeight()
    );
}

void
CharWriter::SetFontSize(f32 height)
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    NW_FONT_MIN_ASSERT(m_pFont->GetHeight(), 1);
    const f32 scale = height / m_pFont->GetHeight();
    SetScale(scale);
}

f32
CharWriter::GetFontWidth() const
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    return m_pFont->GetWidth() * m_Scale.x;
}

f32
CharWriter::GetFontHeight() const
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    return m_pFont->GetHeight() * m_Scale.y;
}

f32
CharWriter::GetFontAscent() const
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    return m_pFont->GetAscent() * m_Scale.y;
}

f32
CharWriter::GetFontDescent() const
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    return m_pFont->GetDescent() * m_Scale.y;
}

/* ------------------------------------------------------------------------
        文字描画
   ------------------------------------------------------------------------ */

f32
CharWriter::PrintGlyph(const Glyph& glyph)
{
    const CharWidths& widths = glyph.widths;

    f32 width;
    f32 left;
    if (m_IsWidthFixed)
    {
        f32 margin = (m_FixedWidth - widths.charWidth * m_Scale.x) / 2;

        width   = m_FixedWidth;
        left    = margin + widths.left * m_Scale.x;
    }
    else
    {
        width   = widths.charWidth * m_Scale.x;
        left    = widths.left      * m_Scale.x;
    }

    PrintGlyph(m_CursorPos.x + left, glyph);

    m_CursorPos.x += width;

    return width;
}

f32
CharWriter::Print(CharCode code)
{
    NW_ASSERT_VALID_POINTER(m_pFont);
    NW_ASSERT(code != Font::INVALID_CHARACTER_CODE);

    Glyph glyph;
    m_pFont->GetGlyph(&glyph, code);

    return PrintGlyph(glyph);
}

void
CharWriter::DrawGlyph(const Glyph& glyph)
{
    NW_ASSERT_VALID_POINTER(&glyph);
    PrintGlyph(m_CursorPos.x, glyph);
    m_CursorPos.x += glyph.widths.glyphWidth * m_Scale.x;
}


/* =======================================================================
        private
   ======================================================================== */


void
CharWriter::PrintGlyph(
    f32           x,
    const Glyph&  glyph
)
{
    NW_ASSERT_VALID_POINTER(&glyph);
    NW_FONT_MIN_ASSERT(glyph.texWidth, 1);
    NW_FONT_MIN_ASSERT(glyph.texHeight, 1);

    f32 texLeft;
    f32 texRight;

    // cellYは左上原点の値が入っているので、これをOpenGLの左下原点の
    // テクスチャ座標に変換してセットする。
    f32 texTop;
    f32 texBottom;

    f32 px;
    f32 py;
    f32 width;
    f32 height;

    if (m_pFont->IsEnableExtraMargin())
    {
        const f32 margin = 0.5f;
        texLeft   = (1.0f * glyph.cellX - margin) / glyph.texWidth;
        texRight  = (1.0f * (glyph.cellX + glyph.widths.glyphWidth) + margin)
                              / glyph.texWidth;
        texTop    = 1.f - (static_cast<float>(glyph.cellY) - margin) / glyph.texHeight;
        texBottom = 1.f - (static_cast<float>(glyph.cellY + glyph.height) + margin) / glyph.texHeight;

        px = x - margin * m_Scale.x;
        py = m_CursorPos.y - margin * m_Scale.y;
        width  = (glyph.widths.glyphWidth + margin * 2.0f) * m_Scale.x;
        height = (glyph.height + margin * 2.0f) * m_Scale.y;
    }
    else
    {
        texLeft   = (1.0f * glyph.cellX) / glyph.texWidth;
        texRight  = (1.0f * (glyph.cellX + glyph.widths.glyphWidth))
                              / glyph.texWidth;

        texTop    = 1.f - static_cast<float>(glyph.cellY) / glyph.texHeight;
        texBottom = 1.f - static_cast<float>(glyph.cellY + glyph.height) / glyph.texHeight;

        px = x;
        py = m_CursorPos.y;
        width  = glyph.widths.glyphWidth * m_Scale.x;
        height = glyph.height * m_Scale.y;
    }

    {
        const u32 charIdx = m_pDispStringBuffer->GetCharCount();

        if (charIdx >= m_pDispStringBuffer->GetCharCountMax())
        {
            NW_LOG("nw::font::CharWriter : Vertex Buffer Over.\n");
            return;
        }

        m_pDispStringBuffer->SetCharCount(charIdx + 1);

        internal::CharAttribute* pCharAttrs =
            &m_pDispStringBuffer->GetCharAttributes()[charIdx];

        // ポジションのセット
        // カーソルが進んだら下に行くようにyを反転する
        pCharAttrs->pos.Set(
            width,
            height,
            px,
            py);

        // カラーのセット
        for (int i = 0; i < internal::TEXTCOLOR_MAX; ++i)
        {
            pCharAttrs->color[i] = m_TextColors[i];
        }

        // 影のアルファのセット
        pCharAttrs->shadowAlpha = m_ShadowAlpha;

        // テクスチャ座標のセット
        pCharAttrs->tex.Set(
            texLeft,
            texTop,
            texRight,
            texBottom);

        pCharAttrs->italicOffset = static_cast<s16>(
            m_ItalicRatio * m_Scale.x * m_pFont->GetWidth());

        pCharAttrs->sheetIndex = glyph.sheetIndex;

        // テクスチャオブジェクトへのポインタのセット
        NW_ASSERT_NOT_NULL(glyph.pTextureObject);
        u8 flags = 0;
        if (m_pFont->IsBorderEffectEnabled())
        {
            flags |= internal::CharAttribute::BORDER_EFFECT;
        }
        pCharAttrs->SetTexObjAndFlags(glyph.pTextureObject, flags);
    }
}

void
CharWriter::StartPrint()
{
    NW_ASSERT_NOT_NULL(m_pDispStringBuffer);

    m_pDispStringBuffer->SetCharCount(0);
    m_pDispStringBuffer->ClearCommand();
}

void
CharWriter::EndPrint()
{
    NW_ASSERT_NOT_NULL(m_pDispStringBuffer);
}


/* ------------------------------------------------------------------------
        static
   ------------------------------------------------------------------------ */
/* ------------------------------------------------------------------------
        描画準備
   ------------------------------------------------------------------------ */

u32
CharWriter::GetDispStringBufferSize(u32 charNum)
{
    return sizeof(DispStringBuffer) + sizeof(internal::CharAttribute) * charNum;
}

u32
CharWriter::GetDispStringBufferSizeWithSharedCharAttrs()
{
    return sizeof(DispStringBuffer);
}

u32
CharWriter::GetGraphicsBufferSize(u32 charNum)
{
    return sizeof(internal::Vertex) *
        internal::VERTEX_NUMBER_BY_LETTER * charNum * DispStringBuffer::VERTEX_BUFFER_QUANTITY;
}

DispStringBuffer*
CharWriter::InitDispStringBuffer(
    void* drawBuffer,
    void* graphicsBuffer,
    u32 charNum,
    bool shadowEnabled /* = false */
)
{
    NW_ASSERT_NOT_NULL(drawBuffer);
    NW_ASSERT((nw::ut::GetIntPtr(graphicsBuffer) % GRAPHICS_BUFFER_ALIGNMENT) == 0);

    DispStringBuffer* result = new (drawBuffer) DispStringBuffer(charNum, graphicsBuffer, shadowEnabled);
    result->SetCharAttributes(reinterpret_cast<internal::CharAttribute*>(static_cast<u8*>(drawBuffer) + sizeof(DispStringBuffer)));
    result->SetUseSharedCharAttrs(false);
    return result;
}

DispStringBuffer*
CharWriter::InitDispStringBufferWithSharedCharAttrs(
    void* drawBuffer,
    void* graphicsBuffer,
    u32 charNum,
    bool shadowEnabled /* = false */
)
{
    NW_ASSERT_NOT_NULL(drawBuffer);
    NW_ASSERT((nw::ut::GetIntPtr(graphicsBuffer) % GRAPHICS_BUFFER_ALIGNMENT) == 0);

    DispStringBuffer* result = new (drawBuffer) DispStringBuffer(charNum, graphicsBuffer, shadowEnabled);
    result->SetUseSharedCharAttrs(true);
    return result;
}

}   // namespace font
}   // namespace nw
