﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/assert.h>
#include <nw/font/font_CharStrmReader.h>

namespace nw {
namespace font {
namespace internal {

#if defined(NW_FONT_UNREFERENCE)
const u32 SJIS_LOW_WIDTH    = 0xBC;
const u32 SJIS_LOW_BASE     = 0x40;
#endif
const u32 SJIS_HIGH0_ST     = 0x81;
#if defined(NW_FONT_UNREFERENCE)
const u32 SJIS_HIGH0_ED     = 0x85;
const u32 SJIS_HIGH1_ST     = 0x87;
const u32 SJIS_HIGH1_ED     = 0x88;
const u32 SJIS_HIGH2_ST     = 0x88;
#endif
const u32 SJIS_HIGH2_ED     = 0xA0;
const u32 SJIS_HIGH3_ST     = 0xE0;
#if defined(NW_FONT_UNREFERENCE)
const u32 SJIS_HIGH3_ED     = 0x100;
const u32 ASCII_ST          = 0x20;
const u32 ASCII_ED          = 0x80;
const u32 SJIS_HANKANA_ST   = 0xA0;
const u32 SJIS_HANKANA_ED   = 0xE0;
#endif

/*!--------------------------------------------------------------------------*
  @brief        対象のバイトがSJIS文字のリードバイトがどうか判定します。

  @param[in]    c   判定対象のバイトデータ。

  @return       c がリードバイトなら true そうでなければ false。
 *---------------------------------------------------------------------------*/
inline
bool
IsSJISLeadByte(u8 c)
{
    return ((SJIS_HIGH0_ST <= c) && (c < SJIS_HIGH2_ED))
            || (SJIS_HIGH3_ST <= c);
}

}   // namespace internal


u16
CharStrmReader::ReadNextCharUTF8()
{
    NW_ASSERT_VALID_POINTER( this );
    NW_ASSERT_VALID_POINTER( mCharStrm );
    //---- UTF-8 の1バイト目の条件
    NW_ASSERT( (GetChar<u8>() & 0xC0) != 0x80 );
    u16 code;

    if( (GetChar<u8>() & 0x80) == 0x00 )
    {
        //---- 1バイト文字
        code = GetChar<u8>();
        StepStrm<u8>();
    }
    else if( (GetChar<u8>() & 0xE0) == 0xC0 )
    {
        //---- 2バイト文字
        code = static_cast<u16>(
            ((GetChar<u8>(0) & 0x1F) << 6)
             | ((GetChar<u8>(1) & 0x3F) << 0)
        );
        StepStrm<u8>(2);
    }
    else
    {
        //---- 4バイト以上には対応しない
        NW_ASSERT( (GetChar<u8>() & 0xF0) == 0xE0 );

        //---- 3バイト文字
        code = static_cast<u16>(
            ((GetChar<u8>(0) & 0x1F) << 12)
             | ((GetChar<u8>(1) & 0x3F) << 6)
             | ((GetChar<u8>(2) & 0x3F) << 0)
        );
        StepStrm<u8>(3);
    }

    return code;
}

u16
CharStrmReader::ReadNextCharUTF16()
{
    NW_ASSERT_VALID_POINTER( this );
    NW_ASSERT_VALID_POINTER( mCharStrm );
    NW_ASSERT_ALIGN (mCharStrm, 2);
    u16 code;

    code = GetChar<u16>();
    StepStrm<u16>();

    return code;
}

u16
CharStrmReader::ReadNextCharCP1252()
{
    NW_ASSERT_VALID_POINTER( this );
    NW_ASSERT_VALID_POINTER( mCharStrm );
    u16 code;

    code = GetChar<u8>();
    StepStrm<u8>();

    return code;
}

u16
CharStrmReader::ReadNextCharSJIS()
{
    NW_ASSERT_VALID_POINTER( this );
    NW_ASSERT_VALID_POINTER( mCharStrm );
    u16 code;

    if (internal::IsSJISLeadByte(GetChar<u8>()))
    {
        //---- 2バイト文字
        code = static_cast<u16>((GetChar<u8>(0) << 8) | GetChar<u8>(1));
        StepStrm<u8>(2);
    }
    else
    {
        //---- 1バイト文字
        code = GetChar<u8>();
        StepStrm<u8>();
    }

    return code;
}

}   // namespace font
}   // namespace nw
