﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/gfnd.h>
#include <nw/dev/dev_Pad.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/dw/system/dw_Pointer.h>
#include <nw/dw/window/dw_WindowManager.h>
#include <nw/dw/window/dw_Window.h>
#include <nw/dw/system/dw_InputSettings.h>

namespace nw {
namespace internal {
namespace dw {

const f32 WindowManager::ALPHA_MIN = 0.05f;
const f32 WindowManager::ALPHA_MAX = 1.0f;
const f32 WindowManager::WINDOW_MARGIN = 10.f;

// -----------------------------------------------------------
WindowManager::WindowManager()
: m_HoldWindow( NULL ),
  m_HoldWithA( true ),
  m_PointerWindow(NULL),
  m_ActiveWindow( NULL ),
  m_ActiveWindowToTop( false ),
  m_MasterAlpha( ALPHA_MAX ),
  m_VisibleFlag( true ),
  m_EnableWindowControl( true ),
  m_WindowMenuMode( false ),
  m_WindowControlMode( false ),
  m_WindowMenu( m_WindowList )
{
}

// -----------------------------------------------------------
WindowManager::~WindowManager()
{
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++ )
    {
        itr->Destroy();
    }
}

// -----------------------------------------------------------
void
WindowManager::Initialize(f32 screenWidth, f32 screenHeight, nw::ut::IAllocator* allocator)
{
    m_WindowContext.Initialize(screenWidth, screenHeight);
    m_Pointer.Initialize(allocator);
}

// -----------------------------------------------------------
void
WindowManager::Finalize()
{
    m_Pointer.Finalize();
}

// -----------------------------------------------------------
bool
WindowManager::CreateWindow( Window* window, f32 x, f32 y )
{
    NW_ASSERT_NOT_NULL( window );
    if ( ! window->Create( x, y ) ) {
        return false;
    }

    m_WindowList.PushBack( window );
    m_WindowDrawList.PushFront( window );
    UpdateOrder();

    SetActiveWindow( window );

    return true;
}

// -----------------------------------------------------------
bool
WindowManager::CreateWindow( Window* window, f32 x, f32 y, f32 width, f32 height )
{
    NW_ASSERT_NOT_NULL( window );
    if ( ! window->Create( x, y ) ) {
        return false;
    }
    window->SetSize( nw::math::Vector2(width,height) );

    m_WindowList.PushBack( window );
    m_WindowDrawList.PushFront( window );
    UpdateOrder();

    SetActiveWindow( window );

    return true;
}

// -----------------------------------------------------------
void
WindowManager::DestroyWindow( Window* window )
{
    NW_ASSERT_NOT_NULL( window );

    if(window == m_HoldWindow)
    {
        m_HoldWindow = NULL;
    }
    if(window == m_ActiveWindow)
    {
        m_ActiveWindow = NULL;
    }
    if(window == m_PointerWindow)
    {
        m_PointerWindow = NULL;
    }

    window->Destroy();

    m_WindowList.Erase( window );
    m_WindowDrawList.Erase( window );
    UpdateOrder();

    UpdateActiveWindow();
}

// -----------------------------------------------------------
void
WindowManager::SetActiveWindow( Window* window )
{
    m_ActiveWindow = window;

    for ( WindowZOrderList::Iterator itr = m_WindowDrawList.GetBeginIter();
          itr != m_WindowDrawList.GetEndIter() ; itr++ )
    {
        itr->SetIsActive( &*itr == window );
    }
}

// -----------------------------------------------------------
void
WindowManager::SetForegroundWindow( Window* window )
{
    m_WindowDrawList.Erase( window );
    m_WindowDrawList.PushFront( window );

    if ( m_HoldWindow != NULL && window != m_HoldWindow ) {
        // ホールドウィンドウは、必ず最前面に
        m_WindowDrawList.Erase( m_HoldWindow );
        m_WindowDrawList.PushFront( m_HoldWindow );
    }

    UpdateOrder();
}

// -----------------------------------------------------------
void
WindowManager::SetPointerPosition( const nw::math::Vector2& pos )
{
    m_Pointer.SetPosition( pos );

    UpdateActiveWindow();
}

// -----------------------------------------------------------
void
WindowManager::Update(UIRenderer& renderer)
{
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++ )
    {
        itr->Update(renderer);
    }
}

// -----------------------------------------------------------
void
WindowManager::UpdateInputs(const nw::internal::dw::Inputs& inputs)
{
    if ( m_VisibleFlag )
    {
        if ( inputs.GetPad() != NULL &&
            IsPressWindowControlKey(*inputs.GetPad()) )
        {
            if ( m_WindowMenuMode )
            {
                m_WindowMenu.UpdateInputs(inputs);

                // 現在選択中のウィンドウをアクティブにする
                int activeWindowIndex = m_WindowMenu.GetCurrentWindowIndex();
                Window* activeWindow = GetWindowByIndex( activeWindowIndex );
                if ( activeWindow != NULL ) {
                    SetActiveWindow( activeWindow );
                    m_ActiveWindowToTop = true;
                }
            }
            else
            {
                OnUpdatePadForWindowControl(inputs);

                UpdateHoldWindow(*inputs.GetPad());

                // アクティブウィンドウの更新
                if ( m_HoldWindow == NULL ) {
                    UpdateActiveWindow();
                }
            }

            m_WindowControlMode = true;
        }
        else
        {
            if(m_WindowMenuMode)
            {
                LeaveWindowListMode();
            }

            if(inputs.GetMouse() != NULL && inputs.GetMouse()->IsConnected())
            {
                m_Pointer.SetPosition(inputs.GetPointerPosition());
                UpdatePointerWindow(inputs);
            }

            m_WindowControlMode = false;
        }

        if ( m_ActiveWindow != NULL ) {
            m_ActiveWindow->UpdateInputs(inputs);
        }
    }

    // 表示・非表示切り替え
    if (inputs.GetPad() != NULL &&
        IsPressWindowControlKey(*inputs.GetPad()) &&
        inputs.GetPad()->IsTrig( nw::dev::Pad::MASK_Z ) )
    {
        m_VisibleFlag = !m_VisibleFlag;
    }
}

// -----------------------------------------------------------
void
WindowManager::Draw( UIRenderer& renderer )
{
    /******************************************************************************
      ウィンドウ描画
    ******************************************************************************/
    if ( m_VisibleFlag )
    {
        for ( WindowZOrderList::ReverseIterator itr = m_WindowDrawList.GetBeginReverseIter();
              itr != m_WindowDrawList.GetEndReverseIter() ; ++itr )
        {
            if ( m_ActiveWindowToTop && m_ActiveWindow == &*itr ) {
                // アクティブウィンドウを一時的に最前面へ移動
                continue;
            }

            if ( itr->GetIsVisible() )
            {
                itr->Draw( m_WindowContext, renderer, m_MasterAlpha, itr->GetIsShade() );
            }
        }

        if ( m_ActiveWindow != NULL && m_ActiveWindowToTop ) {
            // アクティブウィンドウを一時的に最前面へ移動
            m_ActiveWindow->Draw( m_WindowContext, renderer, m_MasterAlpha, false );
        }
    }

    /******************************************************************************
      マネージャ情報描画
    ******************************************************************************/
    if ( m_WindowMenuMode )
    {
        // ウィンドウメニュー描画
        m_WindowMenu.Draw( m_WindowContext, renderer );
    }
    else if ( m_WindowControlMode )
    {
        // ウィンドウ位置＆サイズ描画
        if ( m_HoldWindow != NULL )
        {
            const nw::math::Vector2& windowPos  = m_HoldWindow->GetPosition();
            const nw::math::Vector2& windowSize = m_HoldWindow->GetSize();

            nw::math::VEC2 charSize = renderer.MeasureText(DrawTextArgs(), " ");

            static const f32 MARGIN_RIGHT = 20.f;
            static const f32 MARGIN_BOTTOM = 20.f;
            static const f32 WIDTH = charSize.x * 11.f;
            static const f32 HEIGHT = charSize.y * 2.f;

            f32 x = GetScreenSize().x - ( WIDTH + MARGIN_RIGHT );
            f32 y = GetScreenSize().y - ( HEIGHT + MARGIN_BOTTOM );

            renderer.BeginDraw();

            renderer.FillRectangle(
                NULL,
                DrawRectangleArgs().
                    SetTopLeft(x, y).
                    SetSize(WIDTH, HEIGHT).
                    SetColor(nw::ut::Color4f::BLACK()));

            renderer.DrawTextFormat(
                NULL,
                DrawTextArgs().
                SetTopLeft(x, y).
                SetColor(nw::ut::Color4f::WHITE()),
                "%4.0f , %4.0f",
                windowPos.x,
                windowPos.y);

            renderer.DrawTextFormat(
                NULL,
                DrawTextArgs().
                SetTopLeft(x, y + charSize.y).
                SetColor(nw::ut::Color4f::WHITE()),
                "%4.0f , %4.0f",
                windowSize.x,
                windowSize.y);

            renderer.EndDraw();
        }

        // ポインタ描画
        nw::dev::PointerType type;
        if ( m_HoldWindow == NULL )
        {
            type = nw::dev::POINTER_ARROW;
        }
        else if ( m_HoldWithA )
        {
        #if 0
            type = nw::dev::POINTER_MOVE;
        #else
            type = nw::dev::POINTER_ARROW;
        #endif
        }
        else
        {
        #if 0
            type = nw::dev::POINTER_RESIZE;
        #else
            type = nw::dev::POINTER_ARROW;
        #endif
        }

        m_Pointer.SetPointerType( type );
        m_Pointer.Draw(renderer);
    }

    m_WindowControlMode = false;
}

// -----------------------------------------------------------
const nw::math::Vector2&
WindowManager::GetScreenSize() const
{
    return m_WindowContext.GetScreenSize();
}

// -----------------------------------------------------------
void
WindowManager::detail_SortWindowList()
{
    WindowList::Iterator endItr = m_WindowList.GetEndIter();
    while( endItr != m_WindowList.GetBeginIter() ) {
        WindowList::Iterator maxItr = m_WindowList.GetBeginIter();
        NW_ASSERT( maxItr != m_WindowList.GetEndIter() );
        WindowList::Iterator secondItr = maxItr;
        ++secondItr;
        for ( WindowList::Iterator itr = secondItr; itr != endItr ; ++itr )
        {
            if ( itr->detail_GetWindowListOrder() > maxItr->detail_GetWindowListOrder() ) {
                maxItr = itr;
            }
        }
        Window* window = &*maxItr;
        m_WindowList.Erase( maxItr );
        m_WindowList.Insert( endItr, window );

        endItr = m_WindowList.GetIteratorFromPointer( window );
    }
}

// -----------------------------------------------------------
void
WindowManager::detail_SortWindowZOrderList()
{
    WindowZOrderList::Iterator endItr = m_WindowDrawList.GetEndIter();
    while( endItr != m_WindowDrawList.GetBeginIter() ) {
        WindowZOrderList::Iterator maxItr = m_WindowDrawList.GetBeginIter();
        NW_ASSERT( maxItr != m_WindowDrawList.GetEndIter() );
        WindowZOrderList::Iterator secondItr = maxItr;
        ++secondItr;
        for ( WindowZOrderList::Iterator itr = secondItr; itr != endItr ; ++itr )
        {
            if ( itr->detail_GetWindowZOrder() > maxItr->detail_GetWindowZOrder() ) {
                maxItr = itr;
            }
        }
        Window* window = &*maxItr;
        m_WindowDrawList.Erase( maxItr );
        m_WindowDrawList.Insert( endItr, window );

        endItr = m_WindowDrawList.GetIteratorFromPointer( window );
    }
}

// -----------------------------------------------------------
void
WindowManager::OnUpdatePadForWindowControl(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    // ウィンドウリストモード
    if ( m_HoldWindow == NULL )
    {
        if ( pad.IsTrig( nw::dev::Pad::MASK_UP ) ||
             pad.IsTrig( nw::dev::Pad::MASK_DOWN ) )
        {
            EnterWindowListMode(m_WindowContext.GetScreenSize());
        }
    }

    // クラシックコントローラのみに対応しているので、
    // ポインタは使えない

    // ポインター位置変更
    nw::math::Vector2 pointerDiff;
    nw::math::Vector2 pointerDiff_NoClamp;

    {
        nw::math::Vector2 pos;
#if 0
        const RmtPad& rmtPad = pad.GetRmtPad();
        if ( rmtPad.IsDpdValid() )
        {
            // ＤＰＤによる移動
            f32 pointerSpeed = (screenSize.height / 2) * mPointerSpeedDpd;
            pos.x = static_cast<int>( rmtPad.GetDpdPositon().x * pointerSpeed + 0.5f * screenSize.width  );
            pos.y = static_cast<int>( rmtPad.GetDpdPositon().y * pointerSpeed + 0.5f * screenSize.height );
        }
        else
#endif
        {
            pos = m_Pointer.GetPosition();

            // スティックによる移動
            pos.x += (  pad.GetLeftStick().x * 18 ) * ( pad.IsHold(nw::dev::Pad::MASK_B) ? 2 : 1 );
            pos.y += ( -pad.GetLeftStick().y * 18 ) * ( pad.IsHold(nw::dev::Pad::MASK_B) ? 2 : 1 );

            // ウィンドウを掴んだ状態での移動
            if ( m_HoldWindow != NULL )
            {
                const f32 REPEAT_STEP = 4.f;

                f32 diffX=0.f;
                f32 diffY=0.f;
                if ( pad.IsTrig( nw::dev::Pad::MASK_RIGHT ) ) {
                    diffX++;
                }
                else if ( pad.IsRepeatAccel( nw::dev::Pad::MASK_RIGHT ) ) {
                    diffX+=REPEAT_STEP;
                }
                if ( pad.IsTrig( nw::dev::Pad::MASK_LEFT ) ) {
                    diffX--;
                }
                else if ( pad.IsRepeatAccel( nw::dev::Pad::MASK_LEFT ) ) {
                    diffX-=REPEAT_STEP;
                }
                if ( pad.IsTrig( nw::dev::Pad::MASK_UP ) ) {
                    diffY--;
                }
                else if ( pad.IsRepeatAccel( nw::dev::Pad::MASK_UP ) ) {
                    diffY-=REPEAT_STEP;
                }
                if ( pad.IsTrig( nw::dev::Pad::MASK_DOWN ) ) {
                    diffY++;
                }
                else if ( pad.IsRepeatAccel( nw::dev::Pad::MASK_DOWN ) ) {
                    diffY+=REPEAT_STEP;
                }
                pos.x += diffX;
                pos.y += diffY;
            }
        }

        pointerDiff_NoClamp.x = pos.x - m_Pointer.GetPosition().x;
        pointerDiff_NoClamp.y = pos.y - m_Pointer.GetPosition().y;

        pos.x = ut::Clamp( pos.x, 0.f, GetScreenSize().x );
        pos.y = ut::Clamp( pos.y, 0.f, GetScreenSize().y );

        pointerDiff.x = pos.x - m_Pointer.GetPosition().x;
        pointerDiff.y = pos.y - m_Pointer.GetPosition().y;

        m_Pointer.SetPosition( pos );
        UpdatePointerWindow(inputs);
    }

    if ( m_VisibleFlag )
    {
        // アルファ値変更
        {
            f32 add = pad.GetRightStick().y * 0.02f;
            ChangeMasterAlpha( add );
        }

        if ( m_HoldWindow != NULL )
        {
            if ( m_HoldWithA )
            {
                // ウィンドウ位置変更
                nw::math::Vector2 pos = m_HoldWindow->GetPosition();
                pos.x += pointerDiff.x;
                pos.y += pointerDiff.y;
                m_HoldWindow->SetPosition( pos );
            }
            else
            {
                // ウィンドウサイズ変更
                nw::math::Vector2 size = m_HoldWindow->GetSize();
                size.x += pointerDiff_NoClamp.x;
                size.y += pointerDiff_NoClamp.y;
                ChangeWindowSize( m_HoldWindow, size );
            }
        }

        if ( m_ActiveWindow != NULL )
        {
            // ウィンドウシェード
            if ( pad.IsTrig( nw::dev::Pad::MASK_Y ) )
            {
                bool shadeFlag = m_ActiveWindow->GetIsShade();
                if ( shadeFlag )
                {
                    SetForegroundWindow( m_ActiveWindow );
                }
                m_ActiveWindow->SetIsShade( ! shadeFlag );
            }
        }
    }
}

// -----------------------------------------------------------
void
WindowManager::ChangeWindowSize( Window* window, const nw::math::Vector2& size )
{
    NW_ASSERT_NOT_NULL( window );

    nw::math::Vector2 pos = window->GetPosition();
    f32 right = pos.x + size.x;
    f32 bottom = pos.y + size.y;

    nw::math::Vector2 windowSize = size;
    if ( right < WINDOW_MARGIN ) {
        windowSize.x = WINDOW_MARGIN - pos.x;
    }
    if ( bottom < WINDOW_MARGIN ) {
        windowSize.y = WINDOW_MARGIN - pos.y;
    }

    window->SetSize( windowSize );
}

// -----------------------------------------------------------
void
WindowManager::EnterWindowListMode( const nw::math::Vector2& screenSize )
{
    int activeWindowIndex = GetWindowIndex( m_ActiveWindow );
    if ( activeWindowIndex < 0 ) {
        activeWindowIndex = m_WindowMenu.GetCurrentWindowIndex();
    }
    m_WindowMenu.SetCurrentWindowIndex( activeWindowIndex );

    const f32 MARGIN_WIDTH  = 200.f;
    const f32 MARGIN_HEIGHT = 100.f;
    const f32 WINDOW_WIDTH  = screenSize.x - MARGIN_WIDTH*2;
    const f32 WINDOW_HEIGHT = screenSize.y - MARGIN_HEIGHT*2;

    m_WindowMenu.Create( MARGIN_WIDTH, MARGIN_HEIGHT );
    m_WindowMenu.SetSize( nw::math::Vector2(WINDOW_WIDTH, WINDOW_HEIGHT) );

    m_WindowMenuMode = true;
}

// -----------------------------------------------------------
void
WindowManager::LeaveWindowListMode()
{
    int activeWindowIndex = m_WindowMenu.GetCurrentWindowIndex();
    Window* activeWindow = GetWindowByIndex( activeWindowIndex );
    if ( activeWindow != NULL ) {
        // ポインターをウィンドウ位置へ移動
        m_Pointer.SetPosition( activeWindow->GetPosition() );
    }

    UpdateOrder();

    m_WindowMenuMode = false;
}

// -----------------------------------------------------------
void
WindowManager::ChangeMasterAlpha( f32 add )
{
    f32 alpha = m_MasterAlpha;
    alpha += add;

    alpha = ut::Clamp<f32>( alpha, ALPHA_MIN, ALPHA_MAX );

    m_MasterAlpha = alpha;
}

// -----------------------------------------------------------
void
WindowManager::UpdateHoldWindow( const nw::dev::Pad& pad )
{
    if ( m_HoldWindow == NULL )
    {
        if ( IsPressWindowControlKey( pad ) )
        {
            if ( pad.IsTrig( nw::dev::Pad::MASK_A ) )
            {
                Window* pointerWindow = GetPointerWindow();
                if ( pointerWindow != NULL ) {
                    m_HoldWindow = pointerWindow;
                    m_HoldWithA = true;
                    SetForegroundWindow( pointerWindow );
                    UpdateActiveWindow();
                }
            }
            if ( pad.IsTrig( nw::dev::Pad::MASK_X ) )
            {
                Window* pointerWindow = GetPointerWindow();
                if ( pointerWindow != NULL ) {
                    m_HoldWindow = pointerWindow;
                    m_HoldWithA = false;
                    SetForegroundWindow( pointerWindow );
                    UpdateActiveWindow();
                }
            }
        }
    }
    else
    {
        if ( m_HoldWithA )
        {
            if ( ! IsPressWindowControlKey( pad ) || ! pad.IsHold( nw::dev::Pad::MASK_A ) )
            {
                m_HoldWindow = NULL;
            }
        }
        else
        {
            if ( ! IsPressWindowControlKey( pad ) || ! pad.IsHold( nw::dev::Pad::MASK_X ) )
            {
                m_HoldWindow = NULL;
            }
        }
    }
}

// -----------------------------------------------------------
void
WindowManager::UpdatePointerWindow(const Inputs& inputs)
{
    Window* pNewPointerWindow = GetPointerWindow();

    if(pNewPointerWindow == m_PointerWindow)
    {
        return;
    }

    if(m_PointerWindow != NULL)
    {
        // ウィンドウが重なっている場合にも PointerLeave を発生させるために、
        // ポインタをウィンドウ領域外にとばします。
        m_PointerWindow->UpdateInputs(
            Inputs(inputs).SetPointerPosition(-1.f, -1.f));
    }

    m_PointerWindow = pNewPointerWindow;
}

// -----------------------------------------------------------
void
WindowManager::UpdateActiveWindow()
{
    if ( m_HoldWindow != NULL ) {
        SetActiveWindow( m_HoldWindow );
        m_ActiveWindowToTop = true;
    }
    else if ( Window* pointerWindow = GetPointerWindow() ) {
        SetActiveWindow( pointerWindow );
        m_ActiveWindowToTop = IsPointerOverWindow( pointerWindow, true );
    }
    else {
        SetActiveWindow( NULL );
        m_ActiveWindowToTop = false;
    }
}

// -----------------------------------------------------------
bool
WindowManager::IsPressWindowControlKey( const nw::dev::Pad& pad ) const
{
    if ( ! m_EnableWindowControl ) return false;
    return pad.IsHoldAll( nw::internal::dw::InputSettings::GetInstance().GetWindowManagerSetting().GetWindowControlModifier() );
}

// -----------------------------------------------------------
Window*
WindowManager::GetWindowByIndex( int specifiedIndex )
{
    int index = 0;
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++ )
    {
        if ( index == specifiedIndex ) return &*itr;
        index++;
    }
    return NULL;
}

// -----------------------------------------------------------
int
WindowManager::GetWindowIndex( Window* window )
{
    int index = 0;
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++ )
    {
        if ( window == &*itr ) return index;
        index++;
    }
    return -1;
}

// -----------------------------------------------------------
bool
WindowManager::IsPointerOverWindow( const Window* window, bool onTitleBar ) const
{
    NW_ASSERT_NOT_NULL( window );

    if ( ! window->GetIsVisible() ) return false;

    const nw::math::Vector2& pos = window->GetPosition();
    nw::math::Vector2 size = window->GetSize();
    if ( onTitleBar ) {
        size.y = window->GetTitleBarHeight();
    }

    const nw::math::Vector2& pointerPos = m_Pointer.GetPosition();
    if (
        pos.x <= pointerPos.x && pointerPos.x <= pos.x + size.x &&
        pos.y <= pointerPos.y && pointerPos.y <= pos.y + size.y
    )
    {
        return true;
    }

    return false;
}

// -----------------------------------------------------------
Window*
WindowManager::GetPointerWindow()
{
    if ( m_ActiveWindow != NULL && m_ActiveWindowToTop )
    {
        // アクティブウィンドウを一時的に最前面へ移動
        if ( IsPointerOverWindow( m_ActiveWindow, true ) ) {
            return m_ActiveWindow;
        }
    }

    for ( WindowZOrderList::Iterator itr = m_WindowDrawList.GetBeginIter();
          itr != m_WindowDrawList.GetEndIter() ; itr++ )
    {
        Window* window = &*itr;

        if ( IsPointerOverWindow( window, window->GetIsShade() ) ) {
            return window;
        }
    }
    return NULL;
}

// -----------------------------------------------------------
void
WindowManager::UpdateOrder()
{
    int index = 0;
    for ( WindowZOrderList::Iterator itr = m_WindowDrawList.GetBeginIter();
          itr != m_WindowDrawList.GetEndIter() ; itr++, index++ )
    {
        itr->detail_SetWindowZOrder( index );
    }

    index = 0;
    for ( WindowList::Iterator itr = m_WindowList.GetBeginIter();
          itr != m_WindowList.GetEndIter() ; itr++, index++ )
    {
        itr->detail_SetWindowListOrder( index );
    }
}

} // namespace nw::internal::dw
} // namespace nw::internal
} // namespace nw

