﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/window/dw_WindowLayout.h>
#include <nw/dw/window/dw_WindowManager.h>

#if defined(WIN32)
#pragma warning(disable:4815)
#endif

namespace nw {
namespace internal {
namespace dw {

// -----------------------------------------------------------
bool WindowLayoutBase::Constract( const WindowManager& manager )
{
#ifdef NW_COMPILER_MSVC
#pragma warning(push)
#pragma warning(disable : 4996)
#endif
    m_ValidFlag = false;

    const WindowList& windowList = manager.GetWindowList();

    m_WindowStateCount = static_cast<int>( windowList.GetSize() );

    int index = 0;
    for ( WindowList::ConstIterator itr = windowList.GetBeginIter();
          itr != windowList.GetEndIter() ; ++itr, ++index )
    {
        WindowState* windowState = GetWindowState( index );
        if ( windowState == NULL ) return false;

        std::strncpy( windowState->title, itr->GetTitle(), sizeof(windowState->title) );
        windowState->title[ sizeof(windowState->title)-1 ] = '\0';
        windowState->pos = itr->GetPosition();
        windowState->size = itr->GetSize();
        windowState->order = itr->detail_GetWindowListOrder();
        windowState->zorder = itr->detail_GetWindowZOrder();
        windowState->visible = itr->GetIsVisible();
        windowState->shade = itr->GetIsShade();
    }

    m_ValidFlag = true;

#ifdef NW_COMPILER_MSVC
#pragma warning(pop)
#endif

    return true;
}

// -----------------------------------------------------------
bool WindowLayoutBase::RestoreLayout( WindowManager* manager ) const
{
    NW_NULL_ASSERT( manager );

    if ( ! m_ValidFlag ) {
        NW_WARNING( false, "WindowLayout is incomplete" );
        return false;
    }

    const nw::math::Vector2& screenSize = manager->GetScreenSize();

    WindowList& windowList = manager->GetWindowList();

    for( int index = 0 ; index < m_WindowStateCount ; ++index ) {
        const WindowState* windowState = GetWindowState( index );
        if ( windowState == NULL ) return false;

        Window* window = FindWindow( windowList, windowState->title );
        if ( window == NULL ) continue;

        f32 left = windowState->pos.x;
        f32 right = left + windowState->size.x;
        f32 top = windowState->pos.y;
        f32 bottom = top + windowState->size.y;

        if ( left < screenSize.x &&
            right > 0 &&
            top < screenSize.y &&
            bottom > 0 )
        {
            window->SetPosition( windowState->pos );
            window->SetSize( windowState->size );
        }
        window->detail_SetWindowListOrder( windowState->order );
        window->detail_SetWindowZOrder( windowState->zorder );
        window->SetIsVisible( windowState->visible );
        window->SetIsShade( windowState->shade );
    }

    manager->detail_SortWindowList();
    manager->detail_SortWindowZOrderList();

    return true;
}

// -----------------------------------------------------------
Window* WindowLayoutBase::FindWindow( WindowList& windowList, const char* windowTitle )
{
    for ( WindowList::Iterator itr = windowList.GetBeginIter();
          itr != windowList.GetEndIter() ; itr++ )
    {
        Window* window = &*itr;
        NW_NULL_ASSERT( window );

        if ( std::strcmp( window->GetTitle(), windowTitle ) == 0 ) {
            return window;
        }
    }

    return NULL;
}

// -----------------------------------------------------------
bool WindowLayoutBase::Serialize( ut::FileStream* fileStream ) const
{
    NW_NULL_ASSERT( fileStream );

    if ( ! m_ValidFlag ) {
        NW_WARNING( false, "WindowLayout is incomplete" );
        return false;
    }

    if ( ! fileStream->CanWrite() ) {
        NW_WARNING( false, "FileStream is not writable" );
        return false;
    }

    const u32 headerSize = sizeof(FileHeader);

    // WindowState配列のオフセット（ブロック先頭からの相対）
    const u32 windowStateDataOffset = ut::RoundUp( sizeof( DataBlock ) + sizeof(u32)*m_WindowStateCount, 32 );

    const u32 blockSize = windowStateDataOffset + sizeof(WindowStateData) * m_WindowStateCount ;

    const u32 fileSize = headerSize + blockSize;

    {
        FileHeader fileHeader;
        fileHeader.common.signature = FILE_HEADER_SIGNATURE;
        fileHeader.common.byteOrder = 0xfeff;
        fileHeader.common.version = FILE_VERSION;
        fileHeader.common.fileSize = fileSize;
        fileHeader.common.headerSize = headerSize;
        fileHeader.common.dataBlocks = 1;
        std::memset( fileHeader.reserved, 0, sizeof(fileHeader.reserved) );

        int writeSize = sizeof(fileHeader);
        if ( fileStream->Write( &fileHeader, static_cast<u32>(writeSize) ) != writeSize ) {
            NW_WARNING( false, "Failed to write file" );
            return false;
        }
    }

    u32 blockBeginOffset = fileStream->Tell();
    {
        DataBlock dataBlock;
        std::memset( &dataBlock, 0, sizeof( dataBlock ) );
        dataBlock.blockHeader.kind = BLOCK_KIND;
        dataBlock.blockHeader.size = blockSize;
        dataBlock.windowCount = static_cast<u32>(m_WindowStateCount);

        int writeSize = sizeof(dataBlock);
        if ( fileStream->Write( &dataBlock, static_cast<u32>(writeSize) ) != writeSize ) {
            NW_WARNING( false, "Failed to write file" );
            return false;
        }
    }

    // オフセットテーブルの出力
    {
        u32 offset = windowStateDataOffset;
        int windowIndex = 0;
        while ( windowIndex < m_WindowStateCount ){
            static const int BUFFER_COUNT = 16;
            int writeCount = 0;
            u32 windowStateOffset[ BUFFER_COUNT ];
            for( int i = 0 ;
                 i < BUFFER_COUNT && windowIndex < m_WindowStateCount ;
                 i++, windowIndex++, writeCount++ )
            {
                NW_ALIGN32_ASSERT( offset );
                windowStateOffset[i] = offset;
                offset += sizeof( WindowStateData );
            }

            int writeSize = static_cast<int>( sizeof(windowStateOffset[0]) * writeCount );
            if ( fileStream->Write( windowStateOffset, static_cast<u32>(writeSize) ) != writeSize ) {
                NW_WARNING( false, "Failed to write file" );
                return false;
            }
        }
    }

    // アライメント
    {
        u32 tell = fileStream->Tell();
        int fraction = static_cast<int>( tell % 32 );
        if ( fraction > 0 ) {
            u8 data[32];
            std::memset( data, 0, sizeof(data) );

            int writeSize = 32 - fraction;
            if ( fileStream->Write( data, static_cast<u32>(writeSize) ) != writeSize ) {
                NW_WARNING( false, "Failed to write file" );
                return false;
            }
        }
    }

    NW_ASSERT( windowStateDataOffset == fileStream->Tell() - blockBeginOffset );

    // WindowStateデータの出力
    {
        for( int windowIndex = 0 ; windowIndex < m_WindowStateCount ; windowIndex++ )
        {
            const WindowState* windowState = GetWindowState( windowIndex );
            if ( windowState == NULL ) return false;

            WindowStateData windowStateData;
            std::memset( &windowStateData, 0, sizeof(windowStateData) );
        #ifdef NW_COMPILER_MSVC
        #pragma warning(push)
        #pragma warning(disable : 4996)
        #endif
            std::strncpy(
                windowStateData.title,
                windowState->title,
                sizeof(windowStateData.title)
            );
        #ifdef NW_COMPILER_MSVC
        #pragma warning(pop)
        #endif
            windowStateData.title[ sizeof(windowStateData.title)-1 ] = '\0';

            windowStateData.x = windowState->pos.x;
            windowStateData.y = windowState->pos.y;
            windowStateData.width = windowState->size.x;
            windowStateData.height = windowState->size.y;
            windowStateData.visible = static_cast<u8>( windowState->visible ? 1 : 0 );
            windowStateData.shade = static_cast<u8>( windowState->shade ? 1 : 0 );
            windowStateData.order = windowState->order;
            windowStateData.zorder = windowState->zorder;

            int writeSize = sizeof(windowStateData);
            if ( fileStream->Write( &windowStateData, static_cast<u32>(writeSize) ) != writeSize ) {
                NW_WARNING( false, "Failed to write file" );
                return false;
            }
        }
    }
    u32 blockEndOffset = fileStream->Tell();

    NW_ALIGN32_ASSERT( blockEndOffset );
    NW_ASSERT( blockBeginOffset + blockSize == blockEndOffset );
    NW_ASSERT( fileSize == blockEndOffset );

    return true;
}

// -----------------------------------------------------------
bool WindowLayoutBase::Deserialize( ut::FileStream* fileStream )
{
    NW_NULL_ASSERT( fileStream );

    m_ValidFlag = false;

    if ( ! fileStream->CanRead() ) {
        NW_WARNING( false, "FileStream is not readable" );
        return false;
    }

    {
        FileHeader fileHeader /*ATTRIBUTE_ALIGN(32)*/; // todo
        int readSize = sizeof(fileHeader);
        if ( fileStream->Read( &fileHeader, static_cast<u32>(readSize) ) != readSize ) {
            NW_WARNING( false, "Failed to read file" );
            return false;
        }

        if ( ! ut::IsValidBinaryFile( &fileHeader.common, FILE_HEADER_SIGNATURE, FILE_VERSION ) )
        {
            NW_WARNING( false, "Unsupported file format" );
            return false;
        }
    }

    u32 dataBlockOffset = fileStream->Tell();
    {
        DataBlock dataBlock /*ATTRIBUTE_ALIGN(32)*/; // todo
        int readSize = sizeof(dataBlock);
        if ( fileStream->Read( &dataBlock, static_cast<u32>(readSize) ) != readSize ) {
            NW_WARNING( false, "Failed to read file" );
            return false;
        }

        if ( dataBlock.blockHeader.kind != BLOCK_KIND ) {
            NW_WARNING( false, "Cannot find WNDS block" );
            return false;
        }

        m_WindowStateCount = static_cast<int>(dataBlock.windowCount);
    }

    {
        int windowIndex = 0;
        while ( windowIndex < m_WindowStateCount )
        {
            static const int BUFFER_COUNT = 16;
            u32 windowStateOffset[ BUFFER_COUNT ] /*ATTRIBUTE_ALIGN(32)*/; // todo

            int readCount = ut::Min( m_WindowStateCount-windowIndex, BUFFER_COUNT );
            int readSize = static_cast<int>( readCount * sizeof(u32) );
            int readSizeRoundUp = ut::RoundUp( readSize, 32 );
            if ( fileStream->Read( windowStateOffset, static_cast<u32>(readSizeRoundUp) ) != readSizeRoundUp ) {
                NW_WARNING( false, "Failed to read file" );
                return false;
            }

            fileStream->Seek( readSize - readSizeRoundUp, ut::FILE_STREAM_SEEK_CURRENT );
            u32 readPosStack = fileStream->Tell();

            for( int i=0; i < readCount; i++, windowIndex++ ) {
                WindowState* windowState = GetWindowState( windowIndex );
                if ( windowState == NULL ) {
                    NW_WARNING( false, "Failed to GetWindowState ( index = %d )", windowIndex );
                    return false;
                }

                u32 dataOffset = windowStateOffset[i];

                fileStream->Seek(
                    static_cast<s32>(dataBlockOffset + dataOffset),
                    ut::FILE_STREAM_SEEK_BEGIN
                );
                WindowStateData windowStateData /*ATTRIBUTE_ALIGN(32)*/; // todo
                int winStateReadSize = sizeof(windowStateData);
                if ( fileStream->Read( &windowStateData, static_cast<u32>(winStateReadSize) ) != winStateReadSize) {
                    NW_WARNING( false, "Failed to read file" );
                    return false;
                }

            #ifdef NW_COMPILER_MSVC
            #pragma warning(push)
            #pragma warning(disable : 4996)
            #endif
                std::strncpy(
                    windowState->title,
                    windowStateData.title,
                    sizeof(windowState->title)
                );
            #ifdef NW_COMPILER_MSVC
            #pragma warning(pop)
            #endif
                windowState->title[ sizeof(windowState->title)-1 ] = '\0';

                windowState->pos.x = windowStateData.x;
                windowState->pos.y = windowStateData.y;
                windowState->size.x = windowStateData.x;
                windowState->size.y = windowStateData.y;
                windowState->visible = ( windowStateData.visible != 0 );
                windowState->shade = ( windowStateData.shade != 0 );
                windowState->order = windowStateData.order;
                windowState->zorder = windowStateData.zorder;
            }

            fileStream->Seek( static_cast<s32>(readPosStack), ut::FILE_STREAM_SEEK_BEGIN );
        }
    }

    m_ValidFlag = true;

    return true;
}

// -----------------------------------------------------------
void WindowLayoutBase::Dump() const
{
#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
    // TODO: NintendoSdk 対応後、このコメントを削除してください。
    using namespace nw::internal::winext;
#endif
    OSReport( "*** WindowLayoutBase::Dump ***\n" );

    if ( ! m_ValidFlag ) {
        OSReport( " Constract is incomplete." );
        return;
    }

    for( int index = 0; index < m_WindowStateCount ; index++ ) {
        const WindowState* windowState = GetWindowState( index );

        OSReport(
            "%s: %d %d %d %d %s %s\n",
            windowState->title,
            windowState->pos.x,
            windowState->pos.y,
            windowState->size.x,
            windowState->size.y,
            windowState->visible ? "VISIBLE" : "HIDE",
            windowState->shade ? "SHADE" : "NORMAL"
        );
    }
}

} // namespace nw::internal::dw
} // namespace nw::internal
} // namespace nw

