﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/window/dw_Window.h>
#include <nw/dw/window/dw_WindowConfig.h>
#include <nw/dw/system/dw_NwTypeUtility.h>
#include <nw/dev/dev_PrimitiveRenderer.h>
#include <nw/gfnd.h>

// -----------------------------------------------------------

namespace nw {
namespace internal {
namespace dw {

const f32 Window::FRAME_WIDTH       = 1.f;                               // フレームの幅
const f32 Window::TITLE_HEIGHT      = 18.f;                              // タイトルの高さ
const f32 Window::CLIENT_OUT_WIDTH  = FRAME_WIDTH * 2.f;                 // クライアント領域外の幅
const f32 Window::CLIENT_OUT_HEIGHT = FRAME_WIDTH * 3.f + TITLE_HEIGHT;  // クライアント領域外の高さ

const f32 Window::WIDTH_MAX  = 1000.f;
const f32 Window::WIDTH_MIN  = 60.f;
const f32 Window::HEIGHT_MAX = 700.f;
const f32 Window::HEIGHT_MIN = TITLE_HEIGHT;
const f32 Window::TITLE_PADDING_LEFT = 8.f;

Window::Window()
  : m_IsCreated(false),
    m_IsActive(true),
    m_IsShade(false),
    m_IsVisible(true),
    m_HasCloseButton(false),
    m_IsClosed(false),
    m_BackgroundColor0(internal::WindowConfig::GetInstance().GetDefaultBackgroundColor0()),
    m_BackgroundColor1(internal::WindowConfig::GetInstance().GetDefaultBackgroundColor1()),
    m_Pos(),
    m_Size(100.f, 100.f),
    m_MinimumSize(WIDTH_MIN, HEIGHT_MIN),
    m_MaximumSize(WIDTH_MAX, HEIGHT_MAX),
    m_WindowOrder(0),
    m_ZOrder(0),
    m_WindowListLink(),
    m_ZOrderLink()
{
    SetTitle("");
}

Window::Window( const char* title, f32 width, f32 height )
  : m_IsCreated(false),
    m_IsActive(true),
    m_IsShade(false),
    m_IsVisible(true),
    m_HasCloseButton(false),
    m_IsClosed(false),
    m_BackgroundColor0(internal::WindowConfig::GetInstance().GetDefaultBackgroundColor0()),
    m_BackgroundColor1(internal::WindowConfig::GetInstance().GetDefaultBackgroundColor1()),
    m_Pos(),
    m_Size(width, height),
    m_MinimumSize(WIDTH_MIN, HEIGHT_MIN),
    m_MaximumSize(WIDTH_MAX, HEIGHT_MAX),
    m_WindowOrder(0),
    m_ZOrder(0),
    m_WindowListLink(),
    m_ZOrderLink()
{
    SetTitle(title);
}

// -----------------------------------------------------------
Window::~Window()
{
    Destroy();
}

// -----------------------------------------------------------
bool
Window::Create( f32 x, f32 y )
{
    return Create(nw::math::Vector2(x, y));
}

// -----------------------------------------------------------
bool
Window::Create( const nw::math::Vector2& pos )
{
    if (m_IsCreated)
    {
        Destroy();
    }

    if (!OnCreate())
    {
        return false;
    }

    SetPosition(pos);

    m_IsCreated = true;

    return true;
}

// -----------------------------------------------------------
void
Window::Destroy()
{
    if (m_IsCreated)
    {
        OnDestroy();
        m_IsCreated = false;
    }
}

// -----------------------------------------------------------
void
Window::UpdateInputs(const nw::internal::dw::Inputs& inputs)
{
    NW_ASSERT(m_IsCreated);

    // 「閉じる」ボタンが押されたかを判定します。
    m_IsClosed = false;
    if ( m_HasCloseButton )
    {
        const nw::dev::Mouse* mouse = inputs.GetMouse();

        if ( mouse && mouse->IsClick( nw::dev::Mouse::MASK_LBUTTON ) )
        {
            const f32 left        = m_Pos.x + FRAME_WIDTH / 2;
            const f32 top         = m_Pos.y + FRAME_WIDTH / 2;
            const f32 width       = m_Size.x - FRAME_WIDTH;
            const f32 titleHeight = TITLE_HEIGHT + FRAME_WIDTH;

            const f32 closeBoxSize = titleHeight - 2.f;
            const f32 closeBoxPosX = left + width - closeBoxSize - 2.f;
            const f32 closeBoxPosY = top + 1.f;

            if ( closeBoxPosX <= mouse->GetPointer().x && mouse->GetPointer().x <= closeBoxPosX + closeBoxSize &&
                 closeBoxPosY <= mouse->GetPointer().y && mouse->GetPointer().y <= closeBoxPosY + closeBoxSize )
            {
                m_IsClosed = true;
            }
        }
    }

    OnUpdateInputs(inputs);
}

// -----------------------------------------------------------
void
Window::Update(UIRenderer& renderer)
{
    NW_ASSERT(m_IsCreated);
    OnUpdate(renderer);
}

// -----------------------------------------------------------
void
Window::Draw( WindowContext& context, UIRenderer& renderer )
{
    NW_ASSERT(m_IsCreated);
    DrawImpl(context, renderer, 1.0f, m_IsShade);
}

// -----------------------------------------------------------
void
Window::Draw( WindowContext& context, UIRenderer& renderer, f32 alpha, bool isShade )
{
    NW_ASSERT(m_IsCreated);
    DrawImpl(context, renderer, alpha, isShade);
}

// -----------------------------------------------------------
void
Window::SetSize( f32 width, f32 height )
{
    m_Size.x = width;
    m_Size.y = height;

    m_Size.x = ut::Max(m_Size.x, m_MinimumSize.x);
    m_Size.x = ut::Min(m_Size.x, m_MaximumSize.x);

    m_Size.y = ut::Max(m_Size.y, m_MinimumSize.y);
    m_Size.y = ut::Min(m_Size.y, m_MaximumSize.y);

    OnSize();
}

// -----------------------------------------------------------
void
Window::SetSize( const nw::math::Vector2& size )
{
    SetSize(size.x, size.y);
}

// -----------------------------------------------------------
void
Window::SetSizeFromClientSize( f32 width, f32 height )
{
    SetSize(
        width + FRAME_WIDTH * 2,
        height + FRAME_WIDTH * 2 + TITLE_HEIGHT + FRAME_WIDTH);
}

void
Window::SetSizeFromClientSize( const nw::math::Vector2& clientSize )
{
    SetSize(
        clientSize.x + FRAME_WIDTH * 2,
        clientSize.y + FRAME_WIDTH * 2 + TITLE_HEIGHT + FRAME_WIDTH);
}

// -----------------------------------------------------------
const ut::Rect
Window::GetClientRect() const
{
    f32 x = m_Pos.x + FRAME_WIDTH;
    f32 y = m_Pos.y + FRAME_WIDTH * 2 + TITLE_HEIGHT;

    return ut::Rect(
        x,
        y,
        x + m_Size.x - CLIENT_OUT_WIDTH,
        y + m_Size.y - CLIENT_OUT_HEIGHT
    );
}

// -----------------------------------------------------------
const nw::math::Vector2
Window::GetClientSize() const
{
    nw::math::Vector2 clientSize;

    clientSize.x = m_Size.x - CLIENT_OUT_WIDTH;
    clientSize.y = m_Size.y - CLIENT_OUT_HEIGHT;

    return clientSize;
}

// -----------------------------------------------------------
const nw::math::Vector2
Window::MeasureTitle(UIRenderer& renderer) const
{
    nw::math::Vector2 size = renderer.MeasureText(DrawTextArgs(), this->GetTitle());
    size.x += TITLE_PADDING_LEFT;
    size.y = TITLE_HEIGHT;
    return size;
}

// -----------------------------------------------------------
void
Window::SetTitle( const char* title )
{
    nw::ut::strcpy(m_Title, TITLE_LEN_MAX - 1, title);
}

// -----------------------------------------------------------
void
Window::DrawImpl( WindowContext& context, UIRenderer& renderer, f32 alpha, bool isShade )
{
    renderer.BeginDraw();

    // ウィンドウフレームの描画
    DrawFrame(context, renderer, alpha, isShade);

    // ウィンドウタイトルの描画
    DrawTitle(context, renderer, alpha, isShade);

    renderer.EndDraw();

    const Window* pPreviousWindow = context.PushWindow(*this, renderer);
    renderer.BeginDraw();

    // クライアント領域の描画
    if (!isShade)
    {
        OnDraw( context, renderer );
    }

    renderer.EndDraw();
    context.PopWindow(pPreviousWindow, renderer);
}

// -----------------------------------------------------------
void
Window::DrawFrame( WindowContext& context, UIRenderer& renderer, f32 alpha, bool isShade )
{
    gfnd::Graphics::GetInstance()->SetScissor(
        m_Pos.x,
        m_Pos.y,
        m_Size.x,
        m_Size.y,
        context.GetScreenSize().x,
        context.GetScreenSize().y);

    gfnd::Graphics::GetInstance()->SetViewport(
        0.0f,
        0.0f,
        context.GetScreenSize().x,
        context.GetScreenSize().y,
        0.f,
        1.f,
        context.GetScreenSize().x,
        context.GetScreenSize().y);

    const internal::WindowConfig& config = internal::WindowConfig::GetInstance();

    ut::Color4f titleColor0      = m_IsActive ? config.GetActiveTitlebarColor0() : config.GetInactiveTitlebarColor0();
    ut::Color4f titleColor1      = m_IsActive ? config.GetActiveTitlebarColor1() : config.GetInactiveTitlebarColor1();
    ut::Color4f borderColor      = m_IsActive ? config.GetActiveBorderColor() : config.GetInactiveBorderColor();
    ut::Color4f backgroundColor0 = m_BackgroundColor0;
    ut::Color4f backgroundColor1 = m_BackgroundColor1;

    titleColor0.a *= alpha;
    titleColor1.a *= alpha;
    borderColor.a *= alpha;
    backgroundColor0.a *= alpha;
    backgroundColor1.a *= alpha;

    const f32 left        = m_Pos.x + FRAME_WIDTH / 2;
    const f32 top         = m_Pos.y + FRAME_WIDTH / 2;
    const f32 width       = m_Size.x - FRAME_WIDTH;
    const f32 height      = m_Size.y - FRAME_WIDTH;
    const f32 titleHeight = TITLE_HEIGHT + FRAME_WIDTH;

    // ウィンドウ背景
    if (!isShade)
    {
        renderer.FillRectangle(
            NULL,
            DrawRectangleArgs().
                SetTopLeft(left, top + titleHeight).
                SetSize(width, height - titleHeight).
                SetColor(backgroundColor0, backgroundColor1));
    }

    // タイトル背景
    renderer.FillRectangle(
        NULL,
        DrawRectangleArgs().
            SetTopLeft(left, top).
            SetSize(width, titleHeight).
            SetColor(titleColor0, titleColor1));

    // タイトル枠
    renderer.DrawRectangle(
        NULL,
        DrawRectangleArgs().
            SetTopLeft(left, top).
            SetSize(width, titleHeight).
            SetColor(borderColor));

    // ウィンドウ枠
    if (!isShade)
    {
        renderer.DrawRectangle(
            NULL,
            DrawRectangleArgs().
                SetTopLeft(left, top).
                SetSize(width, height).
                SetColor(borderColor));
    }

    DrawCloseButton(renderer, alpha);

    // シェードマーク
    if (m_IsShade)
    {
/*      ★TODO
        const int MARGIN_RIGHT = 4;
        const int MARGIN_TOP = 4;
        const int MARGIN_BOTTOM = 4;
        const int WIDTH = 10;

        Point points[3];
        points[0].x = left+width-WIDTH-MARGIN_RIGHT;
        points[0].y = top+MARGIN_TOP;
        points[1].x = left+width-MARGIN_RIGHT;
        points[1].y = top+MARGIN_TOP;
        points[2].x = left+width-WIDTH/2-MARGIN_RIGHT;
        points[2].y = top+titleHeight-MARGIN_BOTTOM;

        gfx.FillTriangle( points[0],points[1],points[2], borderColor );
*/
    }
}

// -----------------------------------------------------------
void
Window::DrawTitle( WindowContext& context, UIRenderer& renderer, f32 alpha, bool isShade )
{
    (void)isShade;

    gfnd::Graphics::GetInstance()->SetScissor(
        m_Pos.x + FRAME_WIDTH,
        m_Pos.y + FRAME_WIDTH,
        m_Size.x - FRAME_WIDTH * 2,
        TITLE_HEIGHT,
        context.GetScreenSize().x,
        context.GetScreenSize().y);

    gfnd::Graphics::GetInstance()->SetViewport(
        0.0f,
        0.0f,
        context.GetScreenSize().x,
        context.GetScreenSize().y,
        0.f,
        1.f,
        context.GetScreenSize().x,
        context.GetScreenSize().y);

    const internal::WindowConfig& config = internal::WindowConfig::GetInstance();

    nw::ut::Color4f color = m_IsActive ? config.GetActiveTitleColor() : config.GetInactiveTitleColor();
    color.a *= alpha;

    renderer.DrawText(
        NULL,
        DrawTextArgs().
        SetTopLeft(m_Pos.x + FRAME_WIDTH + TITLE_PADDING_LEFT, m_Pos.y + FRAME_WIDTH).
        SetColor(color).
        SetScale(1.f),
        m_Title);
}

// -----------------------------------------------------------
void
Window::DrawCloseButton( UIRenderer& renderer, f32 alpha )
{
    if ( m_HasCloseButton )
    {
        const internal::WindowConfig& config = internal::WindowConfig::GetInstance();

        ut::Color4f borderColor      = m_IsActive ? config.GetActiveBorderColor() : config.GetInactiveBorderColor();
        ut::Color4f closeBackColor0  = NwTypeUtility::SRGBToLinear( nw::ut::Color4f( 0.827f, 0.596f, 0.56f, 1.f ) );
        ut::Color4f closeBackColor1  = NwTypeUtility::SRGBToLinear( nw::ut::Color4f( 0.577f, 0.185f, 0.123f, 1.f ) );
        ut::Color4f closeBoxColor    = m_IsActive ? NwTypeUtility::SRGBToLinear( nw::ut::Color4f( 0.7f, 0.7f, 0.7f, 1.f ) ) : NwTypeUtility::SRGBToLinear( nw::ut::Color4f( 0.45f, 0.48f, 0.58f, 1.f ) );
        ut::Color4f closeMarkColor   = m_IsActive ? config.GetActiveTitleColor() : NwTypeUtility::SRGBToLinear( nw::ut::Color4f( 0.45f, 0.48f, 0.58f, 1.f ) );

        borderColor.a *= alpha;
        closeBackColor0.a *= alpha;
        closeBackColor1.a *= alpha;

        const f32 left        = m_Pos.x + FRAME_WIDTH / 2;
        const f32 top         = m_Pos.y + FRAME_WIDTH / 2;
        const f32 width       = m_Size.x - FRAME_WIDTH;
        const f32 titleHeight = TITLE_HEIGHT + FRAME_WIDTH;

        const f32 closeBoxSize = titleHeight - 2.f;
        const f32 closeBoxPosX = left + width - closeBoxSize - 2.f;
        const f32 closeBoxPosY = top + 1.f;

        if (m_IsActive)
        {
            DrawRectangleArgs arg;
            arg.SetTopLeft(closeBoxPosX, closeBoxPosY);
            arg.SetSize(closeBoxSize, closeBoxSize);
            arg.SetColor(closeBackColor0, closeBackColor1);
            renderer.FillRectangle(NULL, arg);
        }

        if (m_IsActive)
        {
            DrawRectangleArgs arg;
            arg.SetTopLeft(closeBoxPosX + 1.f, closeBoxPosY + 1.f);
            arg.SetSize( closeBoxSize, closeBoxSize );
            arg.SetColor(borderColor);
            renderer.DrawRectangle(NULL, arg);
        }

        {
            DrawRectangleArgs arg;
            arg.SetTopLeft(closeBoxPosX, closeBoxPosY);
            arg.SetSize( closeBoxSize, closeBoxSize );
            arg.SetColor(closeBoxColor);
            renderer.DrawRectangle(NULL, arg);
        }

        if (m_IsActive)
        {
            DrawLineArgs arg;
            arg.lineWidth = 3.f;
            arg.SetColor(borderColor);

            {
                arg.SetFrom( nw::math::VEC2( 5.f, 5.f ) );
                arg.SetTo( nw::math::VEC2( 13.5f, 13.5f ) );
                arg.Offset(closeBoxPosX + 1.f, closeBoxPosY + 1.f);
                renderer.DrawLine(NULL, arg);
            }

            {
                arg.SetFrom( nw::math::VEC2( 13.f, 5.f ) );
                arg.SetTo( nw::math::VEC2( 4.5f, 13.5f ) );
                arg.Offset(closeBoxPosX + 1.f, closeBoxPosY + 1.f);
                renderer.DrawLine(NULL, arg);
            }
        }

        {
            DrawLineArgs arg;
            arg.lineWidth = 3.f;
            arg.SetColor(closeMarkColor);

            {
                arg.SetFrom( nw::math::VEC2( 5.f, 5.f ) );
                arg.SetTo( nw::math::VEC2( 13.5f, 13.5f ) );
                arg.Offset(closeBoxPosX, closeBoxPosY);
                renderer.DrawLine(NULL, arg);
            }

            {
                arg.SetFrom( nw::math::VEC2( 13.f, 5.f ) );
                arg.SetTo( nw::math::VEC2( 4.5f, 13.5f ) );
                arg.Offset(closeBoxPosX, closeBoxPosY);
                renderer.DrawLine(NULL, arg);
            }
        }
    }
}

} // namespace dw
} // namespace internal
} // namespace nw

