﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_Separator.h>

namespace nw {
namespace internal {
namespace dw {

Separator::Separator() :
m_Orientation(HORIZONTAL)
{
    SetIsFocusable(false);
    SetMargin(Thickness(2.f));
    SetPadding(Thickness(0.f));
    SetBorderColor(nw::ut::Color4f::X_WHITE_SMOKE());
}

Orientation Separator::GetOrientation() const
{
    return m_Orientation;
}

void Separator::SetOrientation(Orientation value)
{
    m_Orientation = value;
}

const nw::math::Vector2 Separator::OnMeasure(UIRenderer& renderer) const
{
    (void)renderer;

    switch(m_Orientation)
    {
    case HORIZONTAL:
        return MeasureByHorizontal();

    case VERTICAL:
        return MeasureByVertical();

    default:
        NW_ASSERT(false);
        return nw::math::Vector2::Zero();
    }
}

void Separator::OnRender(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
{
    switch(m_Orientation)
    {
    case HORIZONTAL:
        RenderByHorizontal(context, renderer, args);
        break;

    case VERTICAL:
        RenderByVertical(context, renderer, args);
        break;

    default:
        NW_ASSERT(false);
    }
}

bool Separator::CanAlignHorizontal(const UIElement& element) const
{
    (void)element;

    // 横方向 Separator の場合は、横方向のアライメントを無視します。
    return m_Orientation != HORIZONTAL;
}

bool Separator::CanAlignVertical(const UIElement& element) const
{
    (void)element;

    // 縦方向 Separator の場合は、縦方向のアライメントを無視します。
    return m_Orientation != VERTICAL;
}

nw::math::Vector2 Separator::MeasureByHorizontal() const
{
    Thickness padding = GetPadding();
    nw::math::Vector2 size = GetMeasuredSize();
    return nw::math::Vector2(padding.left + size.x, padding.top + LINE_WIDTH);
}

nw::math::Vector2 Separator::MeasureByVertical() const
{
    Thickness padding = GetPadding();
    nw::math::Vector2 size = GetMeasuredSize();
    return nw::math::Vector2(padding.left + LINE_WIDTH, padding.top + size.y);
}

void Separator::RenderByHorizontal(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
{
    nw::math::Vector2 topLeft = GetContentAreaTopLeft();
    nw::math::Vector2 size = GetContentAreaSize();

    renderer.DrawLine(
        &context,
        DrawLineArgs().
        SetFrom(topLeft.x, topLeft.y).
        SetTo(topLeft.x + size.x, topLeft.y).
        SetColor0(args.borderColor).
        SetColor1(args.borderColor));
}

void Separator::RenderByVertical(const UIElementTreeContext& context, UIRenderer& renderer, UIElementRenderArgs& args) const
{
    nw::math::Vector2 topLeft = GetContentAreaTopLeft();
    nw::math::Vector2 size = GetContentAreaSize();

    renderer.DrawLine(
        &context,
        DrawLineArgs().
        SetFrom(topLeft.x, topLeft.y).
        SetTo(topLeft.x, topLeft.y + size.y).
        SetColor0(args.borderColor).
        SetColor1(args.borderColor));
}

} // dw
} // internal
} // nw
