﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dw/control/dw_FileBrowser.h>

namespace nw {
namespace internal {
namespace dw {

const char* FileBrowser::PARENT_DIRECTORY_PATH = "../";

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
FileBrowser::FileBrowser() :
m_FileEntrySelectionChangedEventHandler(this, &FileBrowser::OnFileEntrySelectionChanged),
m_pFilterEventHandler(NULL),
m_pExecuteEventHandler(NULL)
{
    memset(m_CurrentDirectoryPath, 0, sizeof(m_CurrentDirectoryPath));

    SetContents(m_Contents);
    m_Contents.AddItem(&m_Entries);

    m_Entries.SetMargin(Thickness(0.f));
    m_Entries.SetPadding(Thickness(0.f));
    m_Entries.GetSelectionChangedEvent() += m_FileEntrySelectionChangedEventHandler;
}
#if defined(_WIN32)
#pragma warning(pop) // warning: used in base member initializer list
#endif

const char* FileBrowser::GetCurrentDirectoryPath() const
{
    return m_CurrentDirectoryPath;
}

void FileBrowser::SetCurrentDirectoryPath(const char* value)
{
    NW_NULL_ASSERT(value);
    nw::ut::strncpy(m_CurrentDirectoryPath, MAX_FILE_PATH + 1, value, strlen(value));
}

const char* FileBrowser::GetCurrentFilePath() const
{
    return m_CurrentFilePath;
}

FileBrowser::IFilterEventHandler* FileBrowser::GetFilterEventHandler() const
{
    return m_pFilterEventHandler;
}

void FileBrowser::SetFilterEventHandler(IFilterEventHandler* value)
{
    m_pFilterEventHandler = value;
}

FileBrowser::IExecuteEventHandler* FileBrowser::GetExecuteEventHandler() const
{
    return m_pExecuteEventHandler;
}

void FileBrowser::SetExecuteEventHandler(IExecuteEventHandler* value)
{
    m_pExecuteEventHandler = value;
}

void FileBrowser::SelectFileEntry(const char* pText)
{
    if(pText == NULL)
    {
        return;
    }

    for( s32 i = 0; i < m_Entries.GetItemCount(); i++)
    {
        FileBrowserEntry* pFileEntry = static_cast<FileBrowserEntry*>(m_Entries.GetItem(i));
        NW_NULL_ASSERT(pFileEntry);

        u32 length = strlen(pText);

        if(length > 0 && pText[length - 1] == PATH_SEPARATOR)
        {
            length--;
        }

        if(strncmp(pFileEntry->GetText(), pText, length) == 0)
        {
            ListBox::ListBoxItemPlaceHolder& itemPlaceHolder = m_Entries.GetItemPlaceHolder(i);
            itemPlaceHolder.SetIsSelected(true);
            itemPlaceHolder.SetFocus();
            itemPlaceHolder.EnsureVisible();
            break;
        }
    }
}

bool FileBrowser::UpdateFileEntries()
{
    ClearFileEntries();

    // ディレクトリへの移動エントリーを追加します。
    if(CanMoveParentDirectory(GetCurrentDirectoryPath()))
    {
        AddFileEntryImpl(PARENT_DIRECTORY_PATH, true);
    }

    BuildFileEntries(GetCurrentDirectoryPath());

    if(m_Entries.GetItemCount() == 0)
    {
        return true;
    }

    // 選択とフォーカスを再設定します。
    ListBox::ListBoxItemPlaceHolder& itemPlaceHolder = m_Entries.GetItemPlaceHolder(0);
    itemPlaceHolder.SetIsSelected(true);
    itemPlaceHolder.SetFocus();
    itemPlaceHolder.EnsureVisible();

    return true;
}

bool FileBrowser::AddFileEntry(const char* pText, bool isDirectory)
{
    NW_NULL_ASSERT(pText);

    if(m_pFilterEventHandler != NULL)
    {
        FileBrowserFilterEventArgs args(pText, isDirectory);
        m_pFilterEventHandler->Invoke(args);

        if(args.GetIsCanceled())
        {
            return true;
        }
    }

    return AddFileEntryImpl(pText, isDirectory);
}

void FileBrowser::MoveDirectory(const char* pName)
{
    NW_NULL_ASSERT(pName);

    const char* pOldEntryName = NULL;

    // 親ディレクトリへ移動
    if(strncmp(pName, PARENT_DIRECTORY_PATH, strlen(PARENT_DIRECTORY_PATH)) == 0 &&
        CanMoveParentDirectory(GetCurrentDirectoryPath()))
    {
        pOldEntryName = GetEntryName(m_CurrentDirectoryPath);

        bool result = MakeParentDirectoryPath(m_CurrentDirectoryPath);

        if(!result)
        {
            NW_LOG("[FileBrowser] failed to MakeParentDirectoryPath.\n");
            return;
        }
    }
    // 子ディレクトリへ移動
    else
    {
        nw::ut::snprintf(m_CurrentDirectoryPath, MAX_FILE_PATH + 1, MAX_FILE_PATH, "%s%c%s", m_CurrentDirectoryPath, PATH_SEPARATOR, pName);

        // 末尾の PATH_SEPARATOR を削除します。
        u32 length = strlen(m_CurrentDirectoryPath);

        if(m_CurrentDirectoryPath[length - 1] == PATH_SEPARATOR)
        {
            m_CurrentDirectoryPath[length - 1] = '\0';
        }
    }

    UpdateFileEntries();

    if(pOldEntryName != NULL)
    {
        SelectFileEntry(pOldEntryName);
    }
}

bool FileBrowser::OnUpdatePointerInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetMouse() == NULL)
    {
        return false;
    }

    const nw::dev::Mouse& mouse = *inputs.GetMouse();

    if(mouse.IsDoubleClick())
    {
        FileBrowserEntry* pSelectedEntry = GetSelectedEntryFirst();

        if(pSelectedEntry != NULL)
        {
            ExecuteFileEntry(*pSelectedEntry);
            return true;
        }
    }

    return false;
}

bool FileBrowser::OnUpdateFocusedInput(const nw::internal::dw::Inputs& inputs)
{
    if(inputs.GetPad() == NULL)
    {
        return false;;
    }

    const nw::dev::Pad& pad = *inputs.GetPad();

    if(pad.IsTrig(nw::dev::Pad::MASK_A))
    {
        FileBrowserEntry* pSelectedEntry = GetSelectedEntryFirst();

        if(pSelectedEntry != NULL)
        {
            ExecuteFileEntry(*pSelectedEntry);
        }

        return true;
    }

    if(pad.IsTrig(nw::dev::Pad::MASK_X))
    {
        if(CanMoveParentDirectory(GetCurrentDirectoryPath()))
        {
            MoveDirectory(PARENT_DIRECTORY_PATH);
        }

        return true;
    }

    return false;
}

bool FileBrowser::AddFileEntryImpl(const char* pText, bool isDirectory)
{
    NW_NULL_ASSERT(pText);

    if(m_Entries.GetItemCount() == MAX_FILE_ENTRIES)
    {
        return false;
    }

    u32 newIndex = m_Entries.GetItemCount();
    NW_ASSERT(newIndex < MAX_FILE_ENTRIES);

    FileBrowserEntry& entry = m_EntryElements[newIndex];
    entry.SetIsDirectory(isDirectory);

    u32 length = strlen(pText);
    NW_ASSERT(length > 0);

    // ディレクトリには末尾に PATH_SEPARATOR を付加します。
    if(isDirectory && pText[length - 1] != PATH_SEPARATOR)
    {
        char tempText[MAX_FILE_PATH + 1];
        nw::ut::snprintf(tempText, MAX_FILE_PATH + 1, MAX_FILE_PATH, "%s%c", pText, PATH_SEPARATOR);
        entry.SetText(tempText);
    }
    else
    {
        entry.SetText(pText);
    }

    m_Entries.AddItem(&entry);

    return true;
}

void FileBrowser::ClearFileEntries()
{
    m_Entries.ClearItems();
}

FileBrowserEntry* FileBrowser::GetSelectedEntryFirst() const
{
    for(s32 i=0; i<m_Entries.GetItemCount(); ++i)
    {
        const nw::internal::dw::ListBox::ListBoxItemPlaceHolder& placeHolder = m_Entries.GetItemPlaceHolder(i);

        if(placeHolder.GetIsSelected())
        {
            FileBrowserEntry* pFileEntry = static_cast<FileBrowserEntry*>(placeHolder.GetContent());
            NW_ASSERT(pFileEntry != NULL);

            return pFileEntry;
        }
    }

    return NULL;
}

void FileBrowser::ExecuteFileEntry(FileBrowserEntry& entry)
{
    if(entry.GetIsDirectory())
    {
        // ディレクトリ処理
        MoveDirectory(entry.GetText());
    }
    else if(m_pExecuteEventHandler != NULL)
    {
        // ファイル処理
        FileBrowserExecuteEventArgs args(entry, GetCurrentFilePath());
        m_pExecuteEventHandler->Invoke(args);
    }
}

const char* FileBrowser::GetEntryName(const char* pPath) const
{
    NW_NULL_ASSERT(pPath);

    u32 length = strlen(pPath);

    if(length == 0)
    {
        return NULL;
    }

    const char* pCurrent = pPath + length - 1;

    if(*pCurrent == PATH_SEPARATOR)
    {
        pCurrent--;
    }

    while(pCurrent != pPath)
    {
        if(*pCurrent == PATH_SEPARATOR)
        {
            return pCurrent + 1;
        }

        pCurrent--;
    }

    return NULL;
}

bool FileBrowser::MakeParentDirectoryPath(char* pPath) const
{
    NW_NULL_ASSERT(pPath);

    u32 length = strlen(pPath);

    if(length == 0)
    {
        return false;
    }

    if(pPath[length - 1] == PATH_SEPARATOR)
    {
        pPath[length - 1] = '\0';
    }

    char* pLastSeparator = strrchr(pPath, PATH_SEPARATOR);

    if(pLastSeparator == NULL)
    {
        return false;
    }

    *pLastSeparator = '\0';
    return true;
}

void FileBrowser::UpdateCurrentFilePath()
{
    // 選択アイテムから現在のファイルパスを作成します。
    for(s32 i = 0; i < m_Entries.GetItemCount(); ++i)
    {
        ListBox::ListBoxItemPlaceHolder& itemPlaceHolder = m_Entries.GetItemPlaceHolder(i);

        if(itemPlaceHolder.GetIsSelected())
        {
            FileBrowserEntry* pFileEntry = static_cast<FileBrowserEntry*>(itemPlaceHolder.GetContent());
            NW_NULL_ASSERT(pFileEntry);

            nw::ut::snprintf(
                m_CurrentFilePath,
                MAX_FILE_PATH + 1,
                MAX_FILE_PATH,
                "%s%c%s",
                m_CurrentDirectoryPath,
                PATH_SEPARATOR,
                pFileEntry->GetText());

            break;
        }
    }
}

void FileBrowser::OnFileEntrySelectionChanged()
{
    UpdateCurrentFilePath();
}

} // dw
} // internal
} // nw
