﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_PrimitiveRenderer.h>

namespace nw
{
namespace dev
{

PrimitiveRenderer* PrimitiveRenderer::s_pInstance = NULL;

//--------------------------------------------------------------------------
void
PrimitiveRenderer::InitializeFromBinary( nw::ut::IAllocator* allocator, const void* binaryData, u32 binarySize )
{
    Prepare( allocator );
    m_RendererImpl->InitializeFromBinaryImpl( allocator, binaryData, binarySize );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::Initialize( nw::ut::IAllocator* allocator, const char* shaderPath )
{
    Prepare( allocator );
    m_RendererImpl->InitializeImpl( allocator, shaderPath );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::Finalize(nw::ut::IAllocator* allocator)
{
    m_RendererImpl->~PrimitiveRendererBase();
    nw::ut::SafeFree( m_RendererImpl, allocator );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::CreateVertexBuffer( DrawBuffer& buf )
{
    m_RendererImpl->CreateVertexBufferImpl( buf );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::SetViewMtx( const nw::math::MTX34* viewMtx )
{
    m_RendererImpl->SetViewMtxImpl( viewMtx );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::SetProjectionMtx( const nw::math::MTX44* projectionMtx )
{
    m_RendererImpl->SetProjectionMtxImpl( projectionMtx );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::SetViewport( const s32 x, const s32 y, const s32 width, const s32 height )
{
    m_RendererImpl->SetViewportImpl( x, y, width, height );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::Begin()
{
    m_RendererImpl->BeginImpl();
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::End()
{
    m_RendererImpl->EndImpl();
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawTriangle( const nw::ut::Color4u8& color )
{
    m_RendererImpl->DrawTriangleImpl( m_ModelMtx, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawTriangle( const nw::ut::Color4u8& color, const DrawBuffer& buf )
{
    m_RendererImpl->DrawTriangleImpl( m_ModelMtx, color, buf );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawQuad( const nw::ut::Color4u8& colorLeft, const nw::ut::Color4u8& colorRight )
{
    m_RendererImpl->DrawQuadImpl( m_ModelMtx, colorLeft, colorRight );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawQuad(
    const nw::gfnd::Texture& texture,
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight,
    const nw::math::VEC2& uvSrc,
    const nw::math::VEC2& uvSize
)
{
    m_RendererImpl->DrawQuadImpl( m_ModelMtx, texture, colorLeft, colorRight, uvSrc, uvSize );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawQuad( const nw::dev::PrimitiveRenderer::QuadArg& arg )
{
    nw::math::MTX34 mtx;

    if ( arg.IsHorizonal() )
    {
        nw::math::MTX34MakeSRT(
            &mtx,
            nw::math::VEC3( arg.GetSize().y, arg.GetSize().x, 1.f ),
            nw::math::VEC3( 0.f, 0.f, NW_MATH_DEG_TO_RAD( 90.0f ) ),
            arg.GetCenter()
        );
    }
    else
    {
        nw::math::MTX34MakeST(
            &mtx,
            nw::math::VEC3( arg.GetSize().x, arg.GetSize().y, 1.f ),
            arg.GetCenter()
        );
    }

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawQuadImpl( resultMtx, arg.GetColor0(), arg.GetColor1() );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawQuad(
    const nw::gfnd::Texture& texture,
    const nw::dev::PrimitiveRenderer::QuadArg& arg,
    const nw::dev::PrimitiveRenderer::UVArg& uvArg
)
{
    nw::math::MTX34 mtx;

    if ( arg.IsHorizonal() )
    {
        nw::math::MTX34MakeSRT(
            &mtx,
            nw::math::VEC3( arg.GetSize().y, arg.GetSize().x, 1.f ),
            nw::math::VEC3( 0.f, 0.f, NW_MATH_DEG_TO_RAD( 90.0f ) ),
            arg.GetCenter()
        );
    }
    else
    {
        nw::math::MTX34MakeST(
            &mtx,
            nw::math::VEC3( arg.GetSize().x, arg.GetSize().y, 1.f ),
            arg.GetCenter()
        );
    }

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawQuadImpl( resultMtx, texture, arg.GetColor0(), arg.GetColor1(), uvArg.GetUVSrc(), uvArg.GetUVSize() );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawBox(
    const nw::ut::Color4u8& colorLeft,
    const nw::ut::Color4u8& colorRight
)
{
    m_RendererImpl->DrawBoxImpl( m_ModelMtx, colorLeft, colorRight );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawBox( const nw::dev::PrimitiveRenderer::QuadArg& arg )
{
    nw::math::MTX34 mtx;

    if ( arg.IsHorizonal() )
    {
        nw::math::MTX34MakeSRT(
            &mtx,
            nw::math::VEC3( arg.GetSize().y, arg.GetSize().x, 1.f ),
            nw::math::VEC3( 0.f, 0.f, NW_MATH_DEG_TO_RAD( 90.0f ) ),
            arg.GetCenter()
        );
    }
    else
    {
        nw::math::MTX34MakeST(
            &mtx,
            nw::math::VEC3( arg.GetSize().x, arg.GetSize().y, 1.f ),
            arg.GetCenter()
        );
    }

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawBoxImpl( resultMtx, arg.GetColor0(), arg.GetColor1() );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCube( const nw::ut::Color4u8& color0, const nw::ut::Color4u8& color1 )
{
    m_RendererImpl->DrawCubeImpl( m_ModelMtx, color0, color1 );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCube( const nw::dev::PrimitiveRenderer::CubeArg& arg )
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, arg.GetSize(), arg.GetCenter() );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawCubeImpl( resultMtx, arg.GetColor0(), arg.GetColor1() );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawWireCube( const nw::ut::Color4u8& color0, const nw::ut::Color4u8& color1 )
{
    m_RendererImpl->DrawWireCubeImpl( m_ModelMtx, color0, color1 );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawWireCube( const nw::dev::PrimitiveRenderer::CubeArg& arg )
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, arg.GetSize(), arg.GetCenter() );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawWireCubeImpl( resultMtx, arg.GetColor0(), arg.GetColor1() );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawLine(
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    const f32 lineWidth /* = 1.f */
)
{
    m_RendererImpl->DrawLineImpl( m_ModelMtx, color0, color1, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawLine(
    const nw::math::VEC3& from,
    const nw::math::VEC3& to,
    const nw::ut::Color4u8& color0,
    const nw::ut::Color4u8& color1,
    const f32 lineWidth /* = 1.f */
)
{
    nw::math::VEC3 dir = to - from;

    // 線の長さに拡大する行列作成。
    nw::math::MTX34 nextMtx;
    nw::math::MTX34MakeS( &nextMtx, nw::math::VEC3( dir.Length(), 1.f, 1.f ) );
    dir.Normalize();

    // 線を回転させる行列作成。
    nw::math::QUAT q;
    nw::math::QUATMakeVectorRotation( &q, nw::math::VEC3( 1.f, 0.f, 0.f ), dir );

    nw::math::MTX34 qmat;
    nw::math::QUATToMTX34( &qmat, q );

    nw::math::MTX34 mtx = nextMtx;
    nw::math::MTX34Mult( &mtx, qmat, nextMtx );

    // 線の始点に移動させる行列作成。
    dir = ( to - from ) * 0.5f + from;
    mtx.SetColumn( 3, dir );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawLineImpl( resultMtx, color0, color1, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawLine( const nw::ut::Color4u8& color, const DrawBuffer& buf, const f32 lineWidth /* = 1.f */ )
{
    m_RendererImpl->DrawLineImpl( m_ModelMtx, color, buf, lineWidth );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawSphere4x8(
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    m_RendererImpl->DrawSphere4x8Impl( m_ModelMtx, north, south );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawSphere4x8(
    const nw::math::VEC3& pos,
    f32 radius,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, nw::math::VEC3( radius * 2.f, radius * 2.f, radius * 2.f ), pos );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawSphere4x8Impl( resultMtx, north, south );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawSphere8x16(
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    m_RendererImpl->DrawSphere8x16Impl( m_ModelMtx, north, south );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawSphere8x16(
    const nw::math::VEC3& pos,
    f32 radius,
    const nw::ut::Color4u8& north,
    const nw::ut::Color4u8& south
)
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, nw::math::VEC3( radius * 2.f, radius * 2.f, radius * 2.f ), pos );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawSphere8x16Impl( resultMtx, north, south );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCircle16( const nw::ut::Color4u8& color )
{
    m_RendererImpl->DrawCircle16Impl( m_ModelMtx, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCircle16(
    const nw::math::VEC3& pos,
    f32 radius,
    const nw::ut::Color4u8& color
)
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, nw::math::VEC3( radius * 2.f, radius * 2.f, radius * 2.f ), pos );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawCircle16Impl( resultMtx, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCircle32( const nw::ut::Color4u8& color )
{
    m_RendererImpl->DrawCircle32Impl( m_ModelMtx, color );
}

//--------------------------------------------------------------------------
void
PrimitiveRenderer::DrawCircle32(
    const nw::math::VEC3& pos,
    f32 radius,
    const nw::ut::Color4u8& color
)
{
    nw::math::MTX34 mtx;
    nw::math::MTX34MakeST( &mtx, nw::math::VEC3( radius * 2.f, radius * 2.f, radius * 2.f ), pos );

    nw::math::MTX34 resultMtx;
    nw::math::MTX34Mult( &resultMtx, m_ModelMtx, mtx );

    m_RendererImpl->DrawCircle32Impl( resultMtx, color );
}



//--------------------------------------------------------------------------
void
PrimitiveRenderer::Prepare( nw::ut::IAllocator* allocator )
{
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
    m_RendererImpl = new( allocator->Alloc( sizeof( PrimitiveRendererGL ) ) ) PrimitiveRendererGL( allocator );
#elif defined(NW_PLATFORM_CAFE)
    m_RendererImpl = new( allocator->Alloc( sizeof( PrimitiveRendererCafe ) ) ) PrimitiveRendererCafe( allocator );
#else
#error "Unknown platform"
#endif

    NW_ASSERT_NOT_NULL( m_RendererImpl );
}

} // namespace dev
} // namespace nw
