﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_LoggerBase.h>
#include <nw/ut/os/ut_Print.h>
#include <nw/ut/ut_ScopedLock.h>
#include <cstdarg>
#include <cstdio>
#include <cstring>

namespace nw
{
namespace dev
{

template <typename CharType>
LoggerBase<CharType>* LoggerBase<CharType>::s_LoggerList = NULL;
template <typename CharType>
nw::ut::Mutex LoggerBase<CharType>::s_Mutex;
template <typename CharType>
bool LoggerBase<CharType>::s_Initialized = false;

const u32 DIRECT_FONT_WIDTH  = 6;
const u32 DIRECT_FONT_HEIGHT = 7 + 3;
const u32 STRING_BUF_SIZE = 1024;

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::Create( void* buffer, u16 width, u16 height, u16 viewHeight, u16 priority, u16 attr /* = LOGGER_ATTR_DEFAULT */ )
{
    enum
    {
        LOGGER_VIEW_POS_X_DEFAULT  = 30,
        LOGGER_VIEW_POS_Y_DEFAULT  = 50,
        LOGGER_VIEW_LINES_DEFAULT  = 20
    };

    NW_ASSERT_NOT_NULL  ( buffer );
    NW_ALIGN4_ASSERT( buffer );

    if ( ! s_Initialized )
    {
        s_Mutex.Initialize();
        s_Initialized = true;
    }

    m_LoggerBuffer = static_cast<LoggerBuffer*>(buffer);

    m_LoggerBuffer->textBuf          = static_cast<CharType*>(buffer) + sizeof(LoggerBuffer);
    m_LoggerBuffer->width            = width;
    m_LoggerBuffer->height           = height;
    m_LoggerBuffer->priority         = priority;
    m_LoggerBuffer->attr             = attr;

    m_LoggerBuffer->isVisible        = false;

    m_LoggerBuffer->printTop         = 0;
    m_LoggerBuffer->printXPos        = 0;
    m_LoggerBuffer->ringTop          = 0;
    m_LoggerBuffer->ringTopLineCnt   = 0;
    m_LoggerBuffer->viewTopLine      = 0;

    m_LoggerBuffer->viewPosX         = LOGGER_VIEW_POS_X_DEFAULT;
    m_LoggerBuffer->viewPosY         = LOGGER_VIEW_POS_Y_DEFAULT;
    m_LoggerBuffer->viewLines        = viewHeight;

    Clear();

    AppendLoggerToList();
}

//--------------------------------------------------------------------------
template <typename CharType>
void*
LoggerBase<CharType>::Destroy()
{
    RemoveLoggerFromList();

    return m_LoggerBuffer;
}

//--------------------------------------------------------------------------
template <typename CharType>
u16
LoggerBase<CharType>::ChangePriority( u16 priority )
{
    u16 lastPriority;

    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    RemoveLoggerFromList();
    lastPriority = m_LoggerBuffer->priority;
    m_LoggerBuffer->priority = priority;
    AppendLoggerToList();

    return lastPriority;
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::Printf( const CharType* format, ... )
{
    va_list vlist;
    va_start( vlist, format );
    VFPrintf( OUTPUT_ALL, format, vlist );
    va_end( vlist );
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::PrintfD( const CharType* format, ... )
{
    va_list vlist;
    va_start( vlist, format );
    VFPrintf( OUTPUT_DISPLAY, format, vlist );
    va_end( vlist );
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::PrintfT( const CharType* format, ... )
{
    va_list vlist;
    va_start( vlist, format );
    VFPrintf( OUTPUT_TERMINAL, format, vlist );
    va_end( vlist );
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::FPrintf( OutputType type, const CharType* format, ... )
{
    va_list vlist;
    va_start( vlist, format );
    VFPrintf( type, format, vlist );
    va_end( vlist );
}

#if defined(NW_LOGGER_ENABLE)
//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::VFPrintf( OutputType type, const CharType* format, std::va_list vlist )
{
    static CharType sStrBuf[ STRING_BUF_SIZE ];

    // sStrBuf と OSReport() を共有するためロックを取得。
    if ( ! s_Mutex.TryLock() )
    {
        return;
    }

    {
        (void)this->VSNPrintf( sStrBuf, STRING_BUF_SIZE, format, vlist );
        PrintString( type, sStrBuf );
    }

    s_Mutex.Unlock();
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::Draw( nw::font::TextWriterBase<CharType>& writer, nw::font::RectDrawer& drawer )
{
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    // 表示フラグが立っていない場合は終了。
    if ( ! m_LoggerBuffer->isVisible )
    {
        return;
    }

    writer.StartPrint();
    PrintToWriter( writer );
    writer.EndPrint();

    nw::font::DispStringBuffer* dispStringBuffer = writer.GetDispStringBuffer();
    NW_ASSERT_NOT_NULL( dispStringBuffer );

    drawer.BuildVertexElements(dispStringBuffer);

    font::DrawContent content;
    content.dispStringBuffer = dispStringBuffer;
    content.localMatrix = &nw::math::MTX34::Identity();
    drawer.Draw(content);
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::DrawDirect()
{
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    NW_UNUSED_VARIABLE( DIRECT_FONT_WIDTH );
    NW_UNUSED_VARIABLE( DIRECT_FONT_HEIGHT );

#if 0 // TODO: DirectPrint による描画に対応する。
    // DirectPrint が初期化されていないならば終了。
    if ( ! DirectPrint_IsActive() )
    {
        return;
    }

    // 表示フラグが立っていない場合は終了。
    if ( ! m_LoggerBuffer->isVisible )
    {
        return;
    }

    {
        // 背景をクリア。
        {
            int width, height;

            width  = (int)( m_LoggerBuffer->width * DIRECT_FONT_WIDTH + DIRECT_FONT_WIDTH * 2 );
            height = (int)( m_LoggerBuffer->viewLines * DIRECT_FONT_HEIGHT + 4 );

            nw4r::db::DirectPrint_EraseXfb(
                        (int)( m_LoggerBuffer->viewPosX - DIRECT_FONT_WIDTH ), // 左座標
                        (int)( m_LoggerBuffer->viewPosY - 3 ),                 // 上座標
                        width,                                                  // 幅
                        height                                                  // 高さ
                    );
        }

        DoDraw( NULL );
        DirectPrint_StoreCache();
    }
#endif
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::DrawAll( nw::font::TextWriterBase<CharType>& writer, nw::font::RectDrawer& drawer )
{
    if ( ! s_Initialized )
    {
        return;
    }

    LoggerBase* pLogger = s_LoggerList;

    writer.StartPrint();
    {
        while ( pLogger != NULL )
        {
            if ( pLogger->IsVisible() )
            {
                pLogger->PrintToWriter( writer );
            }

            pLogger = pLogger->m_Next;
        }
    }
    writer.EndPrint();

    nw::font::DispStringBuffer* dispStringBuffer = writer.GetDispStringBuffer();
    NW_ASSERT_NOT_NULL( dispStringBuffer );

    drawer.BuildVertexElements( dispStringBuffer );

    font::DrawContent content;
    content.dispStringBuffer = dispStringBuffer;
    content.localMatrix = &nw::math::MTX34::Identity();
    drawer.Draw( content );
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::DrawDirectAll()
{
    if ( ! s_Initialized )
    {
        return;
    }

    LoggerBase* pLogger = s_LoggerList;

    while ( pLogger != NULL )
    {
        pLogger->DrawDirect();
        pLogger = pLogger->m_Next;
    }
}

#endif // #if defined(NW_LOGGER_ENABLE)

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::PrintToWriter( nw::font::TextWriterBase<CharType>& writer )
{
    writer.SetCursor( m_LoggerBuffer->viewPosX, m_LoggerBuffer->viewPosY );

    DoDraw( &writer );
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::Clear()
{
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    if ( ! s_Mutex.TryLock() )
    {
        return;
    }

    m_LoggerBuffer->printTop       = 0;
    m_LoggerBuffer->printXPos      = 0;

    m_LoggerBuffer->ringTop        = 0;
    m_LoggerBuffer->ringTopLineCnt = 0;
    m_LoggerBuffer->viewTopLine    = 0;

    s_Mutex.Unlock();
}

//--------------------------------------------------------------------------
template <typename CharType>
s32
LoggerBase<CharType>::GetTotalLines()
{
    s32 count;
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

    count = m_LoggerBuffer->ringTopLineCnt + GetActiveLines();

    return count;
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::VisitString( LoggerVisitor visitor, u32 userParam )
{
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

    u16 line = m_LoggerBuffer->ringTop;
    s32 cnt = m_LoggerBuffer->ringTopLineCnt;

    for (;;)
    {
        CharType* str;
        if ( (line == m_LoggerBuffer->printTop) && ( m_LoggerBuffer->printXPos == 0 ) )
        {
            break;
        }
        str = GetTextPtr( line, 0 );

        visitor( str, cnt, userParam );

        if ( line == m_LoggerBuffer->printTop )
        {
            break;
        }
        if ( ++line == m_LoggerBuffer->height )
        {
            line = 0;
        }
        ++cnt;
    }
}



//--------------------------------------------------------------------------
template <typename CharType>
LoggerBase<CharType>*
LoggerBase<CharType>::SearchLoggerFromListByPriority( u16 priority )
{
    LoggerBase* pLogger = s_LoggerList;

    if ( pLogger == NULL || pLogger->m_LoggerBuffer->priority < priority )
    {
        return NULL;
    }

    // 引数のものよりも優先度の値が大きなロガーがあった場合は、
    // その一つ前が挿入位置になる
    while ( pLogger->m_Next != NULL )
    {
        if ( pLogger->m_Next->m_LoggerBuffer->priority < priority )
        {
            return pLogger;
        }
        pLogger = pLogger->m_Next;
    }
    return pLogger;
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::AppendLoggerToList()
{
    NW_ASSERT_NOT_NULL( m_LoggerBuffer );

    // リストを操作する為必ずロックします
    nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

    LoggerBase* pLogger = SearchLoggerFromListByPriority( m_LoggerBuffer->priority );

    // 検索結果が NULL の場合にはリストの先頭へ挿入
    if ( pLogger == NULL )
    {
        m_Next = s_LoggerList;
        s_LoggerList = this;
        return;
    }

    // 検索位置に挿入
    m_Next = pLogger->m_Next;
    pLogger->m_Next = this;
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::RemoveLoggerFromList()
{
    // リストを操作する為必ずロックします
    nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

    LoggerBase* pLogger = s_LoggerList;

    NW_ASSERT_NOT_NULL( pLogger );

    // リスト先頭要素だった場合の削除処理
    if ( pLogger == this )
    {
        s_LoggerList = m_Next;
        m_Next = NULL;
        return;
    }

    // リストを辿りながら削除するロガーを検索
    while ( pLogger->m_Next != NULL )
    {
        if ( pLogger->m_Next == this )
        {
            pLogger->m_Next = m_Next;
            m_Next = NULL;
            return;
        }
        pLogger = pLogger->m_Next;
    }

    // don't reach.
    // 指定のロガーがリストに含まれていなかったために削除失敗
    NW_FATAL_ERROR("illegal logger handle");
}

#if defined(NW_LOGGER_ENABLE)
//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::PrintString( OutputType type, const CharType* str )
{
    // --- ターミナルに出力
    if ( type & OUTPUT_TERMINAL )
    {
        // TODO: 現在 char16 版 printf がありません。
        nw::ut::TPrintf( "%s", str );
    }

    // --- ディスプレイ用のバッファに出力
    if ( type & OUTPUT_DISPLAY )
    {
        PrintToBuffer( str );
    }
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::PrintToBuffer( const CharType* str )
{
    // ---- 文字を格納して行く領域
    CharType* storePtr;

    NW_ASSERT_NOT_NULL( m_LoggerBuffer );
    NW_ASSERT_NOT_NULL( str );

    storePtr = GetTextPtr( m_LoggerBuffer->printTop, m_LoggerBuffer->printXPos );

    // -------- 一文字ずつループ
    while ( *str )
    {
        if ( m_LoggerBuffer->attr & ATTR_STOP_ON_BUFFER_FULL )
        // バッファ溢れストップが設定されているなら、行が溢れている場合には書かない
        {
            if ( m_LoggerBuffer->printTop == m_LoggerBuffer->height )
            {
                break;
            }
        }

        // ソース文字列が空になるか、出力先が行末になるまで文字をコピー
        while ( *str )
        {
            bool newLineFlag = false;

            if ( *str == '\n' )
            // 改行文字の処理
            {
                str++;
                storePtr = NextLine();
                break;
            }
            else if ( *str == '\t' )
            // タブ文字の処理
            {
                str++;
                storePtr = PutTab( storePtr );
            }
            else
            // 通常文字コードの処理
            {
                u32 bytes = PutChar( str, storePtr );

                if ( bytes > 0 )
                {
                    str      += bytes;
                    storePtr += bytes;
                }
                else
                {
                    newLineFlag = true;
                }
            }
            if ( m_LoggerBuffer->printXPos >= m_LoggerBuffer->width )
            // 文字位置が端まで来ていたら改行フラグを立てる
            {
                newLineFlag = true;
            }

            if ( newLineFlag )
            // 改行
            {
                if ( m_LoggerBuffer->attr & ATTR_NOT_TURNOVER_LINE )
                {
                    // 行末まで読み飛ばす
                    str = SearchEndOfLine( str );
                }
                else
                {
                    // 次が改行文字であれば１つ読み飛ばす
                    if ( *str == '\n' )
                    {
                        ++str;
                    }
                    storePtr = NextLine();
                }

                break;
            }

            if ( *str == '\0' )
            {
                TerminateLine();
            }
        }
    }

}
#endif // #if defined(NW_LOGGER_ENABLE)

//--------------------------------------------------------------------------
template <typename CharType>
CharType*
LoggerBase<CharType>::NextLine()
{
    // 現在の行末にNUL文字を挿入
    *( GetTextPtr( m_LoggerBuffer->printTop, m_LoggerBuffer->printXPos ) ) = '\0';

    // 出力位置を変更
    m_LoggerBuffer->printXPos = 0;
    m_LoggerBuffer->printTop++;

    if ( m_LoggerBuffer->printTop == m_LoggerBuffer->height )
    {
        if ( ! (m_LoggerBuffer->attr & ATTR_STOP_ON_BUFFER_FULL) )
        {
            m_LoggerBuffer->printTop = 0;
        }
    }

    if ( m_LoggerBuffer->printTop == m_LoggerBuffer->ringTop )
    {
        ++m_LoggerBuffer->ringTopLineCnt;

        if ( ++m_LoggerBuffer->ringTop == m_LoggerBuffer->height )
        {
            m_LoggerBuffer->ringTop = 0;
        }
    }

    return GetTextPtr( m_LoggerBuffer->printTop, 0 );
}

//--------------------------------------------------------------------------
template <typename CharType>
CharType*
LoggerBase<CharType>::PutTab( CharType* dstPtr )
{
    u32 tabWidth = GetTabSize();

    do
    {
        *dstPtr++ = ' ';
        m_LoggerBuffer->printXPos += 1;

        if ( m_LoggerBuffer->printXPos >= m_LoggerBuffer->width )
        {
            break;
        }
    } while ( m_LoggerBuffer->printXPos & (tabWidth - 1) );

    return dstPtr;
}

//--------------------------------------------------------------------------
template <typename CharType>
u32
LoggerBase<CharType>::PutChar( const CharType* str, CharType* dstPtr )
{
    u32 codeWidth = CodeWidth( str );
    u32 cnt;

    if ( m_LoggerBuffer->printXPos + codeWidth > m_LoggerBuffer->width )
    {
        return 0;
    }

    m_LoggerBuffer->printXPos += static_cast<u16>(codeWidth);
    cnt = codeWidth;

    while ( cnt > 0 )
    {
        *dstPtr++ = *str++;
        cnt--;
    }

    return codeWidth;
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::DoDrawString( u32 printLine, const CharType* str, nw::font::TextWriterBase<CharType>* writer )
{
    (void)printLine;
    if ( writer )
    {
        static CharType sStrBuf[ STRING_BUF_SIZE ];
        size_t size = StrCpy( sStrBuf, STRING_BUF_SIZE, str );

        if ( size < STRING_BUF_SIZE - 1 )
        {
            sStrBuf[size++] = '\n';
            sStrBuf[size++] = '\0';
        }

        (void)writer->Print( sStrBuf );
    }
#if 0 // TODO: DirectPrint による描画に対応する。
    else
    {
        s32 height = m_LoggerBuffer->viewPosY + (s32)printLine * (s32) DIRECT_FONT_HEIGHT;
        nw4r::db::DirectPrint_DrawString( m_LoggerBuffer->viewPosX, height, false, "%s\n", str );
    }
#endif
}

//--------------------------------------------------------------------------
template <typename CharType>
void
LoggerBase<CharType>::DoDraw( nw::font::TextWriterBase<CharType>* writer )
{
    nw::ut::ScopedLock<nw::ut::Mutex> lockObj(s_Mutex);

    // 文字を出力
    {
        s32 viewOffset = m_LoggerBuffer->viewTopLine - m_LoggerBuffer->ringTopLineCnt;
        u16 line;                           // カレント表示行番号
        u16 printLines = 0;                 // 表示行カウンタ
        u16 topLine;                        // 表示を終了するライン

        if ( viewOffset < 0 )
        {
            viewOffset = 0;
        }
        else if ( viewOffset > GetActiveLines() )
        {
            return;
        }

        line = (u16)(m_LoggerBuffer->ringTop + viewOffset);
        if ( line >= m_LoggerBuffer->height )
        {
            line -= m_LoggerBuffer->height;
        }

        topLine = u16( m_LoggerBuffer->printTop + ((m_LoggerBuffer->printXPos > 0)? 1 : 0) );
        if ( topLine == m_LoggerBuffer->height )
        {
            topLine = 0;
        }

        while ( line != topLine )
        {
            DoDrawString( printLines, GetTextPtr( line, 0 ), writer );
            ++printLines;

            if ( ++line == m_LoggerBuffer->height )
            {
                if ( m_LoggerBuffer->attr & ATTR_STOP_ON_BUFFER_FULL )
                {
                    break;
                }
                line = 0;
            }

            if ( printLines >= m_LoggerBuffer->viewLines )
            {
                break;
            }
        }
    }
}

template class LoggerBase<char>;
template class LoggerBase<char16>;

} // namespace dev
} // namespace nw

