﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_FileLogger.h>
#include <nw/assert.h>
#include <nw/ut/ut_String.h>
#include <nw/ut/ut_FileStream.h>

namespace nw
{
namespace dev
{

FileLogger::FileLogger() :
m_BufferSize(0),
m_pCurrentBufferInfo(NULL)
{
}

void FileLogger::SetBuffer(char* buffer, u32 bufferSize)
{
    NW_NULL_ASSERT(buffer);

    u32 alignment = GetBufferAlignment();

    if(alignment > 0)
    {
        m_BufferInfo0.buffer = static_cast<char*>(ut::RoundUp(buffer, alignment));
    }
    else
    {
        m_BufferInfo0.buffer = buffer;
    }

    m_BufferInfo1.buffer = NULL;
    m_BufferInfo0.bufferedSize = 0;
    m_BufferInfo1.bufferedSize = 0;

    m_BufferSize = bufferSize;
    m_pCurrentBufferInfo = &m_BufferInfo0;
}

void FileLogger::SetBuffer(char* buffer0, char* buffer1, u32 bufferSize)
{
    NW_NULL_ASSERT(buffer0);
    NW_NULL_ASSERT(buffer1);

    u32 alignment = GetBufferAlignment();

    if(alignment > 0)
    {
        m_BufferInfo0.buffer = static_cast<char*>(ut::RoundUp(buffer0, alignment));
        m_BufferInfo1.buffer = static_cast<char*>(ut::RoundUp(buffer1, alignment));
    }
    else
    {
        m_BufferInfo0.buffer = buffer0;
        m_BufferInfo1.buffer = buffer1;
    }

    m_BufferInfo0.bufferedSize = 0;
    m_BufferInfo1.bufferedSize = 0;

    m_BufferSize = bufferSize - GetBufferAlignment();
    m_pCurrentBufferInfo = &m_BufferInfo0;
}

void FileLogger::Attach(ut::FileStream& fileStream)
{
    NW_ASSERT(IsAttached());
    m_pFileStream = &fileStream;
}

ut::FileStream* FileLogger::Detach()
{
    if(!IsAttached())
    {
        return NULL;
    }

    FlushBuffer();

    ut::FileStream* fileStream = m_pFileStream;

    m_pFileStream->Close();
    m_pFileStream = NULL;

    return fileStream;
}

void FileLogger::Printf(const char* format, ...)
{
    NW_NULL_ASSERT(format);

    va_list vlist;
    va_start(vlist, format);

    VPrintf(format, vlist);

    va_end(vlist);
}

void FileLogger::VPrintf(const char* format, std::va_list vlist)
{
    NW_NULL_ASSERT(format);

    FlushBuffer();

    // ダブルバッファの場合は空きバッファを使って文字列を書式化します。
    BufferInfo* bufferInfo = GetIdleBufferInfo();

    NW_NULL_ASSERT(bufferInfo);
    NW_ASSERT(m_BufferSize > 0);

    int size = ut::vsnprintf(bufferInfo->buffer, m_BufferSize, m_BufferSize - 1, format, vlist);

    if(size == -1)
    {
        size = m_BufferSize;
    }

    m_pFileStream->Write(bufferInfo->buffer, size);
}

void FileLogger::PrintfToBuffer(const char* format, ...)
{
    NW_NULL_ASSERT(format);

    va_list vlist;
    va_start(vlist, format);

    VPrintfToBuffer(format, vlist);

    va_end(vlist);
}

void FileLogger::VPrintfToBuffer(const char* format, std::va_list vlist)
{
    NW_NULL_ASSERT(format);

    // 現在のバッファに追記します。
    BufferInfo* bufferInfo = m_pCurrentBufferInfo;

    NW_NULL_ASSERT(bufferInfo);
    NW_ASSERT(m_BufferSize > 0);

    int size = ut::vsnprintf(
        bufferInfo->buffer + bufferInfo->bufferedSize,
        m_BufferSize - bufferInfo->bufferedSize,
        m_BufferSize - bufferInfo->bufferedSize - 1,
        format,
        vlist);

    NW_ASSERT(size != -1);

    bufferInfo->bufferedSize += size;
}

void FileLogger::FlushBuffer()
{
    if(!IsAttached() || !IsBufferSetuped())
    {
        return;
    }

    // フラッシュ途中で割り込みが発生する可能性を考慮して、バッファを切り替えておきます。
    // ※ダブルバッファの場合のみ
    BufferInfo* bufferInfo = m_pCurrentBufferInfo;
    m_pCurrentBufferInfo = GetIdleBufferInfo();

    NW_NULL_ASSERT(bufferInfo);

    if(bufferInfo->bufferedSize > 0)
    {
        m_pFileStream->Write(bufferInfo->buffer, bufferInfo->bufferedSize);
        bufferInfo->bufferedSize = 0;
    }
}

bool FileLogger::IsBufferSetuped() const
{
    return m_BufferInfo0.buffer != NULL;
}

FileLogger::BufferInfo* FileLogger::GetIdleBufferInfo()
{
    if(m_pCurrentBufferInfo == &m_BufferInfo0)
    {
        if(m_BufferInfo1.buffer != NULL)
        {
            return &m_BufferInfo1;
        }

        return m_BufferInfo0.buffer != NULL ? &m_BufferInfo0 : NULL;
    }
    else
    {
        if(m_BufferInfo0.buffer != NULL)
        {
            return &m_BufferInfo0;
        }

        return m_BufferInfo1.buffer != NULL ? &m_BufferInfo1 : NULL;
    }
}

} // namespace dev
} // namespace nw
