﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/dev/dev_DevTextWriter.h>
#include <nw/gfnd/gfnd_Graphics.h>

namespace nw  {
namespace dev {

char DevTextWriter::s_DefaultFormatBuffer[DevTextWriter::DEFAULT_FORMAT_BUFFER_SIZE];
std::size_t DevTextWriter::s_FormatBufferSize = DevTextWriter::DEFAULT_FORMAT_BUFFER_SIZE;
char* DevTextWriter::s_FormatBuffer = DevTextWriter::s_DefaultFormatBuffer;
const u32 DevTextWriter::DEFAULT_SHADOW_COLOR = 0x101010FF;
const f32 DevTextWriter::DEFAULT_SHADOW_POS_X = 1.f;
const f32 DevTextWriter::DEFAULT_SHADOW_POS_Y = 1.f;

//---------------------------------------------------------------------------
void
DevTextWriter::Initialize( void* fontData, u32 fontDataSize, void* shaderData, u32 shaderDataSize, ut::IAllocator* allocator )
{
    NW_UNUSED_VARIABLE( fontDataSize );

    NW_ASSERT_NOT_NULL( allocator );
    m_Allocator = allocator;

    m_Writer.stringBufferArray.Reset( NULL, 0, allocator, ut::ARRAY_VARIABILITY );
    m_ShadowWriter.stringBufferArray.Reset( NULL, 0, allocator, ut::ARRAY_VARIABILITY );

    // フォントの設定
    {
        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            // フォントリソースのセット
            bool result = m_ResFont.SetResource( fontData );
            NW_ASSERT( result );
        }
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

        m_Writer.textWriter.SetFont( &m_ResFont );
        m_Writer.textWriter.SetTagProcessor( &m_TagProcessor );
        m_ShadowWriter.textWriter.SetFont( &m_ResFont );
        m_ShadowWriter.textWriter.SetTagProcessor( &m_TagProcessor );
        SetShadowColor( nw::ut::Color4u8( DEFAULT_SHADOW_COLOR ) );
        SetShadowPos( DEFAULT_SHADOW_POS_X, DEFAULT_SHADOW_POS_Y );
    }

    // 描画用バッファの確保と設定
    {
    #if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
// TODO: NintendoSdk 対応後、このコメントを削除してください。
        (void)shaderData;
        (void)shaderDataSize;
        const u32 vtxBufferSize = font::RectDrawer::GetWorkBufferSize( m_CharMax );
    #else
        const u32 vtxBufferSize = font::RectDrawer::GetWorkBufferSize( m_CharMax, ut::MakeMemoryRange( shaderData, shaderDataSize ) );
    #endif

        gfnd::Graphics::GetInstance()->LockDrawContext();
        {
            m_DrawerBuffer = allocator->Alloc( vtxBufferSize );

        #if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
// TODO: NintendoSdk 対応後、このコメントを削除してください。
            m_Drawer.Initialize( m_DrawerBuffer, m_CharMax );
        #else
            m_Drawer.Initialize( m_DrawerBuffer, m_CharMax, ut::MakeMemoryRange( shaderData, shaderDataSize ) );
        #endif
        }
        gfnd::Graphics::GetInstance()->UnlockDrawContext();
    }
}

//---------------------------------------------------------------------------
void
DevTextWriter::Finalize()
{
    m_Drawer.Finalize();

    DestroyStringBuffer();
    m_Writer.stringBufferArray.Reset( NULL, 0 );
    m_ShadowWriter.stringBufferArray.Reset( NULL, 0 );

    ut::SafeFree( m_DrawerBuffer, m_Allocator );

    m_ResFont.RemoveResource();
}

//---------------------------------------------------------------------------
void
DevTextWriter::Print( const char* str )
{
    PrintImpl( str, m_Writer );

    m_ShadowWriter.textWriter.SetCursor(
        m_Writer.textWriter.GetCursorX(),
        m_Writer.textWriter.GetCursorY(),
        m_Writer.textWriter.GetCursorZ()
    );
}

//---------------------------------------------------------------------------
void
DevTextWriter::Printf( const char* format, ... )
{
    NW_ASSERT_VALID_POINTER( format );

    std::va_list vargs;
    va_start( vargs, format );

    VPrintf( format, vargs );

    va_end(vargs);
}

//---------------------------------------------------------------------------
void
DevTextWriter::VPrintf( const char* format, std::va_list args )
{
    NW_ASSERT_VALID_POINTER( format );
    NW_ASSERT_NOT_NULL( s_FormatBuffer );

    ut::vsnprintf( s_FormatBuffer, s_FormatBufferSize, s_FormatBufferSize - 1, format, args );

    PrintImpl( s_FormatBuffer, m_Writer );

    m_ShadowWriter.textWriter.SetCursor(
        m_Writer.textWriter.GetCursorX(),
        m_Writer.textWriter.GetCursorY(),
        m_Writer.textWriter.GetCursorZ()
    );
}

//---------------------------------------------------------------------------
void
DevTextWriter::PrintWithShadow( const char* str )
{
    PrintImpl( str, m_ShadowWriter );
    PrintImpl( str, m_Writer );
}

//---------------------------------------------------------------------------
void
DevTextWriter::PrintfWithShadow( const char* format, ... )
{
    NW_ASSERT_VALID_POINTER( format );

    std::va_list vargs;
    va_start( vargs, format );

    VPrintfWithShadow( format, vargs );

    va_end(vargs);
}

//---------------------------------------------------------------------------
void
DevTextWriter::VPrintfWithShadow( const char* format, std::va_list args )
{
    NW_ASSERT_VALID_POINTER( format );
    NW_ASSERT_NOT_NULL( s_FormatBuffer );

    ut::vsnprintf( s_FormatBuffer, s_FormatBufferSize, s_FormatBufferSize - 1, format, args );

    PrintImpl( s_FormatBuffer, m_ShadowWriter );
    PrintImpl( s_FormatBuffer, m_Writer );
}

//---------------------------------------------------------------------------
void
DevTextWriter::CreateStringBuffer( Writer& writer )
{
    NW_ASSERT_NOT_NULL( m_Allocator );

    StringBuffer buffer = { NULL, NULL };

    // 文字列表示用バッファの確保
    const u32 stringBufferSize = font::CharWriter::GetDispStringBufferSize( m_CharMax );
    const u32 graphicsBufferSize = font::CharWriter::GetGraphicsBufferSize( m_CharMax );
    buffer.graphicsBuffer = m_Allocator->Alloc( graphicsBufferSize, font::CharWriter::GRAPHICS_BUFFER_ALIGNMENT );
    buffer.dispStringBuffer = font::CharWriter::InitDispStringBuffer( m_Allocator->Alloc( stringBufferSize ), buffer.graphicsBuffer, m_CharMax );

    writer.stringBufferArray.PushBack( buffer );
}

//---------------------------------------------------------------------------
void
DevTextWriter::DestroyStringBuffer()
{
    NW_ASSERT_NOT_NULL( m_Allocator );

    for ( nw::ut::MoveArray<StringBuffer>::iterator it = m_Writer.stringBufferArray.begin(); it != m_Writer.stringBufferArray.end(); ++it )
    {
        m_Allocator->Free( it->graphicsBuffer );
        m_Allocator->Free( it->dispStringBuffer );
    }

    for ( nw::ut::MoveArray<StringBuffer>::iterator it = m_ShadowWriter.stringBufferArray.begin(); it != m_ShadowWriter.stringBufferArray.end(); ++it )
    {
        m_Allocator->Free( it->graphicsBuffer );
        m_Allocator->Free( it->dispStringBuffer );
    }

    m_Writer.stringBufferArray.Reset( NULL, 0, m_Allocator, ut::ARRAY_VARIABILITY );
    m_ShadowWriter.stringBufferArray.Reset( NULL, 0, m_Allocator, ut::ARRAY_VARIABILITY );
    m_Writer.currentStringBuffer = NULL;
    m_ShadowWriter.currentStringBuffer = NULL;
}

//---------------------------------------------------------------------------
void
DevTextWriter::PreparePrint( Writer& writer )
{
    CreateStringBuffer( writer );

    writer.currentStringBuffer = &writer.stringBufferArray[ writer.stringBufferArray.Size() - 1 ];

    writer.textWriter.SetDispStringBuffer( writer.currentStringBuffer->dispStringBuffer );
    writer.textWriter.StartPrint();
}

//---------------------------------------------------------------------------
void
DevTextWriter::PrintImpl( const char* str, Writer& writer )
{
    if ( writer.currentStringBuffer == NULL )
    {
        PreparePrint( writer );
    }

    // バッファの残りに収まる文字数
    u32 charRestNum = writer.currentStringBuffer->dispStringBuffer->GetCharCountMax() -
        ( writer.currentStringBuffer->dispStringBuffer->GetCharCount() % writer.currentStringBuffer->dispStringBuffer->GetCharCountMax() );

    u32 strLength = std::strlen( str );

    (void)writer.textWriter.Print( str, ut::Min( charRestNum, strLength ) );

    // 現在のバッファが容量を超えたら、これまでを描画して新しいバッファを作成します。
    if ( charRestNum <= strLength )
    {
        Flush( writer );

        PreparePrint( writer );

        u32 charSize = nw::ut::mbslen( str, charRestNum );
        PrintImpl( str + charSize, writer );
    }
}

//---------------------------------------------------------------------------
void
DevTextWriter::Flush( Writer& writer )
{
    if ( writer.currentStringBuffer )
    {
        writer.textWriter.EndPrint();

        m_Drawer.BuildVertexElements(writer.currentStringBuffer->dispStringBuffer);
        font::DrawContent content;
        content.dispStringBuffer = writer.currentStringBuffer->dispStringBuffer;
        content.projectionMatrix = &m_ProjectionMtx;
        content.viewMatrix = &m_ViewMtx;
        content.localMatrix = &writer.modelMtx;
        m_Drawer.Draw(content);

        writer.currentStringBuffer = NULL;
    }
}


//---------------------------------------------------------------------------
DevTextWriter::EdTagProcessor::Operation
DevTextWriter::EdTagProcessor::Process( u16 code, font::PrintContext<char>* pContext )
{
    NW_ASSERT(code < ' ');
    NW_ASSERT_VALID_POINTER(pContext);

    if ( code == '\n' )
    {
        ProcessLinefeed( pContext );
        return OPERATION_DEFAULT;
    }

    return font::TagProcessorBase<char>::Process( code, pContext );
}

//---------------------------------------------------------------------------
DevTextWriter::EdTagProcessor::Operation
DevTextWriter::EdTagProcessor::CalcRect( ut::Rect* pRect, u16 code, font::PrintContext<char>* pContext )
{
    NW_ASSERT_VALID_POINTER(pRect);
    NW_ASSERT(code < ' ');
    NW_ASSERT_VALID_POINTER(pContext);

    if ( code == '\n' )
    {
        {
            font::TextWriterBase<char>& writer = *pContext->writer;

            pRect->right    = writer.GetCursorX();
            pRect->top      = writer.GetCursorY();

            ProcessLinefeed( pContext );

            pRect->left     = writer.GetCursorX();
            pRect->bottom   = writer.GetCursorY() + pContext->writer->GetFontHeight();

            pRect->Normalize();
        }
        return OPERATION_NEXT_LINE;
    }

    return font::TagProcessorBase<char>::CalcRect( pRect, code, pContext );
}

//---------------------------------------------------------------------------
void
DevTextWriter::EdTagProcessor::ProcessLinefeed( font::PrintContext<char>* pContext )
{
    NW_ASSERT_VALID_POINTER(pContext);

    font::TextWriterBase<char>& writer = *pContext->writer;

    writer.SetCursor(
        m_OriginX,
        writer.GetCursorY() + writer.GetLineHeight()
    );
}

} // namespace dev
} // namespace nw
