﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_SpyPacketReader.h>

#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>

#if defined(NW_DEBUG)
//#define STATE_DEBUG_ENABLED
//#define COM_DEBUG_ENABLED
#endif

namespace nw {
namespace snd {
namespace spy {
namespace internal {

//----------------------------------------------------------
SpyPacketReader::SpyPacketReader() :
m_Channel(NULL),
m_State(STATE_NOT_INITIALIZED),
m_PacketHeader(NULL),
m_ReadPacketLength(0)
{
}

//----------------------------------------------------------
void
SpyPacketReader::Initialize(nw::snd::spy::internal::fnd::HioChannel* channel, void* buffer)
{
    NW_ASSERT(m_Channel == NULL);
    NW_ASSERT_NOT_NULL(channel);
    NW_ASSERT_NOT_NULL(buffer);

    m_Channel = channel;
    m_PacketHeader = reinterpret_cast<PacketHeader*>(buffer);
    m_ReadPacketLength = 0;
    m_State = STATE_IDLE;
}

//----------------------------------------------------------
void
SpyPacketReader::Finalize()
{
    m_Channel = NULL;
    m_PacketHeader = NULL;
    m_ReadPacketLength = 0;
    m_State = STATE_NOT_INITIALIZED;
}

//----------------------------------------------------------
bool
SpyPacketReader::ReadPacketHeader()
{
    if(!IsInitialized())
    {
        return false;
    }

    if(STATE_IDLE < m_State && m_State < STATE_READ_BODY)
    {
        NW_ASSERTMSG(false, "packet not completed!\n");
        return false;
    }

    NW_ASSERT(m_State == STATE_IDLE || m_State == STATE_READ_BODY);

    if(!m_Channel->IsOpened())
    {
        m_State = STATE_IDLE;
        return false;
    }

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpyPacketReader] ReadPacketHeader().\n");
#endif

    m_State = STATE_READING_HEADER;

    // HioChannel::Read()は指定の長さ未満で戻ることがあるので
    // 指定の長さに達するまで繰り返し呼び出す。
    const u32 partBegin = 0;
    const u32 partEnd = partBegin + sizeof(PacketHeader);
    while (m_ReadPacketLength < partEnd)
    {
        u32 readLength = partEnd - m_ReadPacketLength;
        u32 offset = m_ReadPacketLength - partBegin;

        u32 readResult = m_Channel->Read(nw::snd::spy::internal::fnd::AddOffsetToPtr(m_PacketHeader, offset), readLength);
        if (readResult == static_cast<u32>(-1) || readResult == 0)
        {
            break;
        }

        m_ReadPacketLength += readResult;
    }

    if(m_ReadPacketLength < partEnd)
    {
        // ヘッダを完全に読み終わるまではアイドル状態にとどまる。
        m_State = STATE_IDLE;
        return false;
    }

    if(m_PacketHeader->bodyLength == 0)
    {
        m_State = STATE_READ_BODY;
    }
    else
    {
        m_State = STATE_READ_HEADER;
    }

    return true;
}

//----------------------------------------------------------
bool
SpyPacketReader::ReadPacketBody(void* buffer, u32 length)
{
    NW_ASSERT_NOT_NULL(buffer);

    if(!IsInitialized())
    {
        return false;
    }

    if(m_State == STATE_READ_BODY)
    {
        return false;
    }

    if(m_ReadPacketLength < sizeof(PacketHeader) || m_State != STATE_READ_HEADER)
    {
        NW_ASSERTMSG(false, "packet header not read!\n");
        return false;
    }

    const PacketHeader* packetHeader = GetCurrentPacketHeader();
    NW_ASSERT_NOT_NULL(packetHeader);
    NW_ASSERT(m_ReadPacketLength < packetHeader->bodyLength + sizeof(PacketHeader));
    // bufferにはパケットボディをすべて格納できるサイズが必要。
    NW_ASSERT(packetHeader->bodyLength <= length);

#if defined(COM_DEBUG_ENABLED)
    NW_LOG("[SpyPacketReader] ReadPacketBody() : bodyLength=%d.\n", packetHeader->bodyLength);
#endif

    m_State = STATE_READING_BODY;

    // HioChannel::Read()は指定の長さ未満で戻ることがあるので
    // 指定の長さに達するまで繰り返し呼び出す。
    const u32 partBegin = sizeof(PacketHeader);
    const u32 partEnd = partBegin + packetHeader->bodyLength;
    while (m_ReadPacketLength < partEnd)
    {
        u32 readLength = partEnd - m_ReadPacketLength;
        u32 offset = m_ReadPacketLength - partBegin;

        u32 readResult = m_Channel->Read(nw::snd::spy::internal::fnd::AddOffsetToPtr(buffer, offset), readLength);
        if (readResult == static_cast<u32>(-1) || readResult == 0)
        {
            break;
        }

        m_ReadPacketLength += readResult;
    }

    if(m_ReadPacketLength < partEnd)
    {
        // ボディを完全に読み終わるまではヘッダ読込済み状態にとどまる。
        m_State = STATE_READ_HEADER;
        return false;
    }

    m_State = STATE_READ_BODY;

    return true;
}

//----------------------------------------------------------
void
SpyPacketReader::Next()
{
    if(m_State == STATE_IDLE)
    {
        return;
    }

    NW_ASSERTMSG(m_State == STATE_READ_BODY, "invalid state(%s).\n", StateToString(m_State));

    m_ReadPacketLength = 0;
    m_State = STATE_IDLE;
}

//----------------------------------------------------------
const char*
SpyPacketReader::StateToString(State value)
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "STATE_NOT_INITIALIZED",
        "STATE_IDLE",
        "STATE_READING_HEADER",
        "STATE_READ_HEADER",
        "STATE_READING_BODY",
        "STATE_READ_BODY"
    };

    return strings[value];
#else
    (void)value;
    return NULL;
#endif
}

} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
