﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_SpyController.h>

#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>
#include <nw/snd/spy/fnd/basis/sndspyfnd_Time.h>
#include <nw/snd/spy/fnd/hio/sndspyfnd_HioChannel.h>
#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>
#include <nw/snd/spy/sndspy_SpyData.h>
#include <nw/snd/spy/protocol/sndspy_Packet.h>

#if defined(NW_DEBUG) || defined(NW_DEVELOP)
//#define MODULE_DEBUG_ENABLED
//#define STATE_DEBUG_ENABLED
#endif

// NN_TEXT() マクロにより日本語を埋め込めるようになるまで、C4566 を無効化
#if defined(NW_PLATFORM_WIN32)
#pragma warning( disable : 4566 )
#endif

namespace nw {
namespace snd {
namespace spy {

namespace {

#if defined(NW_PLATFORM_CAFE)
const u32 DATA_BUFFER_ALIGNMENT = FS_IO_BUFFER_ALIGN;
#else
const u32 DATA_BUFFER_ALIGNMENT = 1;
#endif

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
static SpyController::ConstPortType DEFAULT_SYNC_PORT = 65100;
static SpyController::ConstPortType DEFAULT_DATA_PORT = 65101;
#elif defined(NW_PLATFORM_CAFE)
static SpyController::ConstPortType DEFAULT_SYNC_PORT = "NWSoundSpy_Sync";
static SpyController::ConstPortType DEFAULT_DATA_PORT = "NWSoundSpy_Data";
#endif

static const char* SYNC_THREAD_NAME = "nw::snd::spy::SyncThread";
static const char* DATA_THREAD_NAME = "nw::snd::spy::DataThread";

}

//----------------------------------------------------------
SpyController::OpenArg::OpenArg() :
syncPort(DEFAULT_SYNC_PORT),
dataPort(DEFAULT_DATA_PORT),
syncThreadPriority(nw::snd::spy::internal::fnd::Thread::DEFAULT_THREAD_PRIORITY),
dataThreadPriority(nw::snd::spy::internal::fnd::Thread::DEFAULT_THREAD_PRIORITY)
{
}

//----------------------------------------------------------
SpyController::SpyController() :
m_State(STATE_NOT_INITIALIZED),
m_ModuleTop(NULL),
m_ModuleLast(NULL),
m_Module(NULL),
m_PushDataEvent(false, false),
m_IsSyncThreadEnabled(false),
m_IsDataThreadEnabled(false),
m_IsDataSelected(false)
{
    m_SessionMsgHandlerAdaptor.Initialize(this);

    m_Session.SetStateChangedCallback(&SpyController::SessionStateChangedCallback, this);

    m_CurrentDataBuffer = &m_DataBuffer1;
    m_SyncThreadHandler.Initialize(this, &SpyController::SyncThreadMain);
    m_DataThreadHandler.Initialize(this, &SpyController::DataThreadMain);
}

//----------------------------------------------------------
void
SpyController::Initialize(void* buffer, u32 bufferLength, u32 dataBufferLength)
{
    if(m_State >= STATE_INITIALIZED)
    {
        return;
    }

    // バッファサイズチェック
    NW_ASSERT(bufferLength >= GetRequiredMemorySize(dataBufferLength));

    nw::snd::spy::internal::fnd::FrameHeap frameHeap;
    frameHeap.Initialize(buffer, bufferLength);

    // スレッドスタック
#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    m_SyncThreadStack = frameHeap.Alloc(SYNC_THREAD_STACK_SIZE);
    m_DataThreadStack = frameHeap.Alloc(DATA_THREAD_STACK_SIZE);

    NW_ASSERT_NOT_NULL(m_SyncThreadStack);
    NW_ASSERT_NOT_NULL(m_DataThreadStack);
#endif

    // バッファ
    u32 alignedDataBufferSize =  nw::snd::spy::internal::fnd::RoundUp(
        dataBufferLength,
        DATA_BUFFER_ALIGNMENT);

    m_DataBuffer1.address = frameHeap.Alloc(alignedDataBufferSize, DATA_BUFFER_ALIGNMENT);
    m_DataBuffer2.address = frameHeap.Alloc(alignedDataBufferSize, DATA_BUFFER_ALIGNMENT);
    NW_ASSERT_NOT_NULL(m_DataBuffer1.address);
    NW_ASSERT_NOT_NULL(m_DataBuffer2.address);

    m_DataBuffer1.length = alignedDataBufferSize;
    m_DataBuffer2.length = alignedDataBufferSize;

    m_SpyDataFileChannel.Initialize(frameHeap.Alloc(internal::SpyDataFileChannel::GetRequiredMemorySize()));

    size_t restLength = frameHeap.GetFreeLength();

    m_Session.Initialize(
        frameHeap.Alloc(restLength),
        restLength);

    ClearDataBuffer();

    UnregisterAllConnectionChangedCallbacks();

    // モジュール
    InstallModule(m_TimeModule);
    InstallModule(m_LogModule);
    InstallModule(m_PlotModule);
    InstallModule(m_MarkerModule);

    SetState(STATE_INITIALIZED);
}

//----------------------------------------------------------
void
SpyController::Finalize()
{
    if(m_State <= STATE_NOT_INITIALIZED)
    {
        return;
    }

    Close();
    WaitForClose();

    ClearDataBuffer();

    // モジュール
    UninstallModule(m_TimeModule);
    UninstallModule(m_LogModule);
    UninstallModule(m_PlotModule);
    UninstallModule(m_MarkerModule);

    m_DataBuffer1.address = NULL;
    m_DataBuffer2.address = NULL;
    m_DataBuffer1.length = 0;
    m_DataBuffer2.length = 0;

    m_SpyDataFileChannel.Finalize();

    SetState(STATE_NOT_INITIALIZED);
}

//----------------------------------------------------------
size_t
SpyController::GetRequiredMemorySize(u32 dataBufferLength)
{
    NW_ASSERT(dataBufferLength > 0);

    size_t result = internal::SpySession::GetRequiredMemorySize();

    result += internal::SpyDataFileChannel::GetRequiredMemorySize();

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    result += SYNC_THREAD_STACK_SIZE + DATA_THREAD_STACK_SIZE;
#endif

    // ダブルバッファにするので、× 2
    result += DATA_BUFFER_ALIGNMENT;
    result += nw::snd::spy::internal::fnd::RoundUp(dataBufferLength, DATA_BUFFER_ALIGNMENT) * 2;

    return result;
}

//----------------------------------------------------------
bool
SpyController::Open(const OpenArg& arg)
{
    NW_ASSERTMSG(IsInitialized(), "SpySession is not initialized.\n");

    {
        nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

        if (m_State > STATE_INITIALIZED)
        {
            return false;
        }

        SetState(STATE_OPENING);
    }

    m_SpyDataFileChannel.Open(arg.dataFileChannelParam);

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    m_SyncPort = arg.syncPort;
    m_DataPort = arg.dataPort;
#elif defined(NW_PLATFORM_CAFE)
    NW_ASSERT_NOT_NULL(arg.syncPort);
    NW_ASSERT_NOT_NULL(arg.dataPort);
    nw::snd::spy::internal::fnd::String::Copy(m_SyncPort, nw::snd::spy::internal::fnd::HioChannel::MAX_PORT_LENGTH, arg.syncPort);
    nw::snd::spy::internal::fnd::String::Copy(m_DataPort, nw::snd::spy::internal::fnd::HioChannel::MAX_PORT_LENGTH, arg.dataPort);
#endif

    ResetSpyFrameBase();

    if(!StartSyncThread(arg.syncThreadPriority))
    {
        Close();
        return false;
    }

    if(!StartDataThread(arg.dataThreadPriority))
    {
        Close();
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
SpyController::Close()
{
    NW_ASSERTMSG(IsInitialized(), "SpySession is not initialized.\n");

    {
        nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

        if (m_State < STATE_OPENING)
        {
            return;
        }

        SetState(STATE_DISCONNECTING);
    }

    // スレッド関数から抜けるためのフラグを設定後に、PushData 待ちを解消する
    m_IsDataThreadEnabled = false;
    m_IsSyncThreadEnabled = false;

    m_PushDataEvent.Set();
}

//----------------------------------------------------------
bool
SpyController::WaitForClose()
{
    if(!IsClosing())
    {
        return false;
    }

    // スレッドが終了したらクローズ完了とみなす
    if (m_SyncThread.GetState() != nw::snd::spy::internal::fnd::Thread::STATE_NOT_RUN)
    {
        m_SyncThread.WaitForExit();
    }
    if (m_DataThread.GetState() != nw::snd::spy::internal::fnd::Thread::STATE_NOT_RUN)
    {
        m_DataThread.WaitForExit();
    }

    return true;
}

//----------------------------------------------------------
bool
SpyController::RegisterConnectionChangedCallback(
    ConnectionChangedCallback callback,
    void* userparam/*= NULL*/)
{
    NW_ASSERT_NOT_NULL(callback);

    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    for(u32 index = 0; index < MAX_CONNECTION_CHANGED_CALLBACK_COUNT; ++index)
    {
        if(m_CallbackInfos[index].callback != NULL)
        {
            continue;
        }

        m_CallbackInfos[index].callback = callback;
        m_CallbackInfos[index].param = userparam;

        return true;
    }

    return false;
}

//----------------------------------------------------------
void
SpyController::UnregisterConnectionChangedCallback(ConnectionChangedCallback callback)
{
    NW_ASSERT_NOT_NULL(callback);

    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    for(u32 index = 0; index < MAX_CONNECTION_CHANGED_CALLBACK_COUNT; ++index)
    {
        if(m_CallbackInfos[index].callback != callback)
        {
            continue;
        }

        m_CallbackInfos[index].callback = NULL;
        m_CallbackInfos[index].param = NULL;
        break;
    }
}

//----------------------------------------------------------
void
SpyController::SetCurrentAppFrame(u32 value)
{
    if(GetCurrentAppFrame() == value)
    {
        return;
    }

    m_TimeModule.SetCurrentAppFrame(value);
    m_TimeModule.PushCurrentTimeData();
}

//----------------------------------------------------------
void
SpyController::SetCurrentAudioFrame(u32 value)
{
    if(GetCurrentAudioFrame() == value)
    {
        return;
    }

    m_TimeModule.SetCurrentAudioFrame(value);
    m_TimeModule.PushCurrentTimeData();
}

//----------------------------------------------------------
bool
SpyController::InstallModule(SpyModule& module)
{
    if (!module.IsInstalled())
    {
        // バージョン番号のチェック。
        NW_ASSERT(module.GetDataInfo().GetDataVersion() != 0);
        if (module.GetDataInfo().GetDataVersion() == 0)
        {
            return false;
        }

        // データタイプ名のチェック。
        {
            const char* dataName = module.GetDataInfo().GetDataName();
            NW_NULL_ASSERT(dataName);
            if (dataName == NULL)
            {
                return false;
            }
            else
            {
                size_t dataNameLength = strlen(dataName);
                NW_ASSERT_MINMAX(dataNameLength, 1, 128);
                if (dataNameLength < 1 || 128 < dataNameLength)
                {
                    return false;
                }
            }
        }

        // リンク
        NW_ASSERTMSG(module.GetPrev() == NULL && module.GetNext() == NULL, "インストールする SpyModule は、リンクが切れている必要があります。");

        if(m_ModuleTop == NULL)
        {
            m_ModuleTop = &module;
            m_ModuleLast = &module;
        }
        else
        {
            NW_ASSERT(m_ModuleLast != NULL);
            m_ModuleLast->SetNext(&module);
            module.SetPrev(m_ModuleLast);
            m_ModuleLast = &module;
        }
    }
    else
    {
        if(module.GetController() != this)
        {
            NW_WARNING(false, "複数の SpyController にはインストールできません。");
            return false;
        }
    }

    module.Attach(*this);
    DumpAllModules();

    return true;
}

//----------------------------------------------------------
void
SpyController::UninstallModule(SpyModule& module)
{
    if(module.GetController() != this)
    {
        NW_ASSERTMSG(false, "この SpyController にインストールされている SpyModule を指定してください。");
        return;
    }

    // Prev -> Next をつなぐ
    // Top を更新する
    if(module.GetPrev() != NULL)
    {
        module.GetPrev()->SetNext(module.GetNext());
    }
    else
    {
        NW_ASSERT(m_ModuleTop == &module);
        m_ModuleTop = module.GetNext();
    }

    // Prev <- Next をつなぐ
    // Last を更新する
    if(module.GetNext() != NULL)
    {
        module.GetNext()->SetPrev(module.GetPrev());
    }
    else
    {
        NW_ASSERT(m_ModuleLast == &module);
        m_ModuleLast = module.GetPrev();
    }

    // Current を更新する
    if(m_Module == &module)
    {
        m_Module = m_ModuleTop;
    }

    module.Detach();

    DumpAllModules();
}

//----------------------------------------------------------
bool
SpyController::PushData(SpyModule& module, const void* buffer, u32 length)
{
    NW_ASSERT(buffer);
    NW_ASSERT(length > 0);

    // Push 対象でない場合は失敗させます。
    if(!module.IsInstalled() || !module.IsRequested())
    {
        return false;
    }

    return PushDataPacket(module.GetDataInfo().GetDataID(), buffer, length);
}

//----------------------------------------------------------
void
SpyController::ClearDataBuffer()
{
    m_DataBuffer1.currentPosition = 0;
    m_DataBuffer2.currentPosition = 0;
}

//----------------------------------------------------------
void
SpyController::OnSessionStateChanged()
{
    if(m_Session.IsPrepared())
    {
        OnSessionConnected();
    }

    if(!m_Session.IsConnected() && IsConnected())
    {
        OnSessionDisconnected();
    }

    if(!m_Session.IsOpened() && IsOpened())
    {
        OnSessionClosed();
    }
}

//----------------------------------------------------------
void
SpyController::OnSessionConnected()
{
    InitializeDataInfo();

    // Initializeパケットを受信すると、SpyController は STATE_PREPARING に遷移します。
    // 付加的な初期化パケットを受信したのち、最後に有効なフラグを持った
    // SelectDataIDパケットを受信すると、STATE_PREPARED に遷移します。
    SetState(STATE_PREPARING);
}

//----------------------------------------------------------
void
SpyController::OnSessionDisconnected()
{
    FinalizeDataInfo();
}

//----------------------------------------------------------
void
SpyController::OnSessionClosed()
{
    // SpyController がクローズ中でなければ、SpySession を再オープンするために
    // SpyController の状態を STATE_OPENING に更新する
    if (!IsClosing())
    {
        SetState(STATE_OPENING);
    }
}

//----------------------------------------------------------
void
SpyController::ResetSpyFrameBase()
{
    m_TimeModule.ResetSpyFrame();

    // Dataパケットの送信時間(Timestamp)の基準になります。
    m_TimestampBase = nw::snd::spy::internal::fnd::Time::Current();
    m_LastTimestamp = 0;
}

//----------------------------------------------------------
bool
SpyController::StartSyncThread(u32 priority)
{
    // 未リリース時は、リリースする
    if (m_SyncThread.GetState() == nw::snd::spy::internal::fnd::Thread::STATE_EXITED)
    {
        m_SyncThread.Release();
    }

    m_IsSyncThreadEnabled = true;

    nw::snd::spy::internal::fnd::Thread::RunArgs args;
    args.name = SYNC_THREAD_NAME;
    args.handler = &m_SyncThreadHandler;
    args.priority = priority;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    args.stack = m_SyncThreadStack;
    args.stackSize = SYNC_THREAD_STACK_SIZE;
#endif

    return m_SyncThread.Run(args);
}

//----------------------------------------------------------
bool
SpyController::StartDataThread(u32 priority)
{
    // 未リリース時は、リリースする
    if (m_DataThread.GetState() == nw::snd::spy::internal::fnd::Thread::STATE_EXITED)
    {
        m_DataThread.Release();
    }

    m_IsDataThreadEnabled = true;

    m_PushDataEvent.Reset();

    nw::snd::spy::internal::fnd::Thread::RunArgs args;
    args.name = DATA_THREAD_NAME;
    args.handler = &m_DataThreadHandler;
    args.priority = priority;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
#else
    args.stack = m_DataThreadStack;
    args.stackSize = DATA_THREAD_STACK_SIZE;
#endif

    return m_DataThread.Run(args);
}

//----------------------------------------------------------
u32
SpyController::SyncThreadMain(void* param)
{
    (void)param;

    while(m_IsSyncThreadEnabled)
    {
        if(m_State == STATE_OPENING)
        {
            NW_ASSERT(!m_Session.IsOpened());

            if(m_Session.Open(m_SyncPort, m_DataPort))
            {
                SetState(STATE_CONNECTING);

                // 成功したら、スリープせずに即座に次の処理へ
                continue;
            }
        }
        else if(m_State >= STATE_CONNECTING)
        {
            if (!m_Session.IsConnected())
            {
                if (m_Session.Connect())
                {
                    // 成功したら、スリープせずに即座に次の処理へ
                    continue;
                }

                // NOTE : ここでは状態遷移しない（初期化が終わるまで接続完了としない）
            }
            else if (m_Session.IsPreparing() || m_Session.IsPrepared())
            {
                if (m_Session.ProcessSyncPacket(&m_SessionMsgHandlerAdaptor))
                {
                    // 成功したら、スリープせずに即座に次の処理へ
                    continue;
                }
            }
        }

        // 何かしらの理由で失敗した場合は、スリープして、再試行
        nw::snd::spy::internal::fnd::Thread::Sleep(
            nw::snd::spy::internal::fnd::TimeSpan::FromMilliSeconds(10));
    }

    CloseImpl();

    return 0;
}

//----------------------------------------------------------
u32
SpyController::DataThreadMain(void* param)
{
    (void)param;

    for(;;)
    {
        if(!m_IsDataThreadEnabled)
        {
            break;
        }

        if(!IsPrepared())
        {
            nw::snd::spy::internal::fnd::Thread::Sleep(
                nw::snd::spy::internal::fnd::TimeSpan::FromMilliSeconds(100));
            continue;
        }

        if(GetCurrentDataBuffer().currentPosition == 0)
        {
            m_PushDataEvent.Wait();
            continue;
        }

        SwapBuffer();
        SendDataPacket();
    }

    return 0;
}

//----------------------------------------------------------
void
SpyController::CloseImpl()
{
    // この関数は SyncThread から呼ばれる前提。
    // スレッドインスタンスの後始末はここでは行わず、再 Open() または Finalize() 時に行う。

    NW_ASSERT(m_State == STATE_DISCONNECTING);

    m_Session.Close();

    m_IsDataThreadEnabled = false;
    m_IsSyncThreadEnabled = false;

#if defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    m_SyncPort = nw::snd::spy::internal::fnd::HioChannel::INVALID_PORT;
    m_DataPort = nw::snd::spy::internal::fnd::HioChannel::INVALID_PORT;
#elif defined(NW_PLATFORM_CAFE)
    m_SyncPort[0] = '\0';
    m_DataPort[0] = '\0';
#endif

    m_SpyDataFileChannel.Close();

    SetState(STATE_INITIALIZED);
}

//----------------------------------------------------------
void
SpyController::SetState(State value)
{
    if(m_State == value)
    {
        return;
    }

    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

#ifdef STATE_DEBUG_ENABLED
    NW_LOG("[nw::snd::spy::SpyController] State = %s --> %s\n", StateToString(m_State), StateToString(value));
#endif

    bool IsConnectedOld = IsConnected();

    m_State = value;

    if(IsConnectedOld != IsConnected())
    {
        NotifyConnectionChanged();
    }
}

//----------------------------------------------------------
void
SpyController::UnregisterAllConnectionChangedCallbacks()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_StateLock);

    for(u32 index = 0; index < MAX_CONNECTION_CHANGED_CALLBACK_COUNT; ++index)
    {
        m_CallbackInfos[index].callback = NULL;
        m_CallbackInfos[index].param = NULL;
    }
}

//----------------------------------------------------------
void
SpyController::NotifyConnectionChanged()
{
    // NOTE : m_StateLock された状態で呼び出される必要があります。

    for(u32 index = 0; index < MAX_CONNECTION_CHANGED_CALLBACK_COUNT; ++index)
    {
        if(m_CallbackInfos[index].callback != NULL)
        {
            m_CallbackInfos[index].callback(m_CallbackInfos[index].param);
        }
    }
}

//----------------------------------------------------------
bool
SpyController::PushDataPacket(SpyDataID dataID, const void* buffer, u32 length)
{
    NW_ASSERT_MINMAX(dataID, DATA_ID_MIN, DATA_ID_MAX);
    NW_ASSERT(buffer);
    NW_ASSERT(length > 0);

    // 準備が完了していない場合は失敗させます。
    if(!IsPrepared())
    {
        return false;
    }

    nw::snd::spy::internal::fnd::ScopedCriticalSection dataLock(m_DataBufferLock);

    internal::DataPacket* packet = AllocateDataPacketBuffer(length);

    if(packet == NULL)
    {
        NW_LOG("[nw::snd::spy::SpyController::PushDataPacket] out of memory.\n");
        return false;
    }

    // Dataパケットに送信時間(Timestamp)を設定します。
    // Timestampはほぼ実時間として扱えますがフレームが区別できるように補正します。
    {
        // 計測精度はmsec。usecのオーダーは補正に使います。
        u64 timestamp = static_cast<u64>((internal::fnd::Time::Current() - m_TimestampBase).ToMilliSeconds()) * 1000;

        // パケットの送信順を把握するため、必ず新しいTimestampを割り当てます。
        timestamp = internal::fnd::Max(m_LastTimestamp + 1, timestamp);

        packet->SetTimestamp(timestamp);
        m_LastTimestamp = timestamp;
    }

    packet->SetLengths(length);
    packet->body.dataID = dataID;
    memcpy(packet->GetPayload(), buffer, length);

#ifdef BUFFER_DEBUG_ENABLED
    NW_LOG(
        "[nw::snd::spy::SpyController] PushData   : Current(%8d/%8d), Send(%8d/%8d).\n",
        GetCurrentDataBuffer().currentPosition,
        GetCurrentDataBuffer().length,
        GetSendDataPacketBuffer().currentPosition,
        GetSendDataPacketBuffer().length);
#endif

    m_PushDataEvent.Set();

    return true;
}

//----------------------------------------------------------
internal::DataPacket*
SpyController::AllocateDataPacketBuffer(u32 payloadLength)
{
    // m_DataBufferLock でロックがかかった状態で呼び出されることを想定しており、
    // この関数内ではロックしません。

    u32 requiredLength = internal::DataPacket::GetRequiredMemorySize(payloadLength);

    if(GetCurrentDataBuffer().length - GetCurrentDataBuffer().currentPosition < requiredLength)
    {
        // メモリ不足
        return NULL;
    }

    void* buffer = nw::snd::spy::internal::fnd::AddOffsetToPtr(
        GetCurrentDataBuffer().address,
        GetCurrentDataBuffer().currentPosition);

    GetCurrentDataBuffer().currentPosition += requiredLength;

    return new(buffer) internal::DataPacket;
}

//----------------------------------------------------------
void
SpyController::SendDataPacket()
{
    if(GetSendDataPacketBuffer().currentPosition == 0)
    {
        return;
    }

#ifdef BUFFER_DEBUG_ENABLED
    // データが溜まっていない場合はログ出力しない
    if(GetSendDataPacketBuffer().currentPosition > 0)
    {
        NW_LOG(
            "[nw::snd::spy::SpyController] SendData : (%8d/%8d).\n",
            GetSendDataPacketBuffer().currentPosition,
            GetSendDataPacketBuffer().length);
    }
#endif

    bool bOutputByFile = m_SpyDataFileChannel.WriteData(
        this->GetSendDataPacketBuffer().address,
        this->GetSendDataPacketBuffer().currentPosition);

    if (bOutputByFile)
    {
        // Dataチャンネルの生存を通知。
        internal::PingPacket* packet = new(GetSendDataPacketBuffer().address) internal::PingPacket();
        m_Session.SendData(packet, sizeof(*packet));
    }
    else
    {
         // ファイルの代わりにDataチャンネルで送信する。
         m_Session.SendData(
             GetSendDataPacketBuffer().address,
             GetSendDataPacketBuffer().currentPosition);
    }

    GetSendDataPacketBuffer().currentPosition = 0;
}

//----------------------------------------------------------
bool
SpyController::SendSetOutputDirPathReplyPacket(bool active)
{
#ifdef BUFFER_DEBUG_ENABLED
    NW_LOG("[nw::snd::spy::SpyController] SendSetOutputDirPathReplyPacket(active=%s).\n", active? "true" : "false");
#endif

    // 準備中でない場合は失敗させます。
    if (!IsPreparing())
    {
        return false;
    }

    // 有効なデータIDフラグが設定されている場合、バックグラウンドスレッドで
    // Dataチャンネルへの出力が行われているので、ここでDataチャンネルへの
    // 書き込みを行うことはできない。
    if (m_IsDataSelected)
    {
        NW_ASSERT(!m_IsDataSelected);
        return false;
    }

    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_DataBufferLock);

    internal::SetOutputDirReplyPacket packet;
    packet.header.resultCode = active? 1 : 0;

    if(m_Session.SendData(&packet, sizeof(packet)) != sizeof(packet))
    {
        NW_WARNING(false, "could not send SetOutputDirReplyPacket.");
        return false;
    }

    return true;
}

//----------------------------------------------------------
void
SpyController::SwapBuffer()
{
    nw::snd::spy::internal::fnd::ScopedCriticalSection lock(m_DataBufferLock);

    m_CurrentDataBuffer = &GetSendDataPacketBuffer();
}

//----------------------------------------------------------
void
SpyController::SelectDataIDs(u32 numFlags, u32 flags[])
{
    // Initializeパケットを受信するとPREPARING状態に入ります。
    // 付加的な初期化パケットを受信したのち、最後に、
    // 有効なフラグを持ったSelectDataIDパケットを受信すると
    // PREPARED 状態に遷移します。
    if (IsPreparing() && numFlags > 0)
    {
        SetState(STATE_PREPARED);
    }

    bool dataSelected = false;
    for (SpyModule* module = m_ModuleTop;
         module != NULL;
         module = module->GetNext())
    {
        u32 dataId = module->GetDataInfo().GetDataID();

        u32 index = dataId / 32;
        u32 mask = 1 << (dataId % 32);

        if (index < numFlags && (flags[index] & mask) != 0)
        {
            module->OnRequested(true);
            dataSelected = true;
        }
        else
        {
            module->OnRequested(false);
        }
    }

    m_IsDataSelected = dataSelected;

    DumpAllModules();
}

//----------------------------------------------------------
void
SpyController::InitializeDataInfo()
{
    u32 no = 0;
    for (SpyModule* module = m_ModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        module->AllocateDataID(static_cast<SpyDataID>(++no));
        module->OnSessionStarted();
    }

    m_Module = m_ModuleTop;
    m_IsDataSelected = false;
}

//----------------------------------------------------------
void
SpyController::FinalizeDataInfo()
{
    for (SpyModule* module = m_ModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        module->OnRequested(false);
    }

    m_IsDataSelected = false;
}

//----------------------------------------------------------
const SpyDataInfo*
SpyController::QueryDataInfo()
{
    SpyModule* module = m_Module;

    if (module == NULL)
    {
        return NULL;
    }
    else
    {
        m_Module = module->GetNext();
    }

    return &module->GetDataInfo();
}


//----------------------------------------------------------
void
SpyController::DumpAllModules()
{
#if defined(MODULE_DEBUG_ENABLED)
    NW_LOG("-- all SpyModules --------------------\n");

    for (SpyModule* module = m_ModuleTop;
        module != NULL;
        module = module->GetNext())
    {
        NW_LOG("[0x%08x] ID=%2d Prev=0x%08x Next=0x%08x : %s\n",
            module,
            module->GetDataInfo().GetDataID(),
            module->GetPrev(),
            module->GetNext(),
            module->GetDataInfo().GetDataName());
    }

    NW_LOG("--------------------------------------\n");
#endif
}

//----------------------------------------------------------
const char*
SpyController::StateToString(State value)
{
#if defined(STATE_DEBUG_ENABLED)
    static const char* strings[] = {
        "STATE_NOT_INITIALIZED",
        "STATE_INITIALIZED",
        "STATE_OPENING",
        "STATE_DISCONNECTING",
        "STATE_CONNECTING",
        "STATE_PREPARING",
        "STATE_PREPARED"
    };

    return strings[value];
#else
    (void)value;
    return NULL;
#endif
}

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
