﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/sndspy_SpyController.h>
#include <nw/snd/spy/modules/sndspy_PlotGroup.h>
#include <nw/snd/spy/modules/sndspy_PlotItem.h>
#include <nw/snd/spy/modules/sndspy_PlotModule.h>
#include <nw/snd/spy/modules/sndspy_PlotState.h>

// NN_TEXT() マクロにより日本語を埋め込めるようになるまで、C4566 を無効化
#if defined(NW_PLATFORM_WIN32)
#pragma warning( disable : 4566 )
#endif

namespace nw {
namespace snd {
namespace spy {

//----------------------------------------------------------
PlotModule::PlotModule()
    : SpySimpleModule("Plot", SpyPlotItemPacket::PACKET_VERSION)
    , m_ItemTop(NULL)
    , m_ItemLast(NULL)
    , m_StateTop(NULL)
    , m_StateLast(NULL)
    , m_GroupTop(NULL)
    , m_GroupLast(NULL)
    , m_IsMetadataPushed(false)
    , m_IsResetPushed(false)
{
}

//----------------------------------------------------------
bool PlotModule::AttachItem(PlotItem& item)
{
    if(item.IsAttached())
    {
        if (item.GetModule() != this)
        {
            NW_WARNING(false, "SpyPlotItem は、複数の SpyPlotModule にアタッチできません。");
            return false;
        }

        return true;
    }

    if(m_ItemTop == NULL)
    {
        m_ItemTop = &item;
        m_ItemLast = &item;
    }
    else
    {
        NW_ASSERT(m_ItemLast != NULL);
        m_ItemLast->SetNext(&item);
        item.SetPrev(m_ItemLast);
        m_ItemLast = &item;
    }

    item.Attach(*this);

    return true;
}

//----------------------------------------------------------
void PlotModule::DetachItem(PlotItem& item)
{
    if (!item.IsAttached())
    {
        return;
    }

    if (item.GetModule() != this)
    {
        NW_ASSERTMSG(false, "アイテムは別のモジュールにアタッチされています。");
        return;
    }

    // Prev -> Next をつなぐ
    // Top を更新する
    if(item.GetPrev() != NULL)
    {
        item.GetPrev()->SetNext(item.GetNext());
    }
    else
    {
        NW_ASSERT(m_ItemTop == &item);
        m_ItemTop = item.GetNext();
    }

    // Prev <- Next をつなぐ
    // Last を更新する
    if(item.GetNext() != NULL)
    {
        item.GetNext()->SetPrev(item.GetPrev());
    }
    else
    {
        NW_ASSERT(m_ItemLast == &item);
        m_ItemLast = item.GetPrev();
    }

    item.Detach();
}

//----------------------------------------------------------
bool PlotModule::AttachState(PlotState& state)
{
    if(state.IsAttached())
    {
        if (state.GetModule() != this)
        {
            NW_WARNING(false, "spy::PlotState は、複数の spy::PlotModule にアタッチできません。");
            return false;
        }

        return true;
    }

    if(m_StateTop == NULL)
    {
        m_StateTop = &state;
        m_StateLast = &state;
    }
    else
    {
        NW_ASSERT(m_StateLast != NULL);
        m_StateLast->SetNext(&state);
        state.SetPrev(m_StateLast);
        m_StateLast = &state;
    }

    state.Attach(*this);

    return true;
}

//----------------------------------------------------------
void PlotModule::DetachState(PlotState& state)
{
    if (!state.IsAttached())
    {
        return;
    }

    if (state.GetModule() != this)
    {
        NW_ASSERTMSG(false, "ステートは別のモジュールにアタッチされています。");
        return;
    }

    // Prev -> Next をつなぐ
    // Top を更新する
    if(state.GetPrev() != NULL)
    {
        state.GetPrev()->SetNext(state.GetNext());
    }
    else
    {
        NW_ASSERT(m_StateTop == &state);
        m_StateTop = state.GetNext();
    }

    // Prev <- Next をつなぐ
    // Last を更新する
    if(state.GetNext() != NULL)
    {
        state.GetNext()->SetPrev(state.GetPrev());
    }
    else
    {
        NW_ASSERT(m_StateLast == &state);
        m_StateLast = state.GetPrev();
    }

    state.Detach();
}

//----------------------------------------------------------
bool PlotModule::AttachGroup(PlotGroup& group)
{
    if(group.IsAttached())
    {
        if (group.GetModule() != this)
        {
            NW_WARNING(false, "spy::PlotGroup は、複数の spy::PlotModule にアタッチできません。");
            return false;
        }

        return true;
    }

    if(m_GroupTop == NULL)
    {
        m_GroupTop = &group;
        m_GroupLast = &group;
    }
    else
    {
        NW_ASSERT(m_GroupLast != NULL);
        m_GroupLast->SetNext(&group);
        group.SetPrev(m_GroupLast);
        m_GroupLast = &group;
    }

    group.Attach(*this);

    return true;
}

//----------------------------------------------------------
void PlotModule::DetachGroup(PlotGroup& group)
{
    if (!group.IsAttached())
    {
        return;
    }

    if (group.GetModule() != this)
    {
        NW_ASSERTMSG(false, "グループは別のモジュールにアタッチされています。");
        return;
    }

    if (IsRequested())
    {
        group.PushDetachPacket();
    }

    // Prev -> Next をつなぐ
    // Top を更新する
    if(group.GetPrev() != NULL)
    {
        group.GetPrev()->SetNext(group.GetNext());
    }
    else
    {
        NW_ASSERT(m_GroupTop == &group);
        m_GroupTop = group.GetNext();
    }

    // Prev <- Next をつなぐ
    // Last を更新する
    if(group.GetNext() != NULL)
    {
        group.GetNext()->SetPrev(group.GetPrev());
    }
    else
    {
        NW_ASSERT(m_GroupLast == &group);
        m_GroupLast = group.GetPrev();
    }

    group.Detach();
}

//------------------------------------------------------------------------------
void PlotModule::OnSessionStarted()
{
    m_IsMetadataPushed = false;
    m_IsResetPushed = false;

    SpySimpleModule::OnSessionStarted();
}

//------------------------------------------------------------------------------
void PlotModule::OnRequested(bool isRequested)
{
    SpySimpleModule::OnRequested(isRequested);

    if(isRequested)
    {
        m_IsResetPushed = false;
    }

    // Plot データが要求されたら、メタデータを送信するように設定する
    if(isRequested && !m_IsMetadataPushed)
    {
        for (PlotGroup* pGroup = m_GroupTop; pGroup != NULL; pGroup = pGroup->GetNext())
        {
            pGroup->InvalidateMetadata();
        }

        for (PlotState* pState = m_StateTop; pState != NULL; pState = pState->GetNext())
        {
            pState->InvalidateMetadata();
        }

        for (PlotItem* item = m_ItemTop; item != NULL; item = item->GetNext())
        {
            item->InvalidateMetadata();
        }

        m_IsMetadataPushed = true;
    }
}

bool PlotModule::PushData(const void* buffer, u32 length)
{
    if (!m_IsResetPushed)
    {
        m_IsResetPushed = true;

        SpyPlotItemPacket::ResetPacketPayload payload = { 0 };
        payload.common.dataType = SpyPlotItemPacket::DataType_Reset;

        GetController()->PushData(*this, &payload, sizeof(payload));
    }

    return GetController()->PushData(*this, buffer, length);
}

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
