﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../precompiled.h"

#ifdef NW_SND_SPY_ENABLE

#include <nw/snd/spy/fnd/string/sndspyfnd_String.h>
#include <nw/snd/spy/modules/sndspy_PlotGroup.h>
#include <nw/snd/spy/modules/sndspy_PlotItem.h>
#include <nw/snd/spy/modules/sndspy_PlotModule.h>

// NN_TEXT() マクロにより日本語を埋め込めるようになるまで、C4566 を無効化
#if defined(NW_PLATFORM_WIN32)
#pragma warning( disable : 4566 )
#endif

namespace nw {
namespace snd {
namespace spy {

//------------------------------------------------------------------------------
PlotItem::PlotItem()
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    SetName("");
    SetRange(0, 0);
    SetColor(0xFF, 0, 0);

    InitializePacketCommonData();
}

//------------------------------------------------------------------------------
PlotItem::PlotItem(const char* name)
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    SetName(name);
    SetRange(0, 0);
    SetColor(0xFF, 0, 0);

    InitializePacketCommonData();
}

//------------------------------------------------------------------------------
PlotItem::PlotItem(const char* name, double minValue, double maxValue, u8 r, u8 g, u8 b)
    : m_Module(NULL)
    , m_Group(NULL)
    , m_Prev(NULL)
    , m_Next(NULL)
    , m_IsMetadataPushed(false)
{
    SetName(name);
    SetRange(minValue, maxValue);
    SetColor(r, g, b);

    InitializePacketCommonData();
}

//------------------------------------------------------------------------------
PlotItem::~PlotItem()
{
    if(IsAttached())
    {
        m_Module->DetachItem(*this);
    }
}

//------------------------------------------------------------------------------
void PlotItem::SetName(const char* name)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、SpyPlotItem の名前を変更できません。");
        return;
    }

    if(name == NULL)
    {
        NW_WARNING(false, "名前が指定されていません。");
        return;
    }

    m_MetadataPacketPayload.name.Set(name);
}

//------------------------------------------------------------------------------
void PlotItem::SetRange(double minValue, double maxValue)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、SpyPlotItem の範囲を変更できません。");
        return;
    }

    m_MetadataPacketPayload.minValue = minValue;
    m_MetadataPacketPayload.maxValue = maxValue;
}

//------------------------------------------------------------------------------
void PlotItem::SetColor(u8 r, u8 g, u8 b)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、SpyPlotItem の色を変更できません。");
        return;
    }

    m_MetadataPacketPayload.color.r = r;
    m_MetadataPacketPayload.color.g = g;
    m_MetadataPacketPayload.color.b = b;
    m_MetadataPacketPayload.color.a = 0xFF;
}

//------------------------------------------------------------------------------
void PlotItem::SetInterpolationMode(InterpolationMode mode)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、SpyPlotItem の補間モードを変更できません。");
        return;
    }

    m_MetadataPacketPayload.interpolationMode = static_cast<u8>(mode);
}

//------------------------------------------------------------------------------
void PlotItem::SetGroup(PlotGroup* pGroup)
{
    if(IsRequested())
    {
        NW_WARNING(false, "通信中は、SpyPlotItem のグループを設定できません。");
        return;
    }

    m_Group = pGroup;

    if (pGroup != NULL)
    {
        m_MetadataPacketPayload.groupId = pGroup->GetGroupId();
    }
    else
    {
        m_MetadataPacketPayload.groupId = 0;
    }
}

//------------------------------------------------------------------------------
void PlotItem::PlotValue(double value)
{
    if(!IsRequested()) { return; }

    // メタデータをプッシュしていない場合は先にプッシュする
    if(!IsMetadataPushed())
    {
        PushMetaData();
    }

    m_ValuePacketPayload.value = value;

    m_Module->PushData(m_ValuePacketPayload);
}

//------------------------------------------------------------------------------
void PlotItem::InitializePacketCommonData()
{
    m_MetadataPacketPayload.common.dataType = SpyPlotItemPacket::DataType_ItemMetadata;
    m_MetadataPacketPayload.id = reinterpret_cast<u32>(this);
    m_MetadataPacketPayload.groupId = 0;

    m_ValuePacketPayload.common.dataType = SpyPlotItemPacket::DataType_ItemValue;
    m_ValuePacketPayload.id = reinterpret_cast<u32>(this);
}

//------------------------------------------------------------------------------
void PlotItem::Attach(PlotModule& module)
{
    NW_ASSERTMSG(m_Module == NULL || m_Module == &module, "この SpyPlotItem は、すでに他の SpyPlotModule にアタッチされています。");

    m_Module = &module;
    InvalidateMetadata();
}

//------------------------------------------------------------------------------
void PlotItem::Detach()
{
    m_Module = NULL;
    SetPrev(NULL);
    SetNext(NULL);
}

//------------------------------------------------------------------------------
void PlotItem::PushMetaData()
{
    NW_ASSERT(IsRequested());

    if (m_Group != NULL && !m_Group->IsMetadataPushed())
    {
        m_Group->PushMetadata();
    }

    if(!m_Module->PushData(m_MetadataPacketPayload))
    {
        NW_WARNING(false, "PlotItem '%s' のメタデータ送信に失敗しました。", GetName());
    }

    m_IsMetadataPushed = true;
}

} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_SPY_ENABLE
