﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "../../../precompiled.h"

#include <nw/snd/spy/fnd/basis/sndspyfnd_Memory.h>

//#define FND_ENABLE_MEM_DEBUG

namespace nw {
namespace snd {
namespace spy {
namespace internal {
namespace fnd {

//---------------------------------------------------------------------------
void
FrameHeap::Initialize(void* buffer, size_t length)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);

    m_Buffer = buffer;
    m_BufferLength = length;
}

//---------------------------------------------------------------------------
void
FrameHeap::Finalize()
{
    m_Buffer = NULL;
    m_BufferLength = 0;
}

//---------------------------------------------------------------------------
size_t
FrameHeap::GetFreeLength(s32 alignment/*= MemoryTraits::DEFAULT_ALIGNMENT*/) const
{
    if(!IsInitialized()) { return 0; }

    void* result = AddOffsetToPtr(m_Buffer, m_UsedLength, alignment);
    return m_BufferLength - GetOffsetFromPtr(m_Buffer, result);
}

//---------------------------------------------------------------------------
void*
FrameHeap::Alloc(size_t size, s32 alignment/*= MemoryTraits::DEFAULT_ALIGNMENT*/)
{
    NW_ASSERT(alignment > 0);

    if(!IsInitialized())
    {
        NW_FATAL_ERROR("FrameHeap is not initialized.\n");
        return NULL;
    }

    if(size == 0)
    {
        return NULL;
    }

    void*  result = AddOffsetToPtr(m_Buffer, m_UsedLength, alignment);
    size_t newUsedLength = GetOffsetFromPtr(m_Buffer, result) + size;

    if(newUsedLength > m_BufferLength)
    {
        NW_WARNING(false, "out of memory.\n");
        return NULL;
    }

    m_UsedLength = newUsedLength;

    return result;
}

//---------------------------------------------------------------------------
void
UnitHeap::Initialize(
    void* buffer,
    size_t length,
    size_t unitLength,
    u32 unitCount,
    u32 unitAlignment/*= MemoryTraits::DEFAULT_ALIGNMENT*/)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);
    NW_ASSERT(unitLength > 0);
    NW_ASSERT(unitLength < static_cast<size_t>((std::numeric_limits<s32>::max)()));
    NW_ASSERT(unitCount > 0);
    NW_ASSERT(m_AllocatedUnitCount == 0);

    void*  alignedBuffer = RoundUp(buffer, unitAlignment);
    size_t alignedBufferLength = length - GetOffsetFromPtr(buffer, alignedBuffer);
    size_t alignedUnitLength = RoundUp(unitLength, unitAlignment);

    if(alignedBufferLength < alignedUnitLength * unitCount)
    {
        NW_FATAL_ERROR("out of memory.\n");
        return;
    }

    for(u32 unitIndex = 0; unitIndex < unitCount - 1; ++unitIndex)
    {
        void* unit = AddOffsetToPtr(alignedBuffer, unitIndex * alignedUnitLength, unitAlignment);
        *reinterpret_cast<s32*>(unit) = alignedUnitLength;

#if (defined(NW_DEBUG) || defined(NW_DEVELOP))
        Memset(AddOffsetToPtr(unit, sizeof(s32)), 0xcd, alignedUnitLength - sizeof(s32));
#endif

    }

    m_Buffer = alignedBuffer;
    m_BufferLength = alignedBufferLength;
    m_UnitLength = alignedUnitLength;
    m_FirstFreeUnit = alignedBuffer;
}

//---------------------------------------------------------------------------
void
UnitHeap::Finalize()
{
    NW_ASSERT(m_AllocatedUnitCount == 0);

    m_Buffer = NULL;
    m_BufferLength = 0;
}

//---------------------------------------------------------------------------
void*
UnitHeap::Alloc(size_t size, s32 alignment /*= MemoryTraits::DEFAULT_ALIGNMENT*/)
{
    (void)alignment;

    NW_ASSERT(alignment > 0);

    if(!IsInitialized())
    {
        NW_FATAL_ERROR("UnitHeap is not initialized.\n");
        return NULL;
    }

    if(m_FirstFreeUnit == NULL)
    {
        NW_WARNING(false, "out of memory.\n");
        return NULL;
    }

    // １ユニットあたりのサイズを超えた場合は、失敗させます。
    if(m_UnitLength < size)
    {
        NW_FATAL_ERROR("required size is too large.\n");
        return NULL;
    }

    void* result = m_FirstFreeUnit;
    m_FirstFreeUnit = GetNextFreeUnit(m_FirstFreeUnit);

    ++m_AllocatedUnitCount;

#if defined(FND_ENABLE_MEM_DEBUG)
    NW_LOG(
        "[spyfnd] UnitMemoryAllocator::Alloc() : AllocatedUnitCount=%d, AllocatedBuffer=0x%8x, FirstFreeUnit=0x%8x.\n",
        m_AllocatedUnitCount,
        result,
        m_FirstFreeUnit);
#endif

    return result;
}

//---------------------------------------------------------------------------
void
UnitHeap::Free(void* ptr)
{
    NW_ASSERT_NOT_NULL(ptr);

    if(!IsInitialized())
    {
        NW_FATAL_ERROR("UnitHeap is not initialized.\n");
        return;
    }

#if (defined(NW_DEBUG) || defined(NW_DEVELOP))
    Memset(ptr, 0xcd, m_UnitLength);
#endif

    s32 offset = 0;

    if(m_FirstFreeUnit != NULL)
    {
        offset = GetOffsetFromPtr(ptr, m_FirstFreeUnit);
        NW_ASSERT(offset != 0);
    }

    *reinterpret_cast<s32*>(ptr) = offset;
    m_FirstFreeUnit = ptr;

    --m_AllocatedUnitCount;

#if defined(FND_ENABLE_MEM_DEBUG)
    NW_LOG(
        "[spyfnd] UnitMemoryAllocator::Free() : AllocatedUnitCount=%d, FreedBuffer=0x%8x, FirstFreeUnit=0x%8x.\n",
        m_AllocatedUnitCount,
        ptr,
        m_FirstFreeUnit);
#endif
}

//---------------------------------------------------------------------------
void*
UnitHeap::GetNextFreeUnit(void* unit) const
{
    NW_ASSERT_NOT_NULL(unit);

    s32 offset = *reinterpret_cast<s32*>(unit);

    if(offset == 0)
    {
        return NULL;
    }

    return AddOffsetToPtr(unit, offset);
}

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
#else

//---------------------------------------------------------------------------
bool
ExpandedHeap::Initialize(void* buffer, size_t length)
{
    NW_ASSERT_NOT_NULL(buffer);
    NW_ASSERT(length > 0);

#if defined( NW_PLATFORM_CAFE )
    m_HeapHandle = MEMCreateExpHeapEx(buffer, length, 0);
#elif defined( NW_PLATFORM_CTR )
    m_Heap.Initialize(reinterpret_cast<uptr>(buffer), length);
    m_HeapHandle = &m_Heap;
#endif
    return IsInitialized();
}

//---------------------------------------------------------------------------
void
ExpandedHeap::Finalize()
{
#if defined( NW_PLATFORM_CTR )
    m_Heap.Finalize();
    m_HeapHandle = NULL;
#else
    if(m_HeapHandle != NULL)
    {
        MEMDestroyExpHeap(m_HeapHandle);
        m_HeapHandle = NULL;
    }
#endif
}

//---------------------------------------------------------------------------
void*
ExpandedHeap::Alloc(size_t size, s32 alignment/*= MemoryTraits::DEFAULT_ALIGNMENT*/)
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("ExpandedHeap is not initialized.\n");
        return NULL;
    }

    if(size == 0)
    {
        return NULL;
    }

#if defined( NW_PLATFORM_CTR )
    return m_Heap.Allocate(size, alignment);
#else
    return MEMAllocFromExpHeapEx(m_HeapHandle, size, alignment);
#endif
}

//---------------------------------------------------------------------------
void
ExpandedHeap::Free(void* ptr)
{
    NW_ASSERT_NOT_NULL(ptr);

    if(!IsInitialized())
    {
        NW_FATAL_ERROR("ExpandedHeap is not initialized.\n");
        return;
    }

#if defined( NW_PLATFORM_CTR )
    return m_Heap.Free(ptr);
#else
    MEMFreeToExpHeap(m_HeapHandle, ptr);
#endif
}

#endif

} // namespace nw::snd::spy::internal::fnd
} // namespace nw::snd::spy::internal
} // namespace nw::snd::spy
} // namespace nw::snd
} // namespace nw
