﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/util/sndutil_MidiManager.h>
#include <nw/snd/snd_SoundArchivePlayer.h>
#include <nw/snd/snd_Bank.h>
#include <nw/snd/snd_SoundDataManager.h>
#include <nw/snd/snd_Util.h>

#include <nw/config.h>
#if defined( NW_PLATFORM_CAFE )
  #include <cafe/midi.h>
#else
  // #include <cafe/midi.h> の代わりの定義
  // typedef void (*MIDICallback)(s32 chan, u8* buf, void* context, s32 ret);

  // #define MIDI_JSTAT_MASK           0x3a
  #define MIDI_JSTAT_FLAGS_SHIFT       4
  #define MIDI_JSTAT_FLAGS_MASK     0x30
  #define MIDI_JSTAT_SEND           0x08
  // #define MIDI_JSTAT_RECV           0x02

  #define MIDI_READY                   0
  #define MIDI_NOT_READY               1
  // #define MIDI_BUSY                    2

  void MIDIInit(void) {}
  s32  MIDIGetStatus(s32 /* chan */) { return MIDI_NOT_READY; }
  s32  MIDIRead(s32 /* chan */, u8* /* dst */, u8* /* status */) { return MIDI_NOT_READY; }
  // s32 MIDISend(s32 chan, u8* dst, u32 size, u8* status);
  // s32 MIDIGetStatusAsync(s32 chan, u8* status, MIDICallback callback);
  // s32 MIDIReadAsync(s32 chan, u8* dst, u8* status, void* context, MIDICallback callback);
  // s32 MIDISendAsync(s32 chan, u8* dst, u32 size, u8* status, void *context, MIDICallback callback);
#endif

namespace nw {
namespace snd {
namespace util {

#if defined(NW_PLATFORM_WIN32)
#pragma warning(push)
#pragma warning(disable:4355) // warning: used in base member initializer list
#endif
MidiManager::MidiManager()
    : m_RecieveThread(this)
    , m_InitializeThreadFlag( false )
{
    m_SequenceTrackAllocator.Create( m_SequenceTrack, sizeof( m_SequenceTrack ) );
}
#if defined(NW_PLATFORM_WIN32)
#pragma warning(pop) // disable:4355
#endif

void MidiManager::MidiCallback( u8 status, u8 data1, u8 data2, void* arg )
{
    MidiManager* pMidiManager = reinterpret_cast<MidiManager*>( arg );
    pMidiManager->m_SequencePlayer.SendMessage( status, data1, data2 );
}

nw::snd::internal::driver::Channel* MidiManager::NoteOn(
    nw::snd::internal::driver::SequenceSoundPlayer* pSequencePlayer,
    u8 bankIndex,
    const nw::snd::internal::driver::NoteOnInfo& noteOnInfo
)
{
    NW_UNUSED_VARIABLE( pSequencePlayer );

    if ( bankIndex >= nw::snd::SoundArchive::SEQ_BANK_MAX ) { return NULL; }

    nw::snd::internal::driver::Bank bank;
    nw::snd::internal::driver::Channel* channel = bank.NoteOn(
        pSequencePlayer->GetBankFileReader( bankIndex ),
        pSequencePlayer->GetWaveArchiveFileReader( bankIndex ),
        noteOnInfo
    );
    return channel;
}

void MidiManager::RegisterSoundThreadCallback()
{
    nw::snd::internal::driver::SoundThread::GetInstance().RegisterSoundFrameCallback( this );
}
void MidiManager::UnregisterSoundThreadCallback()
{
    nw::snd::internal::driver::SoundThread::GetInstance().UnregisterSoundFrameCallback( this );
}

size_t MidiManager::GetRequiredMemSize( size_t stackSize ) const
{
    return stackSize;
}

bool MidiManager::Initialize( s32 threadPriority, void* stackBuffer, size_t stackSize )
{
    if ( m_InitializeThreadFlag )
    {
        return true;
    }

    // m_CriticalSection.Initialize(); // Cafe 版では不要
    m_FinalizeFlag = false;

#if defined(NW_PLATFORM_CAFE)
#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    MIDIInit();
#endif
#elif defined(NW_PLATFORM_CTR)
    nn::midi::Initialize();
    {
        nn::midi::DeviceInfo di;
        nn::Result ret = nn::midi::GetDeviceInfo( &di );
        NN_TLOG_("[GetDeviceInfo] ... %d\n", ret.IsSuccess() );
        // NN_TASSERT_( ret.IsSuccess() );

        NN_TLOG_("  id       : %p\n", di.id);
        NN_TLOG_("  version  : %p\n", di.version);
        NN_TLOG_("  revision : %p\n", di.revision);
    }
    nn::Result ret = nn::midi::Open();
    NN_TLOG_("[MidiOpen] ... %d\n", ret.IsSuccess() );
    if ( ret.IsFailure() )
    {
        return false;
    }
#endif

#if defined(NW_PLATFORM_CTR)
    nn::Result result = m_RecieveThread.TryStartUsingAutoStack(
            RecieveThreadFunc,
            reinterpret_cast<uptr>( this ),
            STACK_SIZE,
            threadPriority );
    if ( result.IsFailure() ) { return false; }
#else
    {
        nw::ut::Thread::CreateArg arg;
        {
            arg.priority = threadPriority;
            arg.stackBase = stackBuffer;
            arg.stackSize = stackSize;
        #if defined(NW_PLATFORM_CAFE)
            arg.nameString = "nw:snd::MidiManager";
        #endif
        }
        bool result = m_RecieveThread.Create( arg );
        if ( result )
        {
            m_RecieveThread.Resume();
        }
        else
        {
            return false;
        }
    }
#endif

    for ( int i = 0; i < MIDI_PORT_MAX; i++ )
    {
        m_MidiPort[i].m_Parser.SetCallback( MidiCallback, this );
    }

    m_SequencePlayer.Initialize();
    {
        internal::driver::SequenceSoundPlayer::SetupArg arg;
        {
            arg.trackAllocator = &m_SequenceTrackAllocator;
            arg.allocTracks = 0xffff;
            arg.callback = this;
        }
        m_SequencePlayer.Setup(arg);
    }
    m_SequencePlayer.Start();

    RegisterSoundThreadCallback();

    m_InitializeThreadFlag = true;

    return true;
}

void MidiManager::Finalize()
{
    nw::ut::ScopedLock<nw::ut::CriticalSection> lock( m_CriticalSection );
    UnregisterSoundThreadCallback();

    if ( m_InitializeThreadFlag )
    {
        m_FinalizeFlag = true;
        m_RecieveThread.Join();
        m_RecieveThread.Destroy();

        m_SequencePlayer.Stop();
        m_InitializeThreadFlag = false;
    }

    // m_CriticalSection.Finalize();
}

#if defined(NW_PLATFORM_CTR)
void MidiManager::RecieveThreadFunc( uptr arg )
{
    MidiManager* pMidiManager = reinterpret_cast<MidiManager*>( arg );
    pMidiManager->RecieveThread();
}
#endif

void MidiManager::OnBeginSoundFrame()
{
    for ( int portIndex = 0; portIndex < MIDI_PORT_MAX; portIndex++ )
    {
        MidiPort& port = m_MidiPort[portIndex];

        // swap write <=> read
        {
            nw::ut::ScopedLock<nw::ut::CriticalSection> lock( m_CriticalSection );

            MidiBuffer* tmp = port.m_pWriteBuffer;
            port.m_pWriteBuffer = port.m_pReadBuffer;
            port.m_pReadBuffer = tmp;
            port.m_pWriteBuffer->size = 0;
        }

        port.m_Parser.Parse( port.m_pReadBuffer->buffer, port.m_pReadBuffer->size );
    }
}

#if defined(NW_PLATFORM_CTR)
void MidiManager::RecieveThread()
{
    while ( ! m_FinalizeFlag )
    {
        nn::os::Timer timer( false );

        for ( int portIndex = 0; portIndex < MIDI_PORT_MAX; portIndex++ )
        {
            int readSize = 0;
        #ifndef NW_SND_STATIC_BUFFER_ENABLE
            u8 buffer[ nn::midi::MIDI_RECV_BUFFER_MAX ];
        #endif

            bool isReadable = false;
            s32 readableLength = 0;
            nn::Result result = nn::midi::GetReadableLength( &readableLength );
            if ( result.IsSuccess() )
            {
                if ( readableLength > 0 )
                {
                    isReadable = true;
                }
            }
            else if ( result == nn::midi::ResultDeviceFifoFull() )
            {
                nn::midi::Reset();
                continue;
            }

            if ( isReadable == false )
            {
                timer.StartOneShot( nn::fnd::TimeSpan::FromMilliSeconds( 2 ) );
                timer.Wait();   // 2ms ごとにしか GetReadableLength は更新されない
                continue;
            }

            result = nn::midi::Read( // ほげ
                    buffer,
                    &readSize,
                    nn::midi::MIDI_RECV_BUFFER_MAX );

            if ( result.IsSuccess() )
            {
                s_MidiReadCount += readSize;

                MidiPort& port = m_MidiPort[portIndex];

                nw::ut::ScopedLock<nw::ut::CriticalSection> lock( m_CriticalSection );
                for ( int i = 0; i < readSize; ++i )
                {
                    port.m_pWriteBuffer->buffer[ port.m_pWriteBuffer->size ] = buffer[i];
                    port.m_pWriteBuffer->size++;
                    NW_ASSERT( port.m_pWriteBuffer->size < nn::midi::MIDI_RECV_BUFFER_MAX );
                }
                port.m_RecieveFlag = true;
            }
            else if ( result == nn::midi::ResultDeviceFifoFull() )
            {
                nn::midi::Reset();
                continue;   // 現状うしろに処理がないのであまり意味はないけど・・・
            }
        }
    }
}
#elif defined(NW_PLATFORM_CAFE)
void MidiManager::ThreadHandlerProc()
{
#if defined(NW_DEBUG) || defined(NW_DEVELOP)
    while ( ! m_FinalizeFlag )
    {
        bool midiActiveFlag = false;

        for ( int portIndex = 0; portIndex < MIDI_PORT_MAX; portIndex++ )
        {
            s32 result = MIDIGetStatus(portIndex);
            if ( result != MIDI_READY )
            {
                continue;
            }
            u8 status = 0;
            u8 buf[4] = {0};
            result = MIDIRead(portIndex, buf, &status);

            // NW_LOG("ret(%d) status(%d) %02x %02x %02x %02x\n", result, status, buf[0], buf[1], buf[2], buf[3]);
            if ( result != MIDI_READY )
            {
                continue;
            }

            midiActiveFlag = true;

            u8 count = status;
            count &= 0x70;
            count >>= MIDI_JSTAT_FLAGS_SHIFT;

            {
                MidiPort& port = m_MidiPort[portIndex];

                nw::ut::ScopedLock<nw::ut::CriticalSection> lock( m_CriticalSection );
                for ( int i = 1; i < count; i++ )
                {
                    port.m_pWriteBuffer->buffer[port.m_pWriteBuffer->size] = buf[i];
                    port.m_pWriteBuffer->size++;
                    NW_ASSERT(port.m_pWriteBuffer->size < MIDI_BUFFER_SIZE);
                }
                port.m_RecieveFlag = true;
            }
        }

        if ( ! midiActiveFlag )
        {
            OSSleepMicroseconds(MIDI_PORT_CHECK_INTERVAL_US);
        }
    }
#endif
}
#else
void MidiManager::ThreadHandlerProc()
{
}
#endif

void MidiManager::Prepare(
        SoundArchive::ItemId bankIds[],
        const SoundArchive& arc,
        const SoundDataManager& mgr)
{
    // for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    // {
    //     NW_LOG("[%s] bank(%d) => %08x\n", __FUNCTION__, i, bankIds[i]);
    // }

    const void* bankFiles[SoundArchive::SEQ_BANK_MAX] = {NULL};
    const void* warcFiles[SoundArchive::SEQ_BANK_MAX] = {NULL};
    bool isLoadIndividual[SoundArchive::SEQ_BANK_MAX] = {false};

    for ( int i = 0; i < SoundArchive::SEQ_BANK_MAX; i++ )
    {
        if ( bankIds[i] == SoundArchive::INVALID_ID )
        {
            continue;
        }
        SoundArchive::BankInfo bankInfo;
        if ( ! arc.ReadBankInfo( bankIds[i], &bankInfo ) )
        {
            continue;
        }

        bankFiles[i] = mgr.detail_GetFileAddress(bankInfo.fileId);
        if ( bankFiles[i] == NULL )
        {
            continue;
        }

        internal::LoadItemInfo loadWarc;
        internal::Util::WaveArchiveLoadStatus status =
            internal::Util::GetWaveArchiveOfBank(
                    loadWarc,
                    isLoadIndividual[i],
                    bankFiles[i],
                    arc,
                    mgr );
        if ( status == internal::Util::WARC_LOAD_NONEED
          || status == internal::Util::WARC_LOAD_NOT_YET
          || status == internal::Util::WARC_LOAD_ERROR )
        {
            continue;
        }

        warcFiles[i] = loadWarc.address;
    }
    Prepare( bankFiles, warcFiles, isLoadIndividual );
}


}}} // namespace nw::snd::util

