﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/util/sndutil_FileReader.h>
#include <nw/ut/ut_Inlines.h>

namespace nw {
namespace snd {
namespace util {

//------------------------------------------------------------------------------
FileReader::FileReader()
: m_FileSize(0)
, m_FilePosition(0)
, m_IsInitialized(false)
, m_IsOpen(false)
{}

//------------------------------------------------------------------------------
FileReader::~FileReader()
{
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
FileReader::Result FileReader::Initialize(FSClient* client)
{
    if ( m_IsInitialized )
    {
        return RESULT_NOT_INITIALIZED;
    }

    NW_ASSERT_NOT_NULL(client);
    m_pFsClient = client;
    FSInitCmdBlock(&m_FsCmdBlock);

    m_IsInitialized = true;
    return RESULT_SUCCESS;
}

#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)

FileReader::Result FileReader::Initialize()
{
    if ( m_IsInitialized )
    {
        return RESULT_NOT_INITIALIZED;
    }
    m_IsInitialized = true;
    return RESULT_SUCCESS;
}

#endif

//------------------------------------------------------------------------------
FileReader::Result FileReader::Finalize()
{
    if ( m_IsInitialized == false )
    {
        return RESULT_NOT_INITIALIZED;
    }

#if defined(NW_PLATFORM_CAFE)
    m_pFsClient = NULL;
#endif

    m_IsInitialized = false;
    return RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
FileReader::Result FileReader::Open(const char* path)
{
    if ( m_IsInitialized == false )
    {
        return RESULT_NOT_INITIALIZED;
    }
    NW_ASSERT_NOT_NULL(path);

#if defined(NW_PLATFORM_CAFE)
    FSStatus status = FSOpenFile(m_pFsClient, &m_FsCmdBlock, path, "r",
            &m_FsFileHandle, FS_RET_ALL_ERROR);
    if ( status != FS_STATUS_OK )
    {
        NW_WARNING(status == FS_STATUS_OK,
                "FSOpenFile(%s) failed. status(%d)\n", path, status);
        return RESULT_FAILED_OPEN_FILE;
    }

    status = FSGetStatFile(m_pFsClient, &m_FsCmdBlock, m_FsFileHandle,
            &m_FsStat, FS_RET_ALL_ERROR);
    if ( status != FS_STATUS_OK )
    {
        NW_WARNING(status == FS_STATUS_OK,
                "FSGetStatFile(%s) failed. status(%d)\n", path, status);
        return RESULT_FAILED_GET_FILESIZE;
    }

    m_FileSize = m_FsStat.size;
    m_IsOpen = true;
#elif defined(NW_USE_NINTENDO_SDK)

    nn::Result result;
    result = nn::fs::OpenFile(&m_FileHandle, path, nn::fs::OpenMode_Read);
    if (!result.IsSuccess())
    {
        return RESULT_FAILED_OPEN_FILE;
    }

    int64_t fileSize;
    result = nn::fs::GetFileSize(&fileSize, m_FileHandle);
    if (!result.IsSuccess())
    {
        return RESULT_FAILED_GET_FILESIZE;
    }

    m_FileSize = static_cast<u32>(fileSize);
#elif defined(NW_PLATFORM_WIN32)
    HANDLE handle = CreateFileA(
            path, GENERIC_READ, FILE_SHARE_READ, NULL,
            OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, NULL);
    if (handle == INVALID_HANDLE_VALUE)
    {
        return RESULT_FAILED_OPEN_FILE;
    }
    m_pFileHandle = reinterpret_cast<void*>(handle);

    DWORD result = GetFileSize(m_pFileHandle, NULL);
    if(result == -1)
    {
        return RESULT_FAILED_GET_FILESIZE;
    }
    m_FileSize = result;
#endif

    m_FilePosition = 0;
    m_IsOpen = true;
    return RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
FileReader::Result FileReader::Close()
{
    if ( m_IsInitialized == false )
    {
        return RESULT_NOT_INITIALIZED;
    }
    if ( m_IsOpen == false )
    {
        return RESULT_NOT_OPEN;
    }

#if defined(NW_PLATFORM_CAFE)
    FSStatus status = FSCloseFile(m_pFsClient, &m_FsCmdBlock,
            m_FsFileHandle, FS_RET_ALL_ERROR);
    if ( status != FS_STATUS_OK )
    {
        NW_WARNING(status == FS_STATUS_OK, "FSCloseFile failed. status(%d)\n", status);
    }
#elif defined(NW_USE_NINTENDO_SDK)
    nn::fs::CloseFile(m_FileHandle);
#elif defined(NW_PLATFORM_WIN32)
    CloseHandle(m_pFileHandle);
#endif

    m_IsOpen = false;
    return RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
FileReader::Result FileReader::Read(void* buf, u32 length, s32* readSize)
{
    if ( m_IsInitialized == false )
    {
        return RESULT_NOT_INITIALIZED;
    }
    if ( m_IsOpen == false )
    {
        return RESULT_NOT_OPEN;
    }
    NW_ASSERT_NOT_NULL(buf);

    s32 result = 0;

#if defined(NW_PLATFORM_CAFE)
    NW_ASSERT_ALIGN(buf, FS_IO_BUFFER_ALIGN);
    FSStatus status = FSReadFile(
            m_pFsClient, &m_FsCmdBlock, buf, 1, length,
            m_FsFileHandle, 0, FS_RET_ALL_ERROR);
    if ( status < FS_STATUS_OK )
    {
        // NW_WARNING(status >= FS_STATUS_OK, "FSReadFile failed. status(%d)\n", status);
        NW_LOG("FSReadFile failed. status(%d)\n", status);
        return RESULT_FAILED_READ_FILE;
    }
    result = status;
#elif defined(NW_USE_NINTENDO_SDK)
    size_t readFileLength = 0;
    nn::Result nnResult = nn::fs::ReadFile(&readFileLength, m_FileHandle, m_FilePosition, buf, length);

    if (!nnResult.IsSuccess())
    {
        return RESULT_FAILED_READ_FILE;
    }
    result = readFileLength;
#elif defined(NW_PLATFORM_WIN32)
    DWORD readLength = 0;
    if ( ! ReadFile(m_pFileHandle, buf, length, &readLength, NULL ) )
    {
        return RESULT_FAILED_READ_FILE;
    }
    result = readLength;
#endif

    m_FilePosition += result;
    if (readSize)
    {
        *readSize = result;
    }
    return RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
FileReader::Result FileReader::Seek(s32 offset, Origin origin)
{
    if ( m_IsInitialized == false )
    {
        return RESULT_NOT_INITIALIZED;
    }
    if ( m_IsOpen == false )
    {
        return RESULT_NOT_OPEN;
    }

    s32 position = 0;
    switch (origin)
    {
    case ORIGIN_BEGIN:
        position = offset;
        break;
    case ORIGIN_CURRENT:
        position = m_FilePosition + offset;
        break;
    case ORIGIN_END:
        position = m_FileSize - offset;
        break;
    default:
        return RESULT_INVALID_ARGUMENT;
    }

    position = ut::Clamp(position, 0, static_cast<s32>(m_FileSize));

#if defined(NW_PLATFORM_CAFE)
    FSStatus status = FSSetPosFile(
            m_pFsClient, &m_FsCmdBlock, m_FsFileHandle, position, FS_RET_ALL_ERROR);
    if ( status != FS_STATUS_OK )
    {
        NW_WARNING(status == FS_STATUS_OK,
                "FSSetPosFile failed. status(%d) pos(%d)\n", status, position);
        return RESULT_FAILED_SEEK_FILE;
    }
#elif defined(NW_USE_NINTENDO_SDK)
    // NintendoSDK では位置を変更するだけ
#elif defined(NW_PLATFORM_WIN32)
    u32 pos = SetFilePointer(m_pFileHandle, position, NULL, FILE_BEGIN);
    if (pos == INVALID_SET_FILE_POINTER)
    {
        return RESULT_FAILED_SEEK_FILE;
    }
#endif

    m_FilePosition = position;
    return RESULT_SUCCESS;
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//------------------------------------------------------------------------------

} // namespace util
} // namespace snd
} // namespace nw
