﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundEditSession.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/sndedit_Config.h>
#include <nw/snd/edit/sndedit_IErrorProvider.h>
#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/edit/sndedit_SoundArchiveEditor.h>
#include <nw/snd/edit/hio/sndedit_HioSyncChannel.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/edit/protocol/sndedit_SyncPacket.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemsPacket.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemInfoPacket.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {

//----------------------------------------------------------
SoundEditSession::SoundEditSession() :
// HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
m_Interim_IsSoundPlayer(false),
#endif
m_State(STATE_NOT_INITIALIZED),
m_Port0(0),
m_Port1(0),
m_SyncTimeout(0),
m_SoundArchiveEditor(NULL)
{
}

//----------------------------------------------------------
SoundEditSession::~SoundEditSession()
{
    // Finalize() 済みであることを確認します。
    NW_ASSERTMSG(!IsInitialized(), "SoundEditSession is not finalized.\n");
}

//----------------------------------------------------------
void
SoundEditSession::Finalize()
{
    Close();

    UnregisterSoundArchiveEditor(NULL);

    FinalizeFuncChannel();
    FinalizeSyncChannel();

    m_HioManager.Finalize();

    m_Connection.Finalize();

    m_State = STATE_NOT_INITIALIZED;
    m_SyncTimeout = 0;
}

//----------------------------------------------------------
Result
SoundEditSession::RegisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor)
{
    NW_ASSERTMSG(IsInitialized(), "SoundEditSession is not initialized.\n");
    NW_ASSERTMSG(m_SoundArchiveEditor == NULL, "SoundArchiveEditor is already regiestered.\n");

    if(soundArchiveEditor == NULL ||
        !soundArchiveEditor->IsInitialized())
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    m_SoundArchiveEditor = soundArchiveEditor;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundEditSession::UnregisterSoundArchiveEditor(SoundArchiveEditor* soundArchiveEditor)
{
    // TODO : 複数サウンドアーカイブに対応したら利用します。
    (void)soundArchiveEditor;

    NW_ASSERTMSG(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
        m_SoundArchiveEditor = NULL;
    }
}

//----------------------------------------------------------
void
SoundEditSession::Open()
{
    NW_ASSERTMSG(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(IsOpened())
    {
        return;
    }

    ClearBuffer();
    m_State = STATE_SYNC_REQUESTING;

    RequestSync();
    m_SyncStopWatch.Start();
}

//----------------------------------------------------------
void
SoundEditSession::Close()
{
    NW_ASSERTMSG(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(IsOpened())
    {
        Disconnect();
        ClearBuffer();
    }

    m_SyncStopWatch.Stop();

    m_State = STATE_STOPPED;
}

//----------------------------------------------------------
void
SoundEditSession::Update()
{
    NW_ASSERTMSG(IsInitialized(), "SoundEditSession is not initialized.\n");

    if(!IsOpened())
    {
        return;
    }

#if !defined(NW_SND_EDIT_USE_MCS)
    if(!m_SyncChannel.IsOpened())
    {
        if(!m_SyncChannel.Open(GetChannelInfo(internal::HIO_SNDEDIT_SYNC_CHANNEL)))
        {
            return;
        }
    }

    if(!m_FuncChannel.IsOpened())
    {
        if(!m_FuncChannel.Open(GetChannelInfo(internal::HIO_SNDEDIT_FUNC_CHANNEL)))
        {
            return;
        }
    }
#endif

    m_HioManager.Update();

    // SYNC タイムアウトを経過したら...
    // SYNC が未完了なら Close() します。
    // SYNC が完了していたら、再度 SYNC 要求します。
    if(m_SyncStopWatch.GetElapsedTime().ToMilliSeconds() >= m_SyncTimeout)
    {
        if(m_State != STATE_SYNC_COMPLETED)
        {
            if(m_Connection.IsOpened())
            {
                Disconnect();
            }
        }
        else
        {
            m_State = STATE_SYNC_UPDATING;
        }

        RequestSync();
    }

    m_Connection.Update();
}

//----------------------------------------------------------
Result
SoundEditSession::InitializeSyncChannel(
    snd::internal::fnd::FrameHeap& allocator,
    u32 recvStreamBufferSize,
    u32 recvPacketBufferSize)
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);
#if !defined(NW_SND_EDIT_USE_MCS)
    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());
#endif

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

#if defined(NW_SND_EDIT_USE_MCS)
    internal::HioResult result = m_SyncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize);
#else
    internal::HioResult result = m_SyncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );
#endif

    if(result.IsFailed())
    {
        return Result(ResultType(result));
    }

    if(!m_SyncChannel.Open(GetChannelInfo(internal::HIO_SNDEDIT_SYNC_CHANNEL)))
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    InitializeSyncHandlers();

    m_HioManager.RegisterChannel(m_SyncChannel);

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeSyncChannel()
{
    m_HioManager.UnregisterChannel(m_SyncChannel);

    FinalizeSyncHandlers();

    m_SyncChannel.Close();
    m_SyncChannel.Finalize();
}

//----------------------------------------------------------
void
SoundEditSession::InitializeSyncHandlers()
{
    NW_ASSERT(m_SyncChannel.IsInitialized());

    m_SyncReplyHandler.Initialize(*this);
    m_SyncChannel.RegisterMessageHandler(m_SyncReplyHandler);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeSyncHandlers()
{
    if(m_SyncChannel.IsInitialized())
    {
        m_SyncChannel.UnregisterMessageHandler(m_SyncReplyHandler);
        m_SyncReplyHandler.Finalize();
    }
}

//----------------------------------------------------------
Result
SoundEditSession::InitializeFuncChannel(
    snd::internal::fnd::FrameHeap& allocator,
    u32 recvStreamBufferSize,
    u32 recvPacketBufferSize)
{
    void* recvStreamBuffer = allocator.Alloc(recvStreamBufferSize);
    void* recvPacketBuffer = allocator.Alloc(recvPacketBufferSize);
#if !defined(NW_SND_EDIT_USE_MCS)
    void* workBuffer = allocator.Alloc(GetRequiredWorkBufferSize());
#endif

    if(recvStreamBuffer == NULL || recvPacketBuffer == NULL)
    {
        return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
    }

#if defined(NW_SND_EDIT_USE_MCS)
    internal::HioResult result = m_FuncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize);
#else
    internal::HioResult result = m_FuncChannel.Initialize(
        recvStreamBuffer,
        recvStreamBufferSize,
        recvPacketBuffer,
        recvPacketBufferSize,
        workBuffer,
        GetRequiredWorkBufferSize()
    );
#endif

    if(result.IsFailed())
    {
        return Result(ResultType(result));
    }

    if(!m_FuncChannel.Open(GetChannelInfo(internal::HIO_SNDEDIT_FUNC_CHANNEL)))
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    InitializeFuncHandlers();

    m_HioManager.RegisterChannel(m_FuncChannel);

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeFuncChannel()
{
    m_HioManager.UnregisterChannel(m_FuncChannel);

    FinalizeFuncHandlers();

    m_FuncChannel.Close();
    m_FuncChannel.Finalize();
}

//----------------------------------------------------------
void
SoundEditSession::InitializeFuncHandlers()
{
    NW_ASSERT(m_FuncChannel.IsInitialized());
}

//----------------------------------------------------------
void
SoundEditSession::FinalizeFuncHandlers()
{
    if(m_FuncChannel.IsInitialized())
    {
    }
}

//----------------------------------------------------------
u32
SoundEditSession::GetSyncChannelRecvPacketBufferSize(u32 maxItemName) const
{
    u32 result = 0;

    // TODO : SYNC チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : SYNC チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = ut::Max(result, internal::SyncPacket::GetRequiredSize());
    result = ut::Max(result, internal::SyncReplyPacket::GetRequiredSize());
    result = ut::Max(result, internal::QueryItemsPacket::GetRequiredSize());
    result = ut::Max(result, internal::QueryItemsReplyPacket::GetRequiredSize(maxItemName));

    return ut::RoundUp(result, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
u32
SoundEditSession::GetFuncChannelRecvPacketBufferSize(u32 maxItemName, u32 maxItemInfoSize) const
{
    u32 result = 0;

    // TODO : FUNC チャンネルで扱うパケットの種類が増えたら、ここに追加します。
    // TODO : FUNC チャンネルで扱うパケットのサイズが変わったら、ここを編集します。
    result = ut::Max(result, internal::QueryItemInfoPacket::GetRequiredSize(maxItemName));
    result = ut::Max(result, internal::QueryItemInfoReplyPacket::GetRequiredSize(maxItemInfoSize));

    return ut::RoundUp(result, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT);
}

//----------------------------------------------------------
Result
SoundEditSession::Connect()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditSession is not initialized.\n");
        Disconnect();
        ClearBuffer();
        return Result(SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    NW_ASSERT_NOT_NULL(m_SoundArchiveEditor);

    if(IsConnected() && m_SoundArchiveEditor->IsStarted())
    {
        m_State = STATE_SYNC_COMPLETED;

#if defined(NW_ENABLE_COM_DEBUG)
        NW_LOG(
            "[sndedit] SYNC completed : time=%d.\n",
            m_SyncStopWatch.GetElapsedTime().ToMilliSeconds());
#endif

        return Result(SNDEDIT_RESULT_TRUE);
    }

    // もし不正な状態ならば、それを正すために Disconect() します。
    Disconnect();

    Result result = m_Connection.Open(*m_SoundArchiveEditor);

    if(!result.IsTrue())
    {
        Disconnect();
        ClearBuffer();
        return result;
    }

    m_State = STATE_SYNC_COMPLETED;

#if defined(NW_ENABLE_COM_DEBUG)
    if(IsConnected())
    {
        NW_LOG("[sndedit] connected.\n");
    }
#endif

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundEditSession::Disconnect()
{
    if(!IsInitialized())
    {
        NW_FATAL_ERROR("SoundEditSession is not initialized.\n");
        return;
    }

    // TODO : ★マルチサウンドアーカイブ対応 : すべての SoundArchiveEditor を停止します。
    if(m_SoundArchiveEditor != NULL)
    {
        m_SoundArchiveEditor->detail_Stop();
    }

#if defined(NW_ENABLE_COM_DEBUG)
    if(IsConnected())
    {
        NW_LOG("[sndedit] disconnected.\n");
    }
#endif

    m_Connection.Close();

    m_State = STATE_SYNC_REQUESTING;
}

//----------------------------------------------------------
void
SoundEditSession::ClearBuffer()
{
    m_SyncChannel.ClearBuffer();
    m_FuncChannel.ClearBuffer();
}

//----------------------------------------------------------
bool
SoundEditSession::RequestSync()
{
    if(!IsInitialized())
    {
        return false;
    }

    // HACK : ★SoundMaker がゲームアプリと SoundPlayer の区別がつくようになるまでの暫定コード
#if 1
    internal::SyncPacket packet(m_Interim_IsSoundPlayer);
#else
    internal::SyncPacket packet;
#endif

    packet.GetBody().SetMemoryUsage(m_SoundArchiveEditor->GetMemoryUsage());
    packet.GetBody().SetMemoryMax(m_SoundArchiveEditor->GetMemoryMax());
    packet.GetBody().SetEditItemCount(m_SoundArchiveEditor->GetEditItemCount());
    packet.GetBody().SetMaxEditableItemCount(m_SoundArchiveEditor->GetMaxEditableItemCount());
    packet.GetBody().SetEditFileCount(m_SoundArchiveEditor->GetEditFileCount());
    packet.GetBody().SetMaxEditableFileCount(m_SoundArchiveEditor->GetMaxEditableFileCount());
    packet.GetBody().SetIsOutOfMemory(m_SoundArchiveEditor->IsOutOfMemory());
    packet.GetBody().SetIsItemsOverflow(m_SoundArchiveEditor->IsItemInfosOverflow());
    packet.GetBody().SetIsFilesOverflow(m_SoundArchiveEditor->IsFilesOverflow());

    m_SyncStopWatch.Reset();
    return m_SyncChannel.Send(packet).IsSucceeded();
}

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
