﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundArchiveParameterEditor.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/snd_Util.h>
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/edit/sndedit_SoundArchiveEditController.h>
#include <nw/snd/edit/res/sndedit_ResourceManager.h>

namespace nw {
namespace snd {
namespace edit {
namespace internal {

//----------------------------------------------------------
SoundArchiveParameterEditor::SoundArchiveParameterEditor() :
m_SoundArchiveEditController(NULL),
m_ResourceManager(NULL),
m_SoundArchive(NULL)
{
    m_SoundArchiveHook.Initialize(*this);
}

//----------------------------------------------------------
Result
SoundArchiveParameterEditor::Initialize(
    SoundArchiveEditController* editController,
    ResourceManager* resourceManager,
    SoundArchive* soundArchive)
{
    if(editController == NULL ||
        soundArchive == NULL)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

    m_SoundArchiveEditController = editController;
    m_ResourceManager = resourceManager;
    m_SoundArchive = soundArchive;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Finalize()
{
    m_SoundArchiveEditController = NULL;
    m_ResourceManager = NULL;
    m_SoundArchive = NULL;
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Start()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveParameterEditor is not initialized.");
    m_SoundArchive->detail_SetParametersHook(&m_SoundArchiveHook);
}

//----------------------------------------------------------
void
SoundArchiveParameterEditor::Stop()
{
    NW_ASSERTMSG(IsInitialized(), "SoundArchiveParameterEditor is not initialized.");
    m_SoundArchive->detail_SetParametersHook(NULL);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::IsTargetItemImpl(const char* itemLabel)
{
    NW_ASSERT_NOT_NULL(itemLabel);
    NW_ASSERT(*itemLabel != '\0');

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    return m_Owner->m_SoundArchiveEditController->GetItemCacheState(itemLabel) == CACHE_STATE_CACHED;
}

//----------------------------------------------------------
const char*
SoundArchiveParameterEditor::Hook::GetItemLabelImpl(SoundArchive::ItemId id) const
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    EditItemInfo editItemInfo;
    if(!m_Owner->m_ResourceManager->TryGetEditItemInfoFromID(id, &editItemInfo))
    {
        return NULL;
    }

    return editItemInfo.name;
}

//----------------------------------------------------------
SoundArchive::ItemId
SoundArchiveParameterEditor::Hook::GetItemIdImpl(const char* itemLabel) const
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    EditItemInfo editItemInfo;
    if(!m_Owner->m_ResourceManager->TryGetEditItemInfo(itemLabel, &editItemInfo))
    {
        return SoundArchive::INVALID_ID;
    }

    return editItemInfo.id;
}

//----------------------------------------------------------
SoundArchive::SoundType
SoundArchiveParameterEditor::Hook::GetSoundTypeImpl(const char* itemLabel)
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveFileEditor is not initialized.");

    ResDataType dataType = RES_DATA_TYPE_UNKNOWN;
    m_Owner->m_SoundArchiveEditController->GetSoundInfo(itemLabel, NULL, &dataType);

    return ResDataTypeToSoundType(dataType);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::SoundInfo* info) const
{
    return ReadSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundInfoImpl(
    const char* soundName,
    SoundArchive::SoundInfo* info) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = RES_DATA_TYPE_UNKNOWN;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->soundBasicParam.ApplyTo(*info, *m_Owner->m_SoundArchive);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSound3DInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::Sound3DInfo* info) const
{
    return ReadSound3DInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSound3DInfoImpl(
    const char* soundName,
    SoundArchive::Sound3DInfo* info) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = RES_DATA_TYPE_UNKNOWN;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->sound3DParam.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSequenceSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::SequenceSoundInfo* info) const
{
    return ReadSequenceSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSequenceSoundInfoImpl(
    const char* soundName,
    SoundArchive::SequenceSoundInfo* info) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    // キャッシュがあれば、それを適用します。
    // キャッシュがなければ、HIO経由でサウンド情報を取得します。
    const ResSequenceSoundInfo* editInfo = reinterpret_cast<const ResSequenceSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsSequenceSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->sequenceSoundParam.ApplyTo(*info, *m_Owner->m_SoundArchive);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::StreamSoundInfo* info ) const
{
    return ReadStreamSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfoImpl(
    const char* soundName,
    SoundArchive::StreamSoundInfo* info ) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResStreamSoundInfo* editInfo = reinterpret_cast<const ResStreamSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsStreamSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->streamSoundParam.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfo2Impl(
    SoundArchive::ItemId soundId,
    SoundArchive::StreamSoundInfo2* info ) const
{
    return ReadStreamSoundInfo2Impl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadStreamSoundInfo2Impl(
    const char* soundName,
    SoundArchive::StreamSoundInfo2* info ) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResStreamSoundInfo* editInfo = reinterpret_cast<const ResStreamSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsStreamSoundInfoDataType)
        );

    if(editInfo == NULL)
    {
        return false;
    }

    editInfo->streamSoundParam2.ApplyTo(*info);

    return true;
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadWaveSoundInfoImpl(
    SoundArchive::ItemId soundId,
    SoundArchive::WaveSoundInfo* info) const
{
    return ReadWaveSoundInfoImpl(GetItemName(soundId), info);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadWaveSoundInfoImpl(
    const char* soundName,
    SoundArchive::WaveSoundInfo* info) const
{
    NW_ASSERT_NOT_NULL(info);

    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    const ResWaveSoundInfo* editInfo = reinterpret_cast<const ResWaveSoundInfo*>(
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, &ResItemInfoUtility::IsWaveSoundInfoDataType)
        );

    if(editInfo != NULL)
    {
        editInfo->waveSoundParam.ApplyTo(*info);
        return true;
    }

    return false;
}
//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundUserParamImpl(
    SoundArchive::ItemId soundId,
    int index,
    u32& value) const
{
    return ReadSoundUserParamImpl(GetItemName(soundId), index, value);
}

//----------------------------------------------------------
bool
SoundArchiveParameterEditor::Hook::ReadSoundUserParamImpl(
    const char* soundName,
    int index,
    u32& value) const
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return false;
    }

    ResDataType itemInfoDataType = RES_DATA_TYPE_UNKNOWN;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return false;
    }

    value = editInfo->soundBasicParam.userParams[index];
    return true;
}

//----------------------------------------------------------
u32
    SoundArchiveParameterEditor::Hook::GetSoundUserParamImpl(
    SoundArchive::ItemId soundId,
    u32 userParam ) const
{
    return GetSoundUserParamImpl(GetItemName(soundId), userParam);
}

//----------------------------------------------------------
u32
SoundArchiveParameterEditor::Hook::GetSoundUserParamImpl(
    const char* soundName,
    u32 userParam ) const
{
    NW_ASSERT_NOT_NULL(m_Owner);
    NW_ASSERTMSG(m_Owner->IsInitialized(), "SoundArchiveParameterEditor is not initialized.");

    if(soundName == NULL || soundName[0] == '\0')
    {
        return userParam;
    }

    ResDataType itemInfoDataType = RES_DATA_TYPE_UNKNOWN;
    const ResSoundInfo* editInfo =
        m_Owner->m_SoundArchiveEditController->GetSoundInfo(soundName, NULL, &itemInfoDataType);

    if(editInfo == NULL)
    {
        return userParam;
    }

    return editInfo->soundBasicParam.userParams[0];
}

//----------------------------------------------------------
const char*
SoundArchiveParameterEditor::Hook::GetItemName(SoundArchive::ItemId itemID) const
{
    if(itemID == SoundArchive::INVALID_ID)
    {
        return "";
    }

    NW_ASSERT_NOT_NULL(m_Owner);
    return m_Owner->m_SoundArchive == NULL ? "" : m_Owner->m_SoundArchive->GetItemLabel(itemID);
}

bool
SoundArchiveParameterEditor::Hook::IsValidDataType(SoundArchive::SoundType soundType, ResDataType dataType) const
{
    switch(dataType)
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        return soundType == SoundArchive::SOUND_TYPE_STRM;

    case RES_DATA_TYPE_WAVE_SOUND:
        return soundType == SoundArchive::SOUND_TYPE_WAVE;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        return soundType == SoundArchive::SOUND_TYPE_SEQ;

    default:
        break;
    }

    return false;
}

SoundArchive::SoundType
SoundArchiveParameterEditor::Hook::ResDataTypeToSoundType(ResDataType dataType) const
{
    switch(dataType)
    {
    case RES_DATA_TYPE_STREAM_SOUND:
        return SoundArchive::SOUND_TYPE_STRM;

    case RES_DATA_TYPE_WAVE_SOUND:
        return SoundArchive::SOUND_TYPE_WAVE;

    case RES_DATA_TYPE_SEQUENCE_SOUND:
        return SoundArchive::SOUND_TYPE_SEQ;

    default:
        break;
    }

    return SoundArchive::SOUND_TYPE_INVALID;
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
