﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/handler/sndedit_QueryItemsReplyHandler.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/snd/fnd/basis/sndfnd_Inlines.h>
#include <nw/snd/edit/sndedit_SoundArchiveEditor.h>
#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/edit/hio/sndedit_HioProtocol.h>
#include <nw/snd/edit/hio/sndedit_HioPacketStream.h>
#include <nw/snd/edit/protocol/sndedit_QueryItemsPacket.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {
namespace internal {

namespace {

//----------------------------------------------------------
static void* GetTemporaryBuffer(QueryItemsReplyPacket& packet)
{
    if(packet.GetBody().items.GetItemCount() == 0)
    {
        return NULL;
    }

    // パケットの可変長領域をテンポラリバッファとして使いまわします。
    u32 offsetFromBody =
        sizeof(QueryItemsReplyPacket::Body) +
        packet.GetBody().items.GetItemCount() * sizeof(Offset);

    return snd::internal::fnd::AddOffsetToPtr(&packet.GetBody(), offsetFromBody);
}

}

//----------------------------------------------------------
QueryItemsReplyHandler::QueryItemsReplyHandler() :
m_Connection(NULL),
m_ItemSize(0),
m_ReadingItemIndex(0)
{
}

//----------------------------------------------------------
void
QueryItemsReplyHandler::Initialize(SoundEditConnection& connection, u32 maxItemName)
{
    NW_ASSERT(maxItemName > 0);

    m_Connection = &connection;
    m_ItemSize = QueryItemsReplyPacket::ItemInfo::GetRequiredSize(maxItemName);
    m_ReadingItemIndex = 0;
}

//----------------------------------------------------------
void
QueryItemsReplyHandler::Finalize()
{
    m_Connection = NULL;
    m_ItemSize = 0;
    m_ReadingItemIndex = 0;
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::OnInvoke(HioPacketStream& stream)
{
    if(m_Connection == NULL)
    {
        NW_FATAL_ERROR("QueryItemsReplyHandler is not initialized.\n");
        stream.SkipBody();
        return HioResult(HIO_RESULT_TARGET_NOT_INITIALIZED);
    }

    if(!m_Connection->IsInitialized() ||
        m_Connection->GetSoundArchiveEditor() == NULL)
    {
        stream.SkipBody();
        return HioResult(HIO_RESULT_TRUE);
    }

    HioResult result = ReadBodyFixedLengthArea(stream);

    if(result.IsFalse())
    {
        return result;
    }
    else if(!result.IsTrue())
    {
        stream.SkipBody();
        return result;
    }

    QueryItemsReplyPacket* targetPacket =
        reinterpret_cast<QueryItemsReplyPacket*>(stream.GetReadingPacket());
    NW_ASSERT_NOT_NULL(targetPacket);

    // actionForAllItems が指定されていたらすべてのアイテムについて処理します。
    switch(targetPacket->GetBody().actionForAllItems)
    {
    case QueryItemsReplyPacket::ACTION_UPDATE_ITEM:
        m_Connection->GetSoundArchiveEditor()->detail_UpdateAllItemCaches();
        stream.SkipBody();
        return HioResult(HIO_RESULT_TRUE);

    case QueryItemsReplyPacket::ACTION_REMOVE_ITEM:
        NW_ASSERT_NOT_NULL(m_Connection->GetSoundArchiveEditor());
        m_Connection->GetSoundArchiveEditor()->detail_RemoveAllItemCaches();
        stream.SkipBody();
        return HioResult(HIO_RESULT_TRUE);
    }

    ResOffsetTableData& itemsData =
        *reinterpret_cast<ResOffsetTableData*>(&targetPacket->GetBody().items);

    // テーブルのオフセットをすべて読み込みます。
    result = ReadItemsOffsets(itemsData, stream);

    if(!result.IsTrue())
    {
        stream.SkipBody();
        return result;
    }

    // アイテム数が多くても使用メモリを抑えるために、
    // ResOffsetTable::itemOffsets[] 以降の可変長メモリ領域を
    // 逐次読み込みのテンポラリ領域として使いまわします。
    QueryItemsReplyPacket::ItemInfo* item = reinterpret_cast<QueryItemsReplyPacket::ItemInfo*>(
        GetTemporaryBuffer(*targetPacket)
        );

    for(u32 index = m_ReadingItemIndex; index < itemsData.itemCount; ++index)
    {
        u32 itemSize = ReadItemLength(stream, itemsData, index);

        if(stream.GetReadableBytes() < itemSize)
        {
            return HioResult(HIO_RESULT_FALSE);
        }

        // アイテム名が長すぎる場合は、スキップします。
        if(itemSize > m_ItemSize)
        {
            NW_WARNING(false, "[nw::snd::edit::internal::QueryItemsReplyHandler] itemName is too long.\n");
            stream.SkipBodyPart(itemSize - sizeof(snd::internal::fnd::BinU32));
            continue;
        }

        result = stream.TryReadBodyPart(item, itemSize);

        if(!result.IsTrue())
        {
            stream.SkipBody();
            return result;
        }

        ++m_ReadingItemIndex;

        // 終端文字を設定します（保護処理）
        *snd::internal::fnd::AddOffsetToPtr<u8*>(item, itemSize) = '\0';

#if defined(NW_ENABLE_COM_DEBUG)
        NW_LOG(
            "[sndedit] QueryItemsReply : itemName[%d]=%s, action=%s, file=%s\n",
            index,
            item->name.GetName(),
            item->action & QueryItemsReplyPacket::ACTION_UPDATE_ITEM ? "UPDATE" : "REMOVE");
#endif

        NW_ASSERT_NOT_NULL(m_Connection->GetSoundArchiveEditor());

        switch(item->action)
        {
        case QueryItemsReplyPacket::ACTION_UPDATE_ITEM:
            // アイテムキャッシュを更新します。
            m_Connection->GetSoundArchiveEditor()->detail_UpdateItemCache(item->name.GetName());
            break;

        case QueryItemsReplyPacket::ACTION_REMOVE_ITEM:
            // アイテムキャッシュを削除します。
            // 即時にメモリを解放するために、無効化ではなく削除します。
            m_Connection->GetSoundArchiveEditor()->detail_RemoveItemCache(item->name.GetName());
            break;
        }
    }

    return HioResult(HIO_RESULT_TRUE);
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::ReadBodyFixedLengthArea(HioPacketStream& stream)
{
    // 固定長領域を読み込み済の場合は、処理しません。
    if(stream.GetReadBodySize() > sizeof(QueryItemsReplyPacket::Body))
    {
        return HioResult(HIO_RESULT_TRUE);
    }

    // 固定長領域が中途半端に読み込まれることはありません。
    if(stream.GetReadBodySize() > 0)
    {
        NW_FATAL_ERROR("QueryItemsReplyHandler has invalid status.\n");
        return HioResult(HIO_RESULT_FAILED);
    }

    // 固定長領域を一括で読み込めない場合は、処理しません。
    if(stream.GetReadableBytes() < sizeof(QueryItemsReplyPacket::Body))
    {
        return HioResult(HIO_RESULT_FALSE);
    }

    m_ReadingItemIndex = 0;

    HioResult result = stream.ReadBodyPart(
        stream.GetReadingPacket()->GetBody(),
        sizeof(QueryItemsReplyPacket::Body));

#if defined(NW_ENABLE_COM_DEBUG)
    if(result.IsTrue())
    {
        QueryItemsReplyPacket::Body* body =
            reinterpret_cast<QueryItemsReplyPacket::Body*>(stream.GetReadingPacket()->GetBody());

        NW_LOG(
            "[sndedit] QueryItemsReply : isAllItems=%d, itemNamesCount=%d\n",
            body->actionForAllItems ? 1 : 0,
            body->items.GetItemCount());
    }
#endif

    return result;
}

//----------------------------------------------------------
HioResult
QueryItemsReplyHandler::ReadItemsOffsets(
    ResOffsetTableData& items,
    HioPacketStream& stream)
{
    u32 itemsOffsetsSize = items.itemCount * sizeof(Offset);

    // 読み込み済の場合は、処理しません。
    if(stream.GetReadBodySize() >= sizeof(QueryItemsReplyPacket::Body) + itemsOffsetsSize)
    {
        return HioResult(HIO_RESULT_TRUE);
    }

    if(items.itemCount == 0)
    {
        return HioResult(HIO_RESULT_TRUE);
    }

    return stream.ReadBodyPart(items.itemOffsets, itemsOffsetsSize);
}

//----------------------------------------------------------
u32
QueryItemsReplyHandler::ReadItemLength(
    HioPacketStream& stream,
    ResOffsetTableData& itemsData,
    u32 index)
{
    if(itemsData.itemCount <= index)
    {
        NW_FATAL_ERROR("ItemNameTable index.\n");
        return 0;
    }

    if(index + 1 < itemsData.itemCount)
    {
        // テーブルアイテムオフセットの差分
        NW_ASSERT(itemsData.itemOffsets[index + 1] > itemsData.itemOffsets[index]);
        return ut::GetOffsetFromPtr(
            itemsData.itemOffsets[index].to_ptr(),
            itemsData.itemOffsets[index + 1].to_ptr());
    }
    else
    {
        // 最後のアイテムは、残りパケットボディサイズ
        NW_ASSERT(stream.GetReadingPacket()->GetHeader().bodySize > stream.GetReadBodySize());
        return stream.GetReadingPacket()->GetHeader().bodySize - stream.GetReadBodySize();
    }
}

} // namespace nw::snd::edit::internal
} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
