﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/edit/sndedit_SoundEditSession.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nn/hio.h>

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/fnd/basis/sndfnd_Memory.h>
#include <nw/snd/edit/sndedit_Config.h>
#include <nw/snd/edit/sndedit_SoundEditConnection.h>
#include <nw/snd/edit/hio/sndedit_HioSyncChannel.h>
#include <nw/snd/edit/hio/sndedit_HioAsyncChannel.h>
#include <nw/snd/edit/res/sndedit_ResItemInfo.h>

#if !defined(NW_RELEASE)
//#define NW_ENABLE_COM_DEBUG
#endif

namespace nw {
namespace snd {
namespace edit {

//----------------------------------------------------------
Result
SoundEditSession::Initialize(
     void* buffer,
     u32 bufferLength,
     const Configs& configs)
{
    bool isRunning = false;
    nn::Result ret = nn::hio::GetHioDaemonStatus( &isRunning );
    NW_ASSERT(ret.IsSuccess());
    if (!isRunning)
    {
        return Result(SNDEDIT_RESULT_HIOERROR);
    }

    if(buffer == NULL ||
        bufferLength == 0 ||
        configs.channelStreamBufferSize == 0 ||
        configs.syncTimeout == 0 ||
        configs.cacheSyncInterval == 0 ||
        configs.sendTimeout == 0 ||
        configs.maxItemName == 0)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }

//----------------------------------------------------------
#if !defined(NW_RELEASE)
    // バッファサイズをチェックします。
    if(bufferLength < GetRequiredMemorySize(configs))
    {
        NW_FATAL_ERROR("bufferLength too small.\n");
        return Result(SNDEDIT_RESULT_FAILED);
    }
#endif // !defined(NW_RELEASE)
//----------------------------------------------------------

    if(IsInitialized())
    {
        return Result(SNDEDIT_RESULT_FALSE);
    }

    m_Port0 = configs.port0;
    m_Port1 = configs.port1;

    if(m_SyncChannel.IsInitialized() ||
        m_FuncChannel.IsInitialized() ||
        m_SyncChannel.IsOpened() ||
        m_FuncChannel.IsOpened())
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    if(!m_HioManager.Initialize())
    {
        return Result(SNDEDIT_RESULT_FAILED);
    }

    snd::internal::fnd::FrameHeap allocator;
    allocator.Initialize(buffer, bufferLength);

    Result result = InitializeSyncChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetSyncChannelRecvPacketBufferSize(configs.maxItemName));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    result = InitializeFuncChannel(
        allocator,
        configs.channelStreamBufferSize,
        GetFuncChannelRecvPacketBufferSize(
            configs.maxItemName,
            internal::ResItemInfoUtility::GetMaxItemInfoSize(configs.maxItemName, internal::Limits::MaxFilePath)));

    if(result.IsFailed())
    {
        Finalize();
        return result;
    }

    {
        u32 bufferForConnectionLength = m_Connection.GetRequiredMemorySize(configs.maxItemName);
        void* bufferForConnection = allocator.Alloc(bufferForConnectionLength);

        if(bufferForConnection == NULL)
        {
            Finalize();
            return Result(SNDEDIT_RESULT_OUT_OF_MEMORY);
        }

        internal::SoundEditConnection::InitializeArgs initializeArgs;
        initializeArgs.buffer = bufferForConnection;
        initializeArgs.bufferLength = bufferForConnectionLength;
        initializeArgs.cacheSyncInterval = configs.cacheSyncInterval;
        initializeArgs.sendTimeout = configs.sendTimeout;
        initializeArgs.hioManager = &m_HioManager;
        initializeArgs.syncChannel = &m_SyncChannel;
        initializeArgs.funcChannel = &m_FuncChannel;
        initializeArgs.maxItemName = configs.maxItemName;

        result = m_Connection.Initialize(initializeArgs);

        if(result.IsFailed())
        {
            Finalize();
            return result;
        }
    }

    m_SyncTimeout = configs.syncTimeout;
    m_State = STATE_STOPPED;

    return Result(SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
u32
SoundEditSession::GetRequiredMemorySize(const Configs& configs) const
{
    u32 result = m_Connection.GetRequiredMemorySize(configs.maxItemName);

    u32 streamBufferSize = configs.channelStreamBufferSize;

    // SYNC, FUNC チャンネルの受信用ストリームバッファサイズ
    result += ut::RoundUp(streamBufferSize, snd::internal::fnd::MemoryTraits::DEFAULT_ALIGNMENT) * 2;

    // SYNC, FUNC チャンネルの受信用パケットバッファサイズ
    result += GetSyncChannelRecvPacketBufferSize(configs.maxItemName);
    result += GetFuncChannelRecvPacketBufferSize(
        configs.maxItemName,
        internal::ResItemInfoUtility::GetMaxItemInfoSize(configs.maxItemName, internal::Limits::MaxFilePath));

    result += GetRequiredWorkBufferSize() * 2;

    return result;
}

//----------------------------------------------------------
u32
SoundEditSession::GetRequiredWorkBufferSize() const
{
    return nn::hio::WORKMEMORY_SIZE + internal::HioChannel::DEFAULT_WORK_BUFFER_LENGTH;
}

//----------------------------------------------------------
internal::HioStream::ChannelType
SoundEditSession::GetChannelInfo(internal::HioChannelType channel) const
{
    switch(channel)
    {
        case internal::HIO_SNDEDIT_SYNC_CHANNEL:
            return static_cast<internal::HioStream::ChannelType>(m_Port0);
        case internal::HIO_SNDEDIT_FUNC_CHANNEL:
            return static_cast<internal::HioStream::ChannelType>(m_Port1);
        default:
            NW_ASSERT(false);
            return 0;
    }
}

} // namespace nw::snd::edit
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
