﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd.h>
#include <nw/dw.h>
#include <nw/dw/system/dw_InputSettings.h>
#include <nw/dev.h>
#include <nw/snd/dw/snddw_SoundWindowSystem.h>
#include <nw/snd/dw/snddw_SoundWindow.h>

#if defined( NW_PLATFORM_WIN32 ) || defined( NW_USE_NINTENDO_SDK )
using namespace nw::internal::winext;
#endif

namespace nw {
namespace snd {
namespace dw {

namespace internal {

    //! @details :private
    u8 s_PrimitiveRendererShaderBinary[] =
    {
        #include <nw/snd/dw/resource/PrimitiveRendererShaderBinary>
    };

    //! @details :private
    u8 s_FontBuildinShaderBinary[] =
    {
        #include<nw/snd/dw/resource/FontBuildinShaderBinary>
    };

} // namespace internal

//------------------------------------------------------------------------------
SoundWindowSystem::SoundWindowSystem()
: m_DWMgr()
, m_Font()
, m_pRenderer(NULL)
, m_UIRenderer()
, m_TextRenderer()
#ifdef NW_PLATFORM_CAFE
, m_ContextState(NULL)
#endif
, m_Pad(0)
, m_PadStatus()
, m_FontShaderBinarySize(0)
, m_FontBinarySize(0)
, m_FontShaderBinary(NULL)
, m_FontBinary(NULL)
{
}

//------------------------------------------------------------------------------
SoundWindowSystem::~SoundWindowSystem()
{
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::Initialize(const Param& param)
{
    NW_NULL_ASSERT(param.memory);
    m_Allocator.Initialize(param.memory, param.memorySize, MEM_HEAP_OPT_THREAD_SAFE);

    // 設定を変更不可にする
    Config::GetInstance().Seal();

#ifdef NW_PLATFORM_CAFE
    if ( nw::gfnd::Graphics::GetInstance() == NULL) {
        // GX2のコンテキストを作成し、Graphicsに設定
        m_ContextState = static_cast<GX2ContextState*>( m_Allocator.Alloc(sizeof(GX2ContextState), GX2_CONTEXT_STATE_ALIGNMENT) );
        GX2SetupContextState( m_ContextState );

        // グラフィックスインスタンス生成.
        nw::gfnd::Graphics::SetInstance( new( m_Allocator.Alloc( sizeof( nw::gfnd::Graphics ) ) ) nw::gfnd::Graphics() );
        nw::gfnd::Graphics::GetInstance()->Initialize();
        nw::gfnd::Graphics::GetInstance()->SetGX2ContextState(m_ContextState);
    }
#elif defined(NW_PLATFORM_WIN32) || defined(NW_USE_NINTENDO_SDK)
    NW_NULL_ASSERT(nw::gfnd::Graphics::GetInstance());
#endif

    InitializeUIRenderer(param);

    // ウィンドウマネージャの初期化
    m_DWMgr.Initialize(param.width, param.height, &m_Allocator);
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::Finalize()
{
    m_DWMgr.Finalize();

    FinalizeUIRenderer();

#ifdef NW_PLATFORM_CAFE
    if (m_ContextState != NULL) {
        nw::gfnd::Graphics* gfx = nw::gfnd::Graphics::GetInstance();
        nw::ut::SafeFree( gfx, &m_Allocator );

        nw::ut::SafeFree( m_ContextState, &m_Allocator );
    }
#endif

    m_Allocator.Finalize();
}

//------------------------------------------------------------------------------
u32
SoundWindowSystem::GetRequiredSize() const
{
    const u32 titleLenMax = Config::GetInstance().GetWindowTitleLengthMax();
    const u32 windowMax = Config::GetInstance().GetWindowMax();

    const u32 stringBufferSize = font::CharWriter::GetDispStringBufferSize( titleLenMax ) * windowMax;
    const u32 graphicsBufferSize = font::CharWriter::GetGraphicsBufferSize( titleLenMax ) * windowMax;

    return 152 * 1024 + stringBufferSize + graphicsBufferSize;
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::UpdateInputs(const nw::dev::RawPadStatus& padStatus)
{
    m_PadStatus.padStatuses[0] = padStatus;
    m_Pad.SetPadStatus(&m_PadStatus);
    m_Pad.Update();

    m_DWMgr.UpdateInputs(nw::internal::dw::Inputs(&m_Pad, NULL));
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::Update()
{
    m_DWMgr.Update(m_UIRenderer);
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::Draw()
{
#ifdef NW_PLATFORM_CAFE
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        GX2SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);
#endif
        nw::gfnd::GraphicsContext context;
        context.Apply();

        m_UIRenderer.ClearBuffer();

        m_UIRenderer.BeginDraw();
        m_DWMgr.Draw(m_UIRenderer);
        m_UIRenderer.EndDraw();

#ifdef NW_PLATFORM_CAFE
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
#endif
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::AddWindow(f32 posX, f32 posY, SoundWindow& window)
{
    SoundWindow::SetupArg arg;
    arg.fontBinary = m_FontBinary;
    arg.fontBinarySize = m_FontBinarySize;
    arg.fontShaderBinary = m_FontShaderBinary;
    arg.fontShaderBinarySize = m_FontShaderBinarySize;
    arg.projectionMatrix = &m_ProjectionMatrix;
    arg.viewMatrix = &m_ViewMatrix;
    window.Setup(arg);
    m_DWMgr.CreateWindow(&window, posX, posY);
}

//------------------------------------------------------------------------------
void
SoundWindowSystem::SetModifierKey(u32 modifier)
{
    nw::internal::dw::InputSettings::GetInstance().GetWindowManagerSetting().SetWindowControlModifier(modifier);
}

//------------------------------------------------------------------------------
u32
SoundWindowSystem::GetModifierKey() const
{
    return nw::internal::dw::InputSettings::GetInstance().GetWindowManagerSetting().GetWindowControlModifier();
}

//------------------------------------------------------------------------------
// プリミティブレンダラ
void
SoundWindowSystem::InitializePrimitiveRenderer(const Param& param)
{
    (void)param;
    // Win 版ではフレームワーク内で利用されます。
    m_pRenderer = nw::dev::PrimitiveRenderer::GetInstance();
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
#ifndef NW_PLATFORM_CAFE
        m_pRenderer->Initialize( &m_Allocator );
#else
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        {
            u32 size = sizeof(internal::s_PrimitiveRendererShaderBinary);
            m_pRenderer->InitializeFromBinary( &m_Allocator, internal::s_PrimitiveRendererShaderBinary, size );
        }
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();
}

void
SoundWindowSystem::FinalizePrimitiveRenderer()
{
    m_pRenderer->Finalize( &m_Allocator );
    m_pRenderer = NULL;
}

//------------------------------------------------------------------------------
// テキストレンダラ
void
SoundWindowSystem::InitializeTextRenderer(const Param& param)
{
    // フォントシェーダバイナリの読み込み
    {
        m_FontShaderBinarySize = sizeof(internal::s_PrimitiveRendererShaderBinary);
        m_FontShaderBinary = internal::s_FontBuildinShaderBinary;

        NW_NULL_ASSERT(m_FontShaderBinary);
        NW_ASSERTMSG(m_FontShaderBinarySize > 0, "[%d]\n", m_FontShaderBinarySize);
    }

    // フォントの読み込み
    {
        m_FontBinarySize = param.fontBinarySize;
        m_FontBinary = param.fontBinary;

        nw::gfnd::Graphics::GetInstance()->LockDrawContext();
        bool fontInitialized = m_Font.SetResource(m_FontBinary);
        NW_ASSERT(fontInitialized);
        nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

        NW_NULL_ASSERT(m_FontBinary);
        NW_ASSERTMSG(m_FontBinarySize > 0, "[%d]\n", m_FontBinarySize);
    }

    m_TextRenderer.Initialize(m_Allocator, m_FontBinary, m_FontBinarySize, m_FontShaderBinary, m_FontShaderBinarySize);
}

void
SoundWindowSystem::FinalizeTextRenderer()
{
    m_TextRenderer.Finalize();
}

//------------------------------------------------------------------------------
// UIレンダラ
void
SoundWindowSystem::InitializeUIRenderer(const Param& param)
{
    InitializePrimitiveRenderer(param);
    InitializeTextRenderer(param);

    m_UIRenderer.SetTextRenderer(&m_TextRenderer);
    m_UIRenderer.SetPrimitiveRenderer(m_pRenderer);

    m_ProjectionMatrix.SetOrtho(0.0f, param.width, param.height, 0.0f, 0.0f, 1.0f);
    m_ViewMatrix.SetIdentity();

    m_UIRenderer.SetProjectionMatrix(m_ProjectionMatrix);
    m_UIRenderer.SetViewMatrix(m_ViewMatrix);
}

void
SoundWindowSystem::FinalizeUIRenderer()
{
    FinalizeTextRenderer();
    FinalizePrimitiveRenderer();
}

} // dw
} // snd
} // nw
