﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/ctrl/sndctrl_SoundControllerImpl.h>

#ifdef NW_SND_CONFIG_ENABLE_DEV

#include <nw/ut/ut_Inlines.h>
#include <nw/snd/snd_SoundHandle.h>
#include <nw/snd/snd_SequenceSoundHandle.h>
#include <nw/snd/snd_SoundArchive.h>
#include <nw/snd/snd_SoundArchivePlayer.h>

namespace nw {
namespace snd {
namespace ctrl {
namespace internal {

namespace
{

bool
ReadGlobalVariable(
    SequenceSoundHandle* /*handle*/,
    snd::internal::fnd::BinS32 /*trackNo*/,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( var );

    short nWork = -1;
    bool  br    = SequenceSoundHandle::ReadGlobalVariable( varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteGlobalVariable(
    SequenceSoundHandle* /*handle*/,
    snd::internal::fnd::BinS32 /*trackNo*/,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32 var
)
{
    if( internal::SequenceVariableTraits::InvalidValue == var ) return true;
    return SequenceSoundHandle::WriteGlobalVariable( varNo, (short)var );
}

bool ReadLocalVariable(
    SequenceSoundHandle* handle,
    snd::internal::fnd::BinS32 /*trackNo*/,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadVariable( varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteLocalVariable(
    SequenceSoundHandle* handle,
    snd::internal::fnd::BinS32 /*trackNo*/,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32 var
)
{
    NW_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteVariable( varNo, (short)var );
}

bool ReadTrackVariable(
    SequenceSoundHandle* handle,
    snd::internal::fnd::BinS32 trackNo,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32* var
)
{
    NW_ASSERT_NOT_NULL( handle );
    NW_ASSERT_NOT_NULL( var );

    if( !handle->IsAttachedSound() ){
        *var = internal::SequenceVariableTraits::InvalidValue;
        return false;
    }

    short nWork = -1;
    bool  br    = handle->ReadTrackVariable( trackNo, varNo, &nWork );

    if( br ){
        *var = nWork;
    }

    return br;
}

bool WriteTrackVariable(
    SequenceSoundHandle* handle,
    snd::internal::fnd::BinS32 trackNo,
    snd::internal::fnd::BinS32 varNo,
    snd::internal::fnd::BinS32 var
)
{
    NW_ASSERT_NOT_NULL( handle );

    if( !handle->IsAttachedSound() ) return false;
    if( SequenceVariableTraits::InvalidValue == var ) return true;

    return handle->WriteTrackVariable( trackNo, varNo, (short)var );
}

}


//----------------------------------------------------------
SoundControllerImpl::SoundControllerImpl() :
m_SoundHandle(NULL),
m_SoundArchivePlayer(NULL),
m_SoundID(SoundArchive::INVALID_ID),
m_State(STATE_DISABLED),
m_PreplayAction(NULL),
m_UserParam(NULL)
{
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::Initialize(SoundHandle* soundHandle, SoundArchivePlayer* soundArchivePlayer)
{
    if(soundHandle == NULL ||
        soundArchivePlayer == NULL)
    {
        NW_FATAL_ERROR("invalid arguments.\n");
        return edit::Result(edit::SNDEDIT_RESULT_FAILED);
    }

    m_SoundHandle = soundHandle;
    m_SoundArchivePlayer = soundArchivePlayer;

#if defined(NW_PLATFORM_CTR)
    m_Lock.Initialize();
#endif

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundControllerImpl::Finalize()
{
#if defined(NW_PLATFORM_CTR)
    m_Lock.Finalize();
#endif

    m_SoundHandle = NULL;
    m_SoundArchivePlayer = NULL;
}

//----------------------------------------------------------
u32
SoundControllerImpl::GetSoundIndex() const
{
    // HACK : インデックスを抽出する正式な方法が見つかれば置き換える
    return nw::snd::internal::Util::GetItemIndex(m_SoundID);
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::SetSoundIndex(u32 index)
{
    if(!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    return SetSoundID(m_SoundArchivePlayer->GetSoundArchive().GetSoundIdFromIndex(index));
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::SetSoundID(SoundArchive::ItemId soundID)
{
    if (!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    m_SoundID = soundID;
    SetParameterDirty(true);

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
const char*
SoundControllerImpl::GetLabel() const
{
    if (!IsInitialized())
    {
        return "";
    }

    if(m_SoundID == nw::snd::SoundArchive::INVALID_ID)
    {
        return "";
    }

    const char* result = m_SoundArchivePlayer->GetSoundArchive().GetItemLabel(m_SoundID);
    return result == NULL ? "" : result;
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::SetLabel(const char* label)
{
    if(!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    return SetSoundID(m_SoundArchivePlayer->GetSoundArchive().GetItemId(label));
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::Play(const SoundStartable::StartInfo * pStartInfo)
{
    if(!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    Stop();

    if(m_PreplayAction != NULL)
    {
        m_PreplayAction(*this, m_SoundID, m_UserParam);
    }

    SoundStartable::StartResult result = m_SoundArchivePlayer->StartSound(m_SoundHandle, m_SoundID, pStartInfo);
    UpdateState();

    return result.IsSuccess() ?
        edit::Result(edit::SNDEDIT_RESULT_TRUE) : edit::Result(edit::SNDEDIT_RESULT_FAILED);
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::Stop()
{
    if(!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if(!m_SoundHandle->IsAttachedSound())
    {
        return edit::Result(edit::SNDEDIT_RESULT_FALSE);
    }

    m_SoundHandle->Stop(0);
    UpdateState();

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
edit::Result
SoundControllerImpl::Pause()
{
    if(!IsInitialized())
    {
        return edit::Result(edit::SNDEDIT_RESULT_NOT_INITIALIZED);
    }

    if(!m_SoundHandle->IsAttachedSound())
    {
        return edit::Result(edit::SNDEDIT_RESULT_FALSE);
    }

    m_SoundHandle->Pause(!m_SoundHandle->IsPause(), 0);
    UpdateState();

    return edit::Result(edit::SNDEDIT_RESULT_TRUE);
}

//----------------------------------------------------------
void
SoundControllerImpl::Update()
{
    if(!IsInitialized())
    {
        return;
    }

    UpdateState();
    UpdateParameters();
}

//----------------------------------------------------------
void
SoundControllerImpl::SetParameterDirty(bool isDirty)
{
    m_IsParameterDirty = m_IsParameterDirty || isDirty;
}

//----------------------------------------------------------
void
SoundControllerImpl::UpdateState()
{
    State newState = m_State;

    if( m_SoundHandle == NULL ||
        m_SoundID == SoundArchive::INVALID_ID)
    {
        newState = STATE_DISABLED;
    }
    else if( !m_SoundHandle->IsAttachedSound())
    {
        newState = STATE_STOPPED;
    }
    else if( m_SoundHandle->IsPause())
    {
        newState = STATE_PAUSED;
    }
    else
    {
        newState = STATE_PLAYING;
    }

    if(m_State == newState)
    {
        return;
    }

    m_State = newState;
}

//----------------------------------------------------------
bool
SoundControllerImpl::GetVariables(ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    switch( pVariables->header.variableType ){
    case SequenceVariableType_Global:
        return GetVariables( pVariables, ReadGlobalVariable );

    case SequenceVariableType_Local:
        return GetVariables( pVariables, ReadLocalVariable );

    case SequenceVariableType_Track:
        return GetVariables( pVariables, ReadTrackVariable );
    }

    InvalidateVariables( pVariables );
    return false;
}

//----------------------------------------------------------
bool
SoundControllerImpl::SetVariables(const ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    bool result = false;

    switch( pVariables->header.variableType ){
    case SequenceVariableType_Global:
        {
            SequenceVariableContainer globalVariables;
            globalVariables.Parse( pVariables );
            result = SetVariables( globalVariables, WriteGlobalVariable );
        }
        break;

    case SequenceVariableType_Local:
        {
            SequenceVariableContainer& localVariables = GetParameters().localVariables;
            localVariables.Parse( pVariables );
            result = SetVariables( localVariables, WriteLocalVariable );
        }
        break;

    case SequenceVariableType_Track:
        if( !SequenceVariableTraits::ValidateTrackNo( pVariables->header.trackNo ) ){
            NW_ASSERT( false );
            break;
        }
        {
            SequenceVariableContainer* trackVariables = GetParameters().trackVariables;
            trackVariables[ pVariables->header.trackNo ].Parse( pVariables );
            result = SetVariables( trackVariables[ pVariables->header.trackNo ], WriteTrackVariable );
        }
        break;
    }

    SetParameterDirty(result);

    return result;
}

//----------------------------------------------------------
void
SoundControllerImpl::InvalidateAllVariables()
{
    GetParameters().localVariables.InvalidateAll();

    for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
        GetParameters().trackVariables[ i ].InvalidateAll();
    }
}

//----------------------------------------------------------
void
SoundControllerImpl::ApplySequenceVariables()
{
    SetVariables( GetParameters().localVariables, WriteLocalVariable );

    for( s32 i = 0; i < SequenceVariableTraits::TrackCount; i++ )
    {
        SetVariables( GetParameters().trackVariables[ i ], WriteTrackVariable );
    }
}

//----------------------------------------------------------
bool
SoundControllerImpl::GetVariables(ToolSoundSequenceVariableContainer* pVariables, ReadVariableFunc pFunc)
{
    NW_ASSERT_NOT_NULL( pVariables );
    NW_ASSERT_NOT_NULL( pFunc );

    SequenceSoundHandle seqSoundHandle( m_SoundHandle );

    // シーケンス変数の値を取得する（割り込み禁止）
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_Lock);

        for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
            pFunc( &seqSoundHandle, (int)pVariables->header.trackNo, i, &pVariables->variables[ i ] );
        }
    }

    return true;
}

//----------------------------------------------------------
bool
SoundControllerImpl::SetVariables(SequenceVariableContainer& Variables, WriteVariableFunc pFunc)
{
    NW_ASSERT_NOT_NULL( pFunc );

    SequenceSoundHandle seqSoundHandle( m_SoundHandle );

    // シーケンス変数の値を更新する（割り込み禁止）
    {
        snd::internal::fnd::FndCriticalSectionScopedLock lock(m_Lock);

        for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
            pFunc( &seqSoundHandle, (int)Variables.GetTrackNo(), i, Variables[ i ].GetValue() );
        }
    }

    return true;
}

//----------------------------------------------------------
void
SoundControllerImpl::InvalidateVariables(ToolSoundSequenceVariableContainer* pVariables)
{
    NW_ASSERT_NOT_NULL( pVariables );

    // シーケンス変数の値を無効値に設定する
    for( int i=0; i<SequenceVariableTraits::VariableCount; i++ ){
        pVariables->variables[ i ] = SequenceVariableTraits::InvalidValue;
    }
}

} // namespace nw::snd::ctrl::internal
} // namespace nw::snd::ctrl
} // namespace nw::snd
} // namespace nw

#endif // NW_SND_CONFIG_ENABLE_DEV
