﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WaveSoundPlayer.h>
#include <nw/snd/snd_WaveFileReader.h>
#include <nw/snd/snd_DisposeCallbackManager.h>
#include <nw/snd/snd_SoundPlayer.h>
#include <nw/snd/snd_TaskManager.h>
#include <nw/snd/snd_SoundDataManager.h>

namespace nw {
namespace snd {
namespace internal {
namespace driver {

/* ========================================================================
        public function
   ======================================================================== */

/*---------------------------------------------------------------------------*
  Name:         WaveSoundPlayer

  Description:  コンストラクタ

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
WaveSoundPlayer::WaveSoundPlayer()
: m_IsInitialized(false),
  m_IsRegisterPlayerCallback(false),
  m_UseContextInfo(false),
  m_pLoaderManager(NULL),
  m_pLoader(NULL)
{
}

WaveSoundPlayer::~WaveSoundPlayer()
{
    Finalize();
}

void WaveSoundPlayer::Initialize()
{
    BasicSoundPlayer::Initialize();

    m_ReleasePriorityFixFlag = false;
    m_PanRange     = 1.0f;

    m_Priority     = DEFAULT_PRIORITY;

    m_pWsdFile = NULL;
    m_pWaveFile = NULL;
    m_WaveSoundIndex = -1;

    m_DelayCount = 0;

    m_WaveSoundInfo.pitch        = 1.0f;
    m_WaveSoundInfo.pan          = 64;
    m_WaveSoundInfo.surroundPan  = 0;
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        m_WaveSoundInfo.fxSend[ i ] = 0;
    }
    m_WaveSoundInfo.mainSend     = 127;

    m_LfoParam.Initialize();

    m_WavePlayFlag = false;
    m_pChannel = NULL;
    m_WaveType = WAVE_TYPE_INVALID;
    m_ResState = RES_STATE_INVALID;
    m_IsRegisterPlayerCallback = false;
    m_IsInitialized = true;
    m_UseContextInfo = false;
}

void WaveSoundPlayer::Finalize()
{
    FinishPlayer();

    if (m_ActiveFlag)
    {
        DisposeCallbackManager::GetInstance().UnregisterDisposeCallback( this );
        CloseChannel();
        m_ActiveFlag = false;
    }

    if (m_IsInitialized)
    {
        BasicSoundPlayer::Finalize();
        m_IsInitialized = false;
    }

    FreeLoader();
}

void WaveSoundPlayer::Prepare(const StartInfo& info, const PrepareArg& arg)
{
    if ( m_ActiveFlag )
    {
        FinishPlayer();
    }

    // StartInfo
    {
        m_WaveSoundIndex = info.index;
        m_StartOffsetType = info.startOffsetType;
        m_StartOffset = info.startOffset;
        m_DelayCount = ToDelayCount(info.delayTime);
    }

    // PrepareArg
    {
        m_pWsdFile = arg.wsdFile;
        m_pWaveFile = arg.waveFile;
        m_WaveType = arg.waveType;
        m_UseContextInfo = arg.useContextInfo;
        if (m_UseContextInfo)
        {
            m_ContextInfo = arg.contextInfo; // 構造体コピー
        }
    }

    m_ResState = RES_STATE_ASSIGNED;
    m_ActiveFlag = true;

    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
}

void WaveSoundPlayer::PrepareForPlayerHeap(const PrepareArg& arg)
{
    if ( m_ActiveFlag )
    {
        FinishPlayer();
    }

    // PrepareArg
    {
        m_pWsdFile = arg.wsdFile;
        m_pWaveFile = arg.waveFile;
        m_WaveType = arg.waveType;
    }

    m_ResState = RES_STATE_ASSIGNED;
    m_ActiveFlag = true;
    DisposeCallbackManager::GetInstance().RegisterDisposeCallback( this );
}

void WaveSoundPlayer::RequestLoad(const StartInfo& info, const WaveSoundLoader::Arg& arg)
{
    // StartInfo
    {
        m_WaveSoundIndex = info.index;
        m_StartOffsetType = info.startOffsetType;
        m_StartOffset = info.startOffset;
        m_DelayCount = ToDelayCount(info.delayTime);
    }

    // Loader::Arg
    m_LoaderArg = arg;  // 構造体コピー
    m_ResState = RES_STATE_RECV_LOAD_REQ;

    SoundThread::GetInstance().RegisterPlayerCallback( this );
    m_IsRegisterPlayerCallback = true;
}

void WaveSoundPlayer::Start()
{
    m_StartedFlag = true;
}

void WaveSoundPlayer::Stop()
{
    FinishPlayer();
}

void WaveSoundPlayer::Pause( bool flag )
{
    m_PauseFlag = static_cast<u8>( flag );

    // チャンネルを一時停止
    if ( IsChannelActive() )
    {
        if ( m_pChannel->IsPause() != flag )
        {
            m_pChannel->Pause( flag );
        }
    }
}

void WaveSoundPlayer::SetPanRange( f32 panRange )
{
    m_PanRange = panRange;
}

void WaveSoundPlayer::SetChannelPriority( int priority )
{
    NW_MINMAX_ASSERT( priority, 0, 127 );
    m_Priority = static_cast<u8>( priority );
}

void WaveSoundPlayer::SetReleasePriorityFix( bool fix )
{
    m_ReleasePriorityFixFlag = fix;
}

void WaveSoundPlayer::InvalidateData( const void* start, const void* end )
{
    if ( m_ActiveFlag )
    {
        const void* current = m_pWsdFile;
        if ( start <= current && current <= end )
        {
            FinishPlayer();
        }
    }
}

void WaveSoundPlayer::FinishPlayer()
{
    m_FinishFlag = true;
    if (m_IsRegisterPlayerCallback)
    {
        SoundThread::GetInstance().UnregisterPlayerCallback( this );
        m_IsRegisterPlayerCallback = false;
    }
    if (m_StartedFlag)
    {
        m_StartedFlag = false;
    }
}

s32 WaveSoundPlayer::GetPlaySamplePosition(bool isOriginalSamplePosition) const
{
    if ( m_pChannel == NULL ) return 0;

    return static_cast<s32>( m_pChannel->GetCurrentPlayingSample(isOriginalSamplePosition) );
}

void WaveSoundPlayer::Update()
{
    // データロード状態の管理
    {
        switch (m_ResState)
        {
        case RES_STATE_ASSIGNED:
            break;
        case RES_STATE_RECV_LOAD_REQ:
            if (TryAllocLoader())
            {
                m_pLoader->Initialize(m_LoaderArg);
            }
            else
            {
                return; // リトライする
            }
            // /* FALLTHROUGH */
        case RES_STATE_APPEND_LOAD_TASK:
            NW_ASSERT_NOT_NULL(m_pLoader);
            if (m_pLoader->TryWait())
            {
                if (m_pLoader->IsLoadSuccess())
                {
                    // タスク完了したので、プリペア
                    PrepareArg arg;
                    arg.wsdFile = m_pLoader->GetWsdFile();
                    arg.waveFile = m_pLoader->GetWaveFile();
                    // arg.waveType はデフォルト (0 = WAVE_TYPE_NWWAV) のまま
                    PrepareForPlayerHeap(arg);
                }
                else
                {
                    // データロードに失敗したので終了
                    FinishPlayer();
                    return;
                }
            }
            else
            {
                return; // リトライする
            }
            break;
        default:
            NW_ASSERT(false);
            break;
        }
    }

    if (m_DelayCount > 0)
    {
        --m_DelayCount;
        return;
    }

    // フラグチェック
    {
        if ( ! m_ActiveFlag )
        {
            return;
        }
        if ( ! m_StartedFlag )
        {
            return;
        }
    }

    // 波形再生
    if ( ! m_PauseFlag )
    {
        // 波形終端終了チェック
        if ( ( m_WavePlayFlag ) && ( m_pChannel == NULL ) )
        {
            FinishPlayer();
            return;
        }

        // 波形再生
        if ( ! m_WavePlayFlag )
        {
            if ( ! StartChannel() )
            {
                FinishPlayer();
                return;
            }
        }
    }

    UpdateChannel();
}

bool WaveSoundPlayer::TryAllocLoader()
{
    if (m_pLoaderManager == NULL)
    {
        return false;
    }

    WaveSoundLoader* loader = m_pLoaderManager->Alloc();
    if (loader == NULL)
    {
        return false;
    }

    m_pLoader = loader;
    m_ResState = RES_STATE_APPEND_LOAD_TASK;
    return true;
}

void WaveSoundPlayer::FreeLoader()
{
    if (m_pLoader)
    {
        m_pLoaderManager->Free(m_pLoader);
        m_pLoader = NULL;
    }
}

bool WaveSoundPlayer::StartChannel()
{
    const int priority = GetChannelPriority() + DEFAULT_PRIORITY;

    WaveInfo waveInfo;
    {
        // NW_LOG("--%20s-- ", GetThName());NW_LOG("WaveSoundPlayer(%p)::%s %c%c%c%c\n", this, __FUNCTION__,
        //         ((char*)(m_pWaveFile))[0],
        //         ((char*)(m_pWaveFile))[1],
        //         ((char*)(m_pWaveFile))[2],
        //         ((char*)(m_pWaveFile))[3]);
        internal::WaveFileReader reader( m_pWaveFile, m_WaveType );
        if ( ! reader.ReadWaveInfo( &waveInfo ) )
        {
            return false;
        }
    }

    // 開始オフセット
    u32 startOffsetSamples = 0;
    if (m_UseContextInfo)
    {
        startOffsetSamples = m_ContextInfo.offsetSample;
    }
    else
    {
        if ( m_StartOffsetType == START_OFFSET_TYPE_SAMPLE )
        {
            startOffsetSamples = static_cast<u32>( m_StartOffset );
        }
        else if ( m_StartOffsetType == START_OFFSET_TYPE_MILLISEC )
        {
            startOffsetSamples = static_cast<u32>(
                static_cast<s64>( m_StartOffset ) * waveInfo.sampleRate / 1000
            );
        }
    }

    // 開始オフセットが範囲外なので再生せずに終了
    if ( startOffsetSamples > waveInfo.loopEndFrame ) return false;

    Channel* channel;
    channel = Channel::AllocChannel(
        ut::Min( static_cast<int>( waveInfo.channelCount ), 2 ),
        priority,
        ChannelCallbackFunc,
        this,
        GetVoiceRendererType()
    );

    // 鳴らせなかったらエラーでサウンド終了
    if ( channel == NULL )
    {
        return false;
    }

    // ウェーブサウンド情報取得
    {
        WaveSoundFileReader reader( m_pWsdFile );
        if ( ! reader.ReadWaveSoundInfo( &m_WaveSoundInfo, m_WaveSoundIndex ) )
        {
            return false;
        }
    }

    // エンベロープ設定
    channel->SetAttack( m_WaveSoundInfo.adshr.attack );
    channel->SetHold( m_WaveSoundInfo.adshr.hold );
    channel->SetDecay( m_WaveSoundInfo.adshr.decay );
    channel->SetSustain( m_WaveSoundInfo.adshr.sustain );
    channel->SetRelease( m_WaveSoundInfo.adshr.release );

    channel->SetReleasePriorityFix( m_ReleasePriorityFixFlag );
    channel->SetFrontBypass( IsFrontBypass() );

    if (m_UseContextInfo)
    {
        channel->SetOffsetContext(m_ContextInfo);
    }
    channel->Start( waveInfo, -1, startOffsetSamples );

    // チャンネルリストへの結合
    m_pChannel = channel;

    m_WavePlayFlag = true;

    return true;
}

/*---------------------------------------------------------------------------*
  Name:         CloseChannel

  Description:  チャンネルを閉じます

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSoundPlayer::CloseChannel()
{
    // Release
    if ( IsChannelActive() )
    {
        UpdateChannel();
        m_pChannel->Release();
    }

    // Free
    if ( m_pChannel != NULL )
    {
        Channel::DetachChannel( m_pChannel );
        m_pChannel = NULL;
    }
}

/*---------------------------------------------------------------------------*
  Name:         UpdateChannel

  Description:  トラックが保持しているチャンネルのパラメータを更新します

  Arguments:    None.

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSoundPlayer::UpdateChannel()
{
    if ( m_pChannel == NULL ) return;

    // volume
    f32 volume = 1.0f;
    volume *= GetVolume();

    f32 pitchRatio = 1.0f;
    pitchRatio *= GetPitch();
    pitchRatio *= m_WaveSoundInfo.pitch;

    // lpf freq
    f32 lpfFreq = static_cast<f32>( m_WaveSoundInfo.lpfFreq - 64 ) / 64.0f;
    lpfFreq += GetLpfFreq();

    // biquad filter
    int biquadType = static_cast<int>( m_WaveSoundInfo.biquadType );
    f32 biquadValue = static_cast<f32>( m_WaveSoundInfo.biquadValue ) / 127.0f;
    int handleBiquadType = GetBiquadFilterType();
    if ( handleBiquadType != static_cast<int>(BIQUAD_FILTER_TYPE_INHERIT) )
    {
        biquadType  = handleBiquadType;
        biquadValue = GetBiquadFilterValue();
    }

    // データの値を算出する (TODO:キャッシュできそうなので、キャッシュする)
    f32 panBase = 0.0f;
    if ( m_WaveSoundInfo.pan <= 1 ) // pan の 1 と 2 は同じ値になる
    {
        panBase = static_cast<f32>( static_cast<int>( m_WaveSoundInfo.pan ) - 63 ) / 63.0f;
    }
    else
    {
        panBase = static_cast<f32>( static_cast<int>( m_WaveSoundInfo.pan ) - 64 ) / 63.0f;
    }
    panBase *= m_PanRange;

    f32 spanBase = 0.0f;
    if ( m_WaveSoundInfo.surroundPan <= 63 )
    {
        spanBase = static_cast<f32>( m_WaveSoundInfo.surroundPan ) / 63.0f;
    }
    else
    {
        spanBase = static_cast<f32>( m_WaveSoundInfo.surroundPan + 1 ) / 64.0f;
            // NOTE: y = (1/64) * x + 1/64 = (x/64) + (1/64) = (x+1)/64
    }

    f32 mainSendBase = static_cast<f32>( m_WaveSoundInfo.mainSend ) / 127.0f - 1.0f;
    f32 fxSendBase[AUX_BUS_NUM];
    for ( int i = 0; i < AUX_BUS_NUM; i++ )
    {
        fxSendBase[i] = static_cast<f32>( m_WaveSoundInfo.fxSend[i] ) / 127.0f;
    }

    // TV 出力向けパラメータ (データの値と重ね合わせる)
    OutputParam tvParam = GetTvParam();
    {
        tvParam.pan += panBase;
        tvParam.span += spanBase;
        tvParam.mainSend += mainSendBase;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            tvParam.fxSend[i] += fxSendBase[i];
        }
    }

    // DRC 出力向けパラメータ (データの値と重ね合わせる)
    OutputParam drcParam[DRC_OUT_COUNT];
    for ( u32 drc = 0; drc < DRC_OUT_COUNT; drc++ )
    {
        OutputParam& param = drcParam[drc];
        param = GetDrcParam( drc );

        param.pan += panBase;
        param.span += spanBase;
        param.mainSend += mainSendBase;
        for ( int i = 0; i < AUX_BUS_NUM; i++ )
        {
            param.fxSend[i] += fxSendBase[i];
        }
    }

    // Channel インスタンスへの反映
    m_pChannel->SetPanMode( GetPanMode() );
    m_pChannel->SetPanCurve( GetPanCurve() );
    m_pChannel->SetRemoteFilter( GetRemoteFilter() );
    m_pChannel->SetUserVolume( volume );
    m_pChannel->SetUserPitchRatio( pitchRatio );
    m_pChannel->SetLfoParam( m_LfoParam );
    m_pChannel->SetUserLpfFreq( lpfFreq );
    m_pChannel->SetBiquadFilter( biquadType, biquadValue );
    m_pChannel->SetOutputLine( GetOutputLine() );

    m_pChannel->SetTvParam( tvParam );
    for ( u32 i = 0; i < DRC_OUT_COUNT; i++ )
    {
        m_pChannel->SetDrcParam( i, drcParam[i] );
    }
    for ( u32 i = 0; i < REMOTE_OUT_COUNT; i++ )
    {
        m_pChannel->SetRemoteParam( i, GetRemoteParam(i) );
    }
}

/*---------------------------------------------------------------------------*
  Name:         ChannelCallbackFunc

  Description:  チャンネルから呼びだされるコールバック関数

  Arguments:    dropChannel - チャンネルポインタ
                status - チャンネルコールバックステータス
                userData - トラックポインタを格納したコールバックユーザーデータ

  Returns:      None.
 *---------------------------------------------------------------------------*/
void WaveSoundPlayer::ChannelCallbackFunc(
    Channel* dropChannel,
    Channel::ChannelCallbackStatus /* status */,
    void* userData
)
{
    WaveSoundPlayer* player = reinterpret_cast<WaveSoundPlayer*>( userData );

    NW_NULL_ASSERT( dropChannel );
    NW_NULL_ASSERT( player );
    NW_ASSERT( dropChannel == player->m_pChannel );

    // チャンネル参照の切断
    player->m_pChannel = NULL;
}

void WaveSoundLoader::Initialize(const Arg& arg)
{
    m_Task.Initialize();
    m_Task.m_Arg = arg;
    m_Task.m_pPlayerHeap = NULL;
}

void WaveSoundLoader::Finalize()
{
    m_Task.FreePlayerHeap();
}

bool WaveSoundLoader::TryWait()
{
    // プレイヤーヒープの確保
    if (m_Task.TryAllocPlayerHeap() == false)
    {
        return false;
    }

    // データのアペンド
    Task::Status status = m_Task.GetStatus();
    switch (status)
    {
    case Task::STATUS_FREE:
        TaskManager::GetInstance().AppendTask(&m_Task, TaskManager::PRIORITY_MIDDLE);
        return false;
    case Task::STATUS_DONE:
    case Task::STATUS_CANCEL:
        return true;
    default:
        return false;
    }
}

bool WaveSoundLoader::IsInUse()
{
    if (m_Task.TryWait())
    {
        return false;
    }
    return true;
}





void WaveSoundLoader::DataLoadTask::Initialize()
{
    Task::InitializeStatus();
    m_Data.Initialize();
    m_IsLoadSuccess = false;
}

bool WaveSoundLoader::DataLoadTask::TryAllocPlayerHeap()
{
    if (m_pPlayerHeap == NULL)
    {
        NW_ASSERT_NOT_NULL(m_Arg.soundPlayer);
        m_pPlayerHeap = m_Arg.soundPlayer->detail_AllocPlayerHeap();
        if (m_pPlayerHeap == NULL)
        {
            return false;
        }
        m_pPlayerHeap->Clear();
    }
    return true;
}

void WaveSoundLoader::DataLoadTask::FreePlayerHeap()
{
    if (m_pPlayerHeap)
    {
        m_Arg.soundPlayer->detail_FreePlayerHeap(m_pPlayerHeap);
    }
}

void WaveSoundLoader::DataLoadTask::Execute()
{
    NW_ASSERT_NOT_NULL(m_pPlayerHeap);

    PlayerHeapDataManager playerHeapDataManager;
    playerHeapDataManager.Initialize(m_Arg.soundArchive);

    if ( ( m_Arg.loadInfoWsd.address == NULL ) &&
         ( m_Arg.loadInfoWsd.itemId != SoundArchive::INVALID_ID ) )
    {
        SoundArchive::ItemId soundId = m_Arg.loadInfoWsd.itemId;

        if ( ! playerHeapDataManager.LoadData(
                    soundId, m_pPlayerHeap, SoundArchiveLoader::LOAD_WSD) )
        {
            NW_WARNING( false, "failed to load WSD(%08x) to PlayerHeap", soundId );
            m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
            m_IsLoadSuccess = false;
            return;
        }
        m_Arg.loadInfoWsd.address =
            playerHeapDataManager.detail_GetFileAddressByItemId( soundId );
    }

    NW_NULL_ASSERT(m_Arg.loadInfoWsd.address);

    // SoundDataManager 側で波形アーカイブがロード済みか？
    const void* waveFile = Util::GetWaveFileOfWaveSound(
            m_Arg.loadInfoWsd.address, m_Arg.index,
            *m_Arg.soundArchive, *m_Arg.soundDataManager );
    // 本来であれば WaveSound は SoundArchive や SoundDataManager に依存したくないが、
    // プレイヤーヒープ利用時のみ、我慢する。
    if ( waveFile == NULL )
    {
        // 未ロードなら、ここでロードする
        if ( ! playerHeapDataManager.detail_LoadWaveArchiveByWaveSoundFile(
                    m_Arg.loadInfoWsd.address, m_Arg.index, m_pPlayerHeap ) )
        {
            NW_WARNING( false,
                    "failed to load WSD(%08x)'s WARC to PlayerHeap", m_Arg.loadInfoWsd.itemId );
            m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
            m_IsLoadSuccess = false;
            return;
        }
        waveFile = Util::GetWaveFileOfWaveSound(
            m_Arg.loadInfoWsd.address,
            m_Arg.index,
            *m_Arg.soundArchive,
            playerHeapDataManager );
    }

    m_Data.wsdFile = m_Arg.loadInfoWsd.address;
    m_Data.waveFile = waveFile;

    m_pPlayerHeap->SetState(PlayerHeap::STATE_TASK_FINISHED);
    playerHeapDataManager.Finalize();
    m_IsLoadSuccess = true;
}


} // namespace nw::snd::internal::driver
} // namespace nw::snd::internal
} // namespace nw::snd
} // namespace nw

