﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nw/snd/snd_WavePlayer.h>

#include <nw/snd/snd_MultiVoiceManager.h>
#include <nw/snd/snd_DriverCommand.h>
#include <nw/snd/snd_SoundThread.h>

namespace nw {
namespace snd {

bool WavePlayer::Initialize(
    int channelCount,
    /* int voiceOutCount, */
    int priority,
    VoiceCallback callback,
    void* callbackArg )
{
    NW_ASSERT_MINMAXLT( channelCount, 0, WAVE_CHANNEL_MAX );
    NW_ASSERT_MINMAXLT( priority, 0, 255 );

    internal::driver::MultiVoice* voice = NULL;
    {
        internal::driver::SoundThreadLock lock;

        voice = internal::driver::MultiVoiceManager::GetInstance().AllocVoice(
            channelCount,
            priority,
            MultiVoiceCallbackFunc,
            this );
    }

    if ( voice == NULL )
    {
        return false;
    }
    m_pMultiVoice = voice;
    m_Callback = callback;
    m_pCallbackArg = callbackArg;

    return true;
}

void WavePlayer::Finalize()
{
#if 0
    internal::driver::SoundThreadLock lock;

    if ( m_pMultiVoice )
    {
        m_pMultiVoice->Free();
    }
#else
    Stop();
#endif
}


void WavePlayer::Start()
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePlay>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PLAY;
    cmd->voice = m_pMultiVoice;
    cmd->state = internal::DriverCommandVoicePlay::STATE_START;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::Stop()
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePlay>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PLAY;
    cmd->voice = m_pMultiVoice;
    cmd->state = internal::DriverCommandVoicePlay::STATE_STOP;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::Pause( bool isPauseOn )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePlay* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePlay>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PLAY;
    cmd->voice = m_pMultiVoice;
    cmd->state = isPauseOn ? internal::DriverCommandVoicePlay::STATE_PAUSE_ON :
                             internal::DriverCommandVoicePlay::STATE_PAUSE_OFF;
    cmdmgr.PushCommand( cmd );
}

bool WavePlayer::IsActive() const
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsActive();
    }
    return false;
}

bool WavePlayer::IsRun() const
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsRun();
    }
    return false;
}

bool WavePlayer::IsPause() const
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->IsPause();
    }
    return false;
}

u32 WavePlayer::GetCurrentPlayingSample() const
{
    if ( m_pMultiVoice )
    {
        return m_pMultiVoice->GetCurrentPlayingSample();
    }
    return 0;
}

void WavePlayer::SetWaveInfo( SampleFormat format, int sampleRate, int interpolationType )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceWaveInfo* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceWaveInfo>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_WAVEINFO;
    cmd->voice = m_pMultiVoice;
    cmd->format = format;
    cmd->sampleRate = sampleRate;
    cmd->interpolationType = interpolationType;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetAdpcmParam( int channel, /* int voiceOut, */const AdpcmParam& param )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceAdpcmParam* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceAdpcmParam>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_ADPCM_PARAM;
    cmd->voice = m_pMultiVoice;
    cmd->channel = channel;
    // cmd->voiceOut = voiceOut;
    cmd->param = &param;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::AppendWaveBuffer(
    int channel,
    /* int voiceOut, */
    WaveBuffer* pBuffer,
    bool lastFlag )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

#if 0
    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceAppendWaveBuffer* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceAppendWaveBuffer>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_APPEND_WAVEBUFFER;
    cmd->voice = m_pMultiVoice;
    cmd->channel = channel;
    // cmd->voiceOut = voiceOut;
    cmd->buffer = pBuffer;
    cmd->lastFlag = lastFlag;
    cmdmgr.PushCommand( cmd );
#else
    internal::driver::SoundThreadLock lock;
    m_pMultiVoice->AppendWaveBuffer( channel, pBuffer, lastFlag );
#endif
}

void WavePlayer::SetPriority( int priority )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePriority* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePriority>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PRIORITY;
    cmd->voice = m_pMultiVoice;
    cmd->priority = priority;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetVolume( f32 volume )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceVolume* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceVolume>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_VOLUME;
    cmd->voice = m_pMultiVoice;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPitch( f32 pitch )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePitch* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePitch>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PITCH;
    cmd->voice = m_pMultiVoice;
    cmd->pitch = pitch;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPanMode( PanMode mode )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePanMode* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePanMode>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PAN_MODE;
    cmd->voice = m_pMultiVoice;
    cmd->mode = mode;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPanCurve( PanCurve curve )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePanCurve* cmd = cmdmgr.AllocCommand<internal::DriverCommandVoicePanCurve>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PAN_CURVE;
    cmd->voice = m_pMultiVoice;
    cmd->curve = curve;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetPan( f32 pan )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoicePan* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoicePan>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_PAN;
    cmd->voice = m_pMultiVoice;
    cmd->pan = pan;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetSurroundPan( f32 span )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceSurroundPan* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceSurroundPan>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_SPAN;
    cmd->voice = m_pMultiVoice;
    cmd->span = span;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetLpfFreq( f32 lpfFreq )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceLpfFreq* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceLpfFreq>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_LPF;
    cmd->voice = m_pMultiVoice;
    cmd->lpfFreq = lpfFreq;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetBiquadFilter( int type, f32 value )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceBiquadFilter* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceBiquadFilter>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_BIQUAD;
    cmd->voice = m_pMultiVoice;
    cmd->type = type;
    cmd->value = value;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetOutputLine( u32 lineFlag )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceOutputLine* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceOutputLine>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_OUTPUT_LINE;
    cmd->voice = m_pMultiVoice;
    cmd->lineFlag = lineFlag;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetMainOutVolume( f32 volume )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceMainOutVolume* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceMainOutVolume>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_MAIN_OUT_VOLUME;
    cmd->voice = m_pMultiVoice;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetRemoteOutVolume( int remoteIndex, f32 volume )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceRemoteOutVolume* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceRemoteOutVolume>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_REMOTE_OUT_VOLUME;
    cmd->voice = m_pMultiVoice;
    cmd->remoteIndex = remoteIndex;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::detail_SetDrcOutVolume( int drcIndex, f32 volume )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceDrcOutVolume* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceDrcOutVolume>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_DRC_OUT_VOLUME;
    cmd->voice = m_pMultiVoice;
    cmd->drcIndex = drcIndex;
    cmd->volume = volume;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetMainSend( f32 send )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceMainSend* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceMainSend>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_MAIN_SEND;
    cmd->voice = m_pMultiVoice;
    cmd->send = send;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::SetFxSend( AuxBus bus, f32 send )
{
    NW_ASSERT_NOT_NULL( m_pMultiVoice );

    internal::DriverCommand& cmdmgr = internal::DriverCommand::GetInstance();
    internal::DriverCommandVoiceFxSend* cmd =
        cmdmgr.AllocCommand<internal::DriverCommandVoiceFxSend>();
    cmd->id = internal::DRIVER_COMMAND_VOICE_FX_SEND;
    cmd->voice = m_pMultiVoice;
    cmd->bus = bus;
    cmd->send = send;
    cmdmgr.PushCommand( cmd );
}

void WavePlayer::MultiVoiceCallbackFunc(
    internal::driver::MultiVoice* driverVoice,
    internal::driver::MultiVoice::VoiceCallbackStatus status,
    void* arg )
{
    NW_ASSERT_NOT_NULL( arg );

    switch ( status )
    {
        case internal::driver::MultiVoice::CALLBACK_STATUS_FINISH_WAVE:
        case internal::driver::MultiVoice::CALLBACK_STATUS_CANCEL:
            driverVoice->Free();    // WaveBuffer との関連付けを切る
            break;
        default:
            break;
    }

    WavePlayer* wavePlayer = static_cast<WavePlayer*>(arg);
    if ( wavePlayer->m_Callback )
    {
        wavePlayer->m_Callback( wavePlayer->m_pCallbackArg );
    }

    wavePlayer->m_pMultiVoice = NULL;
    wavePlayer->m_Callback = NULL;
    wavePlayer->m_pCallbackArg = NULL;
}


} // namespace nw::snd
} // namespace nw
